/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.inject.api;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;

/**
 * Aggregates the set of {@link InjectionPointInfo}'s that are dependent upon a specific and common
 * {@link ServiceInfo} definition.
 *
 * @see #builder()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.inject.api.DependencyInfoBlueprint")
public interface DependencyInfo extends DependencyInfoBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static DependencyInfo.Builder builder() {
        return new DependencyInfo.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static DependencyInfo.Builder builder(DependencyInfo instance) {
        return DependencyInfo.builder().from(instance);
    }

    /**
     * Fluent API builder base for {@link DependencyInfo}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends DependencyInfo.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends DependencyInfo> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private final Set<InjectionPointInfo> injectionPointDependencies = new LinkedHashSet<>();
        private ServiceInfoCriteria dependencyTo;
        private ServiceProvider<?> resolvedTo;
        private String elementName;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(DependencyInfo prototype) {
            elementName(prototype.elementName());
            dependencyTo(prototype.dependencyTo());
            addInjectionPointDependencies(prototype.injectionPointDependencies());
            resolvedTo(prototype.resolvedTo());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(DependencyInfo.BuilderBase<?, ?> builder) {
            builder.elementName().ifPresent(this::elementName);
            builder.dependencyTo().ifPresent(this::dependencyTo);
            addInjectionPointDependencies(builder.injectionPointDependencies());
            builder.resolvedTo().ifPresent(this::resolvedTo);
            return self();
        }

        /**
         * Name of the dependency location, such as a field name, or argument name.
         *
         * @param elementName name of the element of this dependency
         * @return updated builder instance
         * @see #elementName()
         */
        public BUILDER elementName(String elementName) {
            Objects.requireNonNull(elementName);
            this.elementName = elementName;
            return self();
        }

        /**
         * The service info describing what the injection point dependencies are dependent upon.
         *
         * @param dependencyTo the service info dependency
         * @return updated builder instance
         * @see #dependencyTo()
         */
        public BUILDER dependencyTo(ServiceInfoCriteria dependencyTo) {
            Objects.requireNonNull(dependencyTo);
            this.dependencyTo = dependencyTo;
            return self();
        }

        /**
         * The service info describing what the injection point dependencies are dependent upon.
         *
         * @param consumer consumer of builder for
         *                 the service info dependency
         * @return updated builder instance
         * @see #dependencyTo()
         */
        public BUILDER dependencyTo(Consumer<ServiceInfoCriteria.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = ServiceInfoCriteria.builder();
            consumer.accept(builder);
            this.dependencyTo(builder.build());
            return self();
        }

        /**
         * The service info describing what the injection point dependencies are dependent upon.
         *
         * @param supplier supplier of
         *                 the service info dependency
         * @return updated builder instance
         * @see #dependencyTo()
         */
        public BUILDER dependencyTo(Supplier<? extends ServiceInfoCriteria> supplier) {
            Objects.requireNonNull(supplier);
            this.dependencyTo(supplier.get());
            return self();
        }

        /**
         * The set of injection points that depends upon {@link #dependencyTo()}.
         *
         * @param injectionPointDependencies the set of dependencies
         * @return updated builder instance
         * @see #injectionPointDependencies()
         */
        public BUILDER injectionPointDependencies(Set<? extends InjectionPointInfo> injectionPointDependencies) {
            Objects.requireNonNull(injectionPointDependencies);
            this.injectionPointDependencies.clear();
            this.injectionPointDependencies.addAll(injectionPointDependencies);
            return self();
        }

        /**
         * The set of injection points that depends upon {@link #dependencyTo()}.
         *
         * @param injectionPointDependencies the set of dependencies
         * @return updated builder instance
         * @see #injectionPointDependencies()
         */
        public BUILDER addInjectionPointDependencies(Set<? extends InjectionPointInfo> injectionPointDependencies) {
            Objects.requireNonNull(injectionPointDependencies);
            this.injectionPointDependencies.addAll(injectionPointDependencies);
            return self();
        }

        /**
         * The set of injection points that depends upon {@link #dependencyTo()}.
         *
         * @param injectionPointDependency the set of dependencies
         * @return updated builder instance
         * @see #injectionPointDependencies()
         */
        public BUILDER addInjectionPointDependency(InjectionPointInfo injectionPointDependency) {
            Objects.requireNonNull(injectionPointDependency);
            this.injectionPointDependencies.add(injectionPointDependency);
            return self();
        }

        /**
         * The set of injection points that depends upon {@link #dependencyTo()}.
         *
         * @param consumer the set of dependencies
         * @return updated builder instance
         * @see #injectionPointDependencies()
         */
        public BUILDER addInjectionPointDependency(Consumer<InjectionPointInfo.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = InjectionPointInfo.builder();
            consumer.accept(builder);
            this.injectionPointDependencies.add(builder.build());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #resolvedTo()
         */
        public BUILDER clearResolvedTo() {
            this.resolvedTo = null;
            return self();
        }

        /**
         * The {@link ServiceProvider} that this dependency is optional resolved and bound to. All dependencies
         * from {@link #injectionPointDependencies()} will be bound to this resolution.
         *
         * @param resolvedTo the optional resolved and bounded service provider
         * @return updated builder instance
         * @see #resolvedTo()
         */
        public BUILDER resolvedTo(ServiceProvider<?> resolvedTo) {
            Objects.requireNonNull(resolvedTo);
            this.resolvedTo = resolvedTo;
            return self();
        }

        /**
         * Name of the dependency location, such as a field name, or argument name.
         *
         * @return the element name
         */
        public Optional<String> elementName() {
            return Optional.ofNullable(elementName);
        }

        /**
         * The service info describing what the injection point dependencies are dependent upon.
         *
         * @return the dependency to
         */
        public Optional<ServiceInfoCriteria> dependencyTo() {
            return Optional.ofNullable(dependencyTo);
        }

        /**
         * The set of injection points that depends upon {@link #dependencyTo()}.
         *
         * @return the injection point dependencies
         */
        public Set<InjectionPointInfo> injectionPointDependencies() {
            return injectionPointDependencies;
        }

        /**
         * The {@link ServiceProvider} that this dependency is optional resolved and bound to. All dependencies
         * from {@link #injectionPointDependencies()} will be bound to this resolution.
         *
         * @return the resolved to
         */
        public Optional<ServiceProvider<?>> resolvedTo() {
            return Optional.ofNullable(resolvedTo);
        }

        @Override
        public String toString() {
            return "DependencyInfoBuilder{"
                    + "elementName=" + elementName + ","
                    + "dependencyTo=" + dependencyTo + ","
                    + "injectionPointDependencies=" + injectionPointDependencies + ","
                    + "resolvedTo=" + resolvedTo
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (elementName == null) {
                collector.fatal(getClass(), "Property \"elementName\" must not be null, but not set");
            }
            if (dependencyTo == null) {
                collector.fatal(getClass(), "Property \"dependency-to\" is required, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * The {@link ServiceProvider} that this dependency is optional resolved and bound to. All dependencies
         * from {@link #injectionPointDependencies()} will be bound to this resolution.
         *
         * @param resolvedTo the optional resolved and bounded service provider
         * @return updated builder instance
         * @see #resolvedTo()
         */
        BUILDER resolvedTo(Optional<ServiceProvider<?>> resolvedTo) {
            Objects.requireNonNull(resolvedTo);
            this.resolvedTo = resolvedTo.map(io.helidon.inject.api.ServiceProvider.class::cast).orElse(this.resolvedTo);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class DependencyInfoImpl implements DependencyInfo {

            private final Optional<ServiceProvider<?>> resolvedTo;
            private final ServiceInfoCriteria dependencyTo;
            private final Set<InjectionPointInfo> injectionPointDependencies;
            private final String elementName;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected DependencyInfoImpl(DependencyInfo.BuilderBase<?, ?> builder) {
                this.elementName = builder.elementName().get();
                this.dependencyTo = builder.dependencyTo().get();
                this.injectionPointDependencies = Collections.unmodifiableSet(new LinkedHashSet<>(builder.injectionPointDependencies()));
                this.resolvedTo = builder.resolvedTo();
            }

            @Override
            public String elementName() {
                return elementName;
            }

            @Override
            public ServiceInfoCriteria dependencyTo() {
                return dependencyTo;
            }

            @Override
            public Set<InjectionPointInfo> injectionPointDependencies() {
                return injectionPointDependencies;
            }

            @Override
            public Optional<ServiceProvider<?>> resolvedTo() {
                return resolvedTo;
            }

            @Override
            public String toString() {
                return "DependencyInfo{"
                        + "elementName=" + elementName + ","
                        + "dependencyTo=" + dependencyTo + ","
                        + "injectionPointDependencies=" + injectionPointDependencies + ","
                        + "resolvedTo=" + resolvedTo
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof DependencyInfo other)) {
                    return false;
                }
                return Objects.equals(elementName, other.elementName())
                        && Objects.equals(dependencyTo, other.dependencyTo())
                        && Objects.equals(injectionPointDependencies, other.injectionPointDependencies())
                        && Objects.equals(resolvedTo, other.resolvedTo());
            }

            @Override
            public int hashCode() {
                return Objects.hash(elementName, dependencyTo, injectionPointDependencies, resolvedTo);
            }

        }

    }

    /**
     * Fluent API builder for {@link DependencyInfo}.
     */
    class Builder extends DependencyInfo.BuilderBase<DependencyInfo.Builder, DependencyInfo> implements io.helidon.common.Builder<DependencyInfo.Builder, DependencyInfo> {

        private Builder() {
        }

        @Override
        public DependencyInfo buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new DependencyInfoImpl(this);
        }

        @Override
        public DependencyInfo build() {
            return buildPrototype();
        }

    }

}
