/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.inject.api;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;

/**
 * Provides optional, contextual tunings to the {@link Injector}.
 *
 * @see Injector
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.inject.api.InjectorOptionsBlueprint")
public interface InjectorOptions extends InjectorOptionsBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static InjectorOptions.Builder builder() {
        return new InjectorOptions.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static InjectorOptions.Builder builder(InjectorOptions instance) {
        return InjectorOptions.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static InjectorOptions create() {
        return InjectorOptions.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link InjectorOptions}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends InjectorOptions.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends InjectorOptions> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private ActivationRequest activationRequest;
        private Injector.Strategy strategy = Injector.Strategy.ANY;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(InjectorOptions prototype) {
            strategy(prototype.strategy());
            activationRequest(prototype.activationRequest());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(InjectorOptions.BuilderBase<?, ?> builder) {
            strategy(builder.strategy());
            builder.activationRequest().ifPresent(this::activationRequest);
            return self();
        }

        /**
         * The strategy the injector should apply. The default is {@link Injector.Strategy#ANY}.
         *
         * @param strategy the injector strategy to use
         * @return updated builder instance
         * @see #strategy()
         */
        public BUILDER strategy(Injector.Strategy strategy) {
            Objects.requireNonNull(strategy);
            this.strategy = strategy;
            return self();
        }

        /**
         * Optionally, customized activator options to use for the {@link Activator}.
         *
         * @param activationRequest activator options, or leave blank to use defaults
         * @return updated builder instance
         * @see #activationRequest()
         */
        public BUILDER activationRequest(ActivationRequest activationRequest) {
            Objects.requireNonNull(activationRequest);
            this.activationRequest = activationRequest;
            return self();
        }

        /**
         * Optionally, customized activator options to use for the {@link Activator}.
         *
         * @param consumer consumer of builder for
         *                 activator options, or leave blank to use defaults
         * @return updated builder instance
         * @see #activationRequest()
         */
        public BUILDER activationRequest(Consumer<ActivationRequest.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = ActivationRequest.builder();
            consumer.accept(builder);
            this.activationRequest(builder.build());
            return self();
        }

        /**
         * Optionally, customized activator options to use for the {@link Activator}.
         *
         * @param supplier supplier of
         *                 activator options, or leave blank to use defaults
         * @return updated builder instance
         * @see #activationRequest()
         */
        public BUILDER activationRequest(Supplier<? extends ActivationRequest> supplier) {
            Objects.requireNonNull(supplier);
            this.activationRequest(supplier.get());
            return self();
        }

        /**
         * The strategy the injector should apply. The default is {@link Injector.Strategy#ANY}.
         *
         * @return the strategy
         */
        public Injector.Strategy strategy() {
            return strategy;
        }

        /**
         * Optionally, customized activator options to use for the {@link Activator}.
         *
         * @return the activation request
         */
        public Optional<ActivationRequest> activationRequest() {
            return Optional.ofNullable(activationRequest);
        }

        @Override
        public String toString() {
            return "InjectorOptionsBuilder{"
                    + "strategy=" + strategy + ","
                    + "activationRequest=" + activationRequest
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            new InjectorOptionsBlueprint.BuilderDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (activationRequest == null) {
                collector.fatal(getClass(), "Property \"activationRequest\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class InjectorOptionsImpl implements InjectorOptions {

            private final ActivationRequest activationRequest;
            private final Injector.Strategy strategy;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected InjectorOptionsImpl(InjectorOptions.BuilderBase<?, ?> builder) {
                this.strategy = builder.strategy();
                this.activationRequest = builder.activationRequest().get();
            }

            @Override
            public Injector.Strategy strategy() {
                return strategy;
            }

            @Override
            public ActivationRequest activationRequest() {
                return activationRequest;
            }

            @Override
            public String toString() {
                return "InjectorOptions{"
                        + "strategy=" + strategy + ","
                        + "activationRequest=" + activationRequest
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof InjectorOptions other)) {
                    return false;
                }
                return Objects.equals(strategy, other.strategy())
                        && Objects.equals(activationRequest, other.activationRequest());
            }

            @Override
            public int hashCode() {
                return Objects.hash(strategy, activationRequest);
            }

        }

    }

    /**
     * Fluent API builder for {@link InjectorOptions}.
     */
    class Builder extends InjectorOptions.BuilderBase<InjectorOptions.Builder, InjectorOptions> implements io.helidon.common.Builder<InjectorOptions.Builder, InjectorOptions> {

        private Builder() {
        }

        @Override
        public InjectorOptions buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new InjectorOptionsImpl(this);
        }

        @Override
        public InjectorOptions build() {
            return buildPrototype();
        }

    }

}
