/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.inject.api;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;

/**
 * Internal bootstrap is what we store when {@link InjectionServices#globalBootstrap(Bootstrap)} is used.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.inject.api.InternalBootstrapBlueprint")
public interface InternalBootstrap extends InternalBootstrapBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static InternalBootstrap.Builder builder() {
        return new InternalBootstrap.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static InternalBootstrap.Builder builder(InternalBootstrap instance) {
        return InternalBootstrap.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static InternalBootstrap create() {
        return InternalBootstrap.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link InternalBootstrap}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends InternalBootstrap.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends InternalBootstrap> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private Bootstrap bootStrap;
        private CallingContext callingContext;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(InternalBootstrap prototype) {
            bootStrap(prototype.bootStrap());
            callingContext(prototype.callingContext());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(InternalBootstrap.BuilderBase<?, ?> builder) {
            builder.bootStrap().ifPresent(this::bootStrap);
            builder.callingContext().ifPresent(this::callingContext);
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #bootStrap()
         */
        public BUILDER clearBootStrap() {
            this.bootStrap = null;
            return self();
        }

        /**
         * The user established bootstrap.
         *
         * @param bootStrap user established bootstrap
         * @return updated builder instance
         * @see #bootStrap()
         */
        public BUILDER bootStrap(Bootstrap bootStrap) {
            Objects.requireNonNull(bootStrap);
            this.bootStrap = bootStrap;
            return self();
        }

        /**
         * The user established bootstrap.
         *
         * @param consumer user established bootstrap
         * @return updated builder instance
         * @see #bootStrap()
         */
        public BUILDER bootStrap(Consumer<Bootstrap.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = Bootstrap.builder();
            consumer.accept(builder);
            this.bootStrap(builder.build());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #callingContext()
         */
        public BUILDER clearCallingContext() {
            this.callingContext = null;
            return self();
        }

        /**
         * Only populated when {@link InjectionServices#TAG_DEBUG} is set.
         *
         * @param callingContext the calling context
         * @return updated builder instance
         * @see #callingContext()
         */
        public BUILDER callingContext(CallingContext callingContext) {
            Objects.requireNonNull(callingContext);
            this.callingContext = callingContext;
            return self();
        }

        /**
         * Only populated when {@link InjectionServices#TAG_DEBUG} is set.
         *
         * @param consumer the calling context
         * @return updated builder instance
         * @see #callingContext()
         */
        public BUILDER callingContext(Consumer<CallingContext.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = CallingContext.builder();
            consumer.accept(builder);
            this.callingContext(builder.build());
            return self();
        }

        /**
         * The user established bootstrap.
         *
         * @return the boot strap
         */
        public Optional<Bootstrap> bootStrap() {
            return Optional.ofNullable(bootStrap);
        }

        /**
         * Only populated when {@link InjectionServices#TAG_DEBUG} is set.
         *
         * @return the calling context
         */
        public Optional<CallingContext> callingContext() {
            return Optional.ofNullable(callingContext);
        }

        @Override
        public String toString() {
            return "InternalBootstrapBuilder{"
                    + "bootStrap=" + bootStrap + ","
                    + "callingContext=" + callingContext
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            new InternalBootstrapBlueprint.BuilderDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * The user established bootstrap.
         *
         * @param bootStrap user established bootstrap
         * @return updated builder instance
         * @see #bootStrap()
         */
        BUILDER bootStrap(Optional<? extends Bootstrap> bootStrap) {
            Objects.requireNonNull(bootStrap);
            this.bootStrap = bootStrap.map(Bootstrap.class::cast).orElse(this.bootStrap);
            return self();
        }

        /**
         * Only populated when {@link InjectionServices#TAG_DEBUG} is set.
         *
         * @param callingContext the calling context
         * @return updated builder instance
         * @see #callingContext()
         */
        BUILDER callingContext(Optional<? extends CallingContext> callingContext) {
            Objects.requireNonNull(callingContext);
            this.callingContext = callingContext.map(CallingContext.class::cast).orElse(this.callingContext);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class InternalBootstrapImpl implements InternalBootstrap {

            private final Optional<Bootstrap> bootStrap;
            private final Optional<CallingContext> callingContext;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected InternalBootstrapImpl(InternalBootstrap.BuilderBase<?, ?> builder) {
                this.bootStrap = builder.bootStrap();
                this.callingContext = builder.callingContext();
            }

            @Override
            public Optional<Bootstrap> bootStrap() {
                return bootStrap;
            }

            @Override
            public Optional<CallingContext> callingContext() {
                return callingContext;
            }

            @Override
            public String toString() {
                return "InternalBootstrap{"
                        + "bootStrap=" + bootStrap + ","
                        + "callingContext=" + callingContext
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof InternalBootstrap other)) {
                    return false;
                }
                return Objects.equals(bootStrap, other.bootStrap())
                        && Objects.equals(callingContext, other.callingContext());
            }

            @Override
            public int hashCode() {
                return Objects.hash(bootStrap, callingContext);
            }

        }

    }

    /**
     * Fluent API builder for {@link InternalBootstrap}.
     */
    class Builder extends InternalBootstrap.BuilderBase<InternalBootstrap.Builder, InternalBootstrap> implements io.helidon.common.Builder<InternalBootstrap.Builder, InternalBootstrap> {

        private Builder() {
        }

        @Override
        public InternalBootstrap buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new InternalBootstrapImpl(this);
        }

        @Override
        public InternalBootstrap build() {
            return buildPrototype();
        }

    }

}
