/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.inject.api;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.types.TypeName;

/**
 * Describes a managed service or injection point.
 *
 * @see Services
 * @see ServiceInfoCriteria
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.inject.api.ServiceInfoBlueprint")
public interface ServiceInfo extends ServiceInfoBlueprint, Prototype.Api, ServiceInfoBasics {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static ServiceInfo.Builder builder() {
        return new ServiceInfo.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static ServiceInfo.Builder builder(ServiceInfo instance) {
        return ServiceInfo.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static ServiceInfo create() {
        return ServiceInfo.builder().buildPrototype();
    }

    /**
     * Create a builder of service info from its basics counterpart.
     *
     * @param prototype instance to copy
     * @return a new builder with data from prototype
     */
    static ServiceInfo.Builder builder(ServiceInfoBasics prototype) {
        return ServiceInfoBlueprint.CustomMethods.builder(prototype);
    }

    /**
     * Fluent API builder base for {@link ServiceInfo}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends ServiceInfo.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends ServiceInfo> extends ServiceInfoBasics.BuilderBase<BUILDER, PROTOTYPE> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private final Set<TypeName> externalContractsImplemented = new LinkedHashSet<>();
        private String moduleName;
        private TypeName activatorTypeName;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ServiceInfo prototype) {
            super.from(prototype);
            addExternalContractsImplemented(prototype.externalContractsImplemented());
            activatorTypeName(prototype.activatorTypeName());
            moduleName(prototype.moduleName());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ServiceInfo.BuilderBase<?, ?> builder) {
            super.from(builder);
            addExternalContractsImplemented(builder.externalContractsImplemented());
            builder.activatorTypeName().ifPresent(this::activatorTypeName);
            builder.moduleName().ifPresent(this::moduleName);
            return self();
        }

        /**
         * Add external contract implemented.
         *
         * @param type type of the external contract
         * @return updated builder instance
         */
        public BUILDER addExternalContractImplemented(Class<?> type) {
            ServiceInfoBlueprint.CustomMethods.addExternalContractImplemented(this, type);
            return self();
        }

        /**
         * Activator type.
         *
         * @param type type of the activator
         * @return updated builder instance
         */
        public BUILDER activatorTypeName(Class<?> type) {
            ServiceInfoBlueprint.CustomMethods.activatorTypeName(this, type);
            return self();
        }

        /**
         * Add a scope type.
         *
         * @param type type of the scope
         * @return updated builder instance
         */
        public BUILDER addScopeTypeName(Class<?> type) {
            ServiceInfoBlueprint.CustomMethods.addScopeTypeName(this, type);
            return self();
        }

        /**
         * The managed services external contracts / interfaces. These should also be contained within
         * {@link #contractsImplemented()}. External contracts are from other modules other than the module containing
         * the implementation typically.
         *
         * @param externalContractsImplemented the service external contracts implemented
         * @return updated builder instance
         * @see #externalContractsImplemented()
         */
        public BUILDER externalContractsImplemented(Set<? extends TypeName> externalContractsImplemented) {
            Objects.requireNonNull(externalContractsImplemented);
            this.externalContractsImplemented.clear();
            this.externalContractsImplemented.addAll(externalContractsImplemented);
            return self();
        }

        /**
         * The managed services external contracts / interfaces. These should also be contained within
         * {@link #contractsImplemented()}. External contracts are from other modules other than the module containing
         * the implementation typically.
         *
         * @param externalContractsImplemented the service external contracts implemented
         * @return updated builder instance
         * @see #externalContractsImplemented()
         */
        public BUILDER addExternalContractsImplemented(Set<? extends TypeName> externalContractsImplemented) {
            Objects.requireNonNull(externalContractsImplemented);
            this.externalContractsImplemented.addAll(externalContractsImplemented);
            return self();
        }

        /**
         * The managed services external contracts / interfaces. These should also be contained within
         * {@link #contractsImplemented()}. External contracts are from other modules other than the module containing
         * the implementation typically.
         *
         * @param externalContractImplemented the service external contracts implemented
         * @return updated builder instance
         * @see ExternalContracts
         * @see #externalContractsImplemented()
         */
        public BUILDER addExternalContractImplemented(TypeName externalContractImplemented) {
            Objects.requireNonNull(externalContractImplemented);
            this.externalContractsImplemented.add(externalContractImplemented);
            return self();
        }

        /**
         * The managed services external contracts / interfaces. These should also be contained within
         * {@link #contractsImplemented()}. External contracts are from other modules other than the module containing
         * the implementation typically.
         *
         * @param consumer the service external contracts implemented
         * @return updated builder instance
         * @see ExternalContracts
         * @see #externalContractsImplemented()
         * @see #externalContractsImplemented()
         */
        public BUILDER addExternalContractImplemented(Consumer<TypeName.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = TypeName.builder();
            consumer.accept(builder);
            this.externalContractsImplemented.add(builder.build());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #activatorTypeName()
         */
        public BUILDER clearActivatorTypeName() {
            this.activatorTypeName = null;
            return self();
        }

        /**
         * The management agent (i.e., the activator) that is responsible for creating and activating - typically build-time created.
         *
         * @param activatorTypeName the activator type name
         * @return updated builder instance
         * @see #activatorTypeName()
         */
        public BUILDER activatorTypeName(TypeName activatorTypeName) {
            Objects.requireNonNull(activatorTypeName);
            this.activatorTypeName = activatorTypeName;
            return self();
        }

        /**
         * The management agent (i.e., the activator) that is responsible for creating and activating - typically build-time created.
         *
         * @param consumer the activator type name
         * @return updated builder instance
         * @see #activatorTypeName()
         */
        public BUILDER activatorTypeName(Consumer<TypeName.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = TypeName.builder();
            consumer.accept(builder);
            this.activatorTypeName(builder.build());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #moduleName()
         */
        public BUILDER clearModuleName() {
            this.moduleName = null;
            return self();
        }

        /**
         * The name of the ascribed module, if known.
         *
         * @param moduleName the module name
         * @return updated builder instance
         * @see #moduleName()
         */
        public BUILDER moduleName(String moduleName) {
            Objects.requireNonNull(moduleName);
            this.moduleName = moduleName;
            return self();
        }

        /**
         * The managed services external contracts / interfaces. These should also be contained within
         * {@link #contractsImplemented()}. External contracts are from other modules other than the module containing
         * the implementation typically.
         *
         * @return the external contracts implemented
         * @see ExternalContracts
         * @see #externalContractsImplemented()
         * @see #externalContractsImplemented()
         */
        public Set<TypeName> externalContractsImplemented() {
            return externalContractsImplemented;
        }

        /**
         * The management agent (i.e., the activator) that is responsible for creating and activating - typically build-time created.
         *
         * @return the activator type name
         */
        public Optional<TypeName> activatorTypeName() {
            return Optional.ofNullable(activatorTypeName);
        }

        /**
         * The name of the ascribed module, if known.
         *
         * @return the module name
         */
        public Optional<String> moduleName() {
            return Optional.ofNullable(moduleName);
        }

        @Override
        public String toString() {
            return "ServiceInfoBuilder{"
                    + "externalContractsImplemented=" + externalContractsImplemented + ","
                    + "activatorTypeName=" + activatorTypeName + ","
                    + "moduleName=" + moduleName
                    + "};"
                    + super.toString();
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            super.preBuildPrototype();
            new ServiceInfoBuildDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            super.validatePrototype();
        }

        /**
         * The management agent (i.e., the activator) that is responsible for creating and activating - typically build-time created.
         *
         * @param activatorTypeName the activator type name
         * @return updated builder instance
         * @see #activatorTypeName()
         */
        BUILDER activatorTypeName(Optional<? extends TypeName> activatorTypeName) {
            Objects.requireNonNull(activatorTypeName);
            this.activatorTypeName = activatorTypeName.map(io.helidon.common.types.TypeName.class::cast).orElse(this.activatorTypeName);
            return self();
        }

        /**
         * The name of the ascribed module, if known.
         *
         * @param moduleName the module name
         * @return updated builder instance
         * @see #moduleName()
         */
        BUILDER moduleName(Optional<String> moduleName) {
            Objects.requireNonNull(moduleName);
            this.moduleName = moduleName.map(java.lang.String.class::cast).orElse(this.moduleName);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class ServiceInfoImpl extends ServiceInfoBasicsImpl implements ServiceInfo {

            private final Optional<TypeName> activatorTypeName;
            private final Optional<String> moduleName;
            private final Set<TypeName> externalContractsImplemented;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected ServiceInfoImpl(ServiceInfo.BuilderBase<?, ?> builder) {
                super(builder);
                this.externalContractsImplemented = Collections.unmodifiableSet(new LinkedHashSet<>(builder.externalContractsImplemented()));
                this.activatorTypeName = builder.activatorTypeName();
                this.moduleName = builder.moduleName();
            }

            @Override
            public Set<TypeName> externalContractsImplemented() {
                return externalContractsImplemented;
            }

            @Override
            public Optional<TypeName> activatorTypeName() {
                return activatorTypeName;
            }

            @Override
            public Optional<String> moduleName() {
                return moduleName;
            }

            @Override
            public String toString() {
                return "ServiceInfo{"
                        + "externalContractsImplemented=" + externalContractsImplemented + ","
                        + "activatorTypeName=" + activatorTypeName + ","
                        + "moduleName=" + moduleName
                        + "};"
                        + super.toString();
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof ServiceInfo other)) {
                    return false;
                }
                return super.equals(other)
                        && Objects.equals(externalContractsImplemented, other.externalContractsImplemented())
                        && Objects.equals(activatorTypeName, other.activatorTypeName())
                        && Objects.equals(moduleName, other.moduleName());
            }

            @Override
            public int hashCode() {
                return 31 * super.hashCode() + Objects.hash(externalContractsImplemented, activatorTypeName, moduleName);
            }

        }

    }

    /**
     * Fluent API builder for {@link ServiceInfo}.
     */
    class Builder extends ServiceInfo.BuilderBase<ServiceInfo.Builder, ServiceInfo> implements io.helidon.common.Builder<ServiceInfo.Builder, ServiceInfo> {

        private Builder() {
        }

        @Override
        public ServiceInfo buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new ServiceInfoImpl(this);
        }

        @Override
        public ServiceInfo build() {
            return buildPrototype();
        }

    }

}
