/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.inject.api;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;
import io.helidon.common.types.TypeName;

/**
 * Basic service info that describes a service provider type.
 *
 * @see ServiceInfo
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.inject.api.ServiceInfoBasicsBlueprint")
public interface ServiceInfoBasics extends ServiceInfoBasicsBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static ServiceInfoBasics.Builder builder() {
        return new ServiceInfoBasics.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static ServiceInfoBasics.Builder builder(ServiceInfoBasics instance) {
        return ServiceInfoBasics.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static ServiceInfoBasics create() {
        return ServiceInfoBasics.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link ServiceInfoBasics}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends ServiceInfoBasics.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends ServiceInfoBasics> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private final Set<Qualifier> qualifiers = new LinkedHashSet<>();
        private final Set<TypeName> contractsImplemented = new LinkedHashSet<>();
        private final Set<TypeName> scopeTypeNames = new LinkedHashSet<>();
        private Double declaredWeight;
        private Integer declaredRunLevel;
        private TypeName serviceTypeName;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ServiceInfoBasics prototype) {
            serviceTypeName(prototype.serviceTypeName());
            addScopeTypeNames(prototype.scopeTypeNames());
            addQualifiers(prototype.qualifiers());
            addContractsImplemented(prototype.contractsImplemented());
            declaredRunLevel(prototype.declaredRunLevel());
            declaredWeight(prototype.declaredWeight());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ServiceInfoBasics.BuilderBase<?, ?> builder) {
            builder.serviceTypeName().ifPresent(this::serviceTypeName);
            addScopeTypeNames(builder.scopeTypeNames());
            addQualifiers(builder.qualifiers());
            addContractsImplemented(builder.contractsImplemented());
            builder.declaredRunLevel().ifPresent(this::declaredRunLevel);
            builder.declaredWeight().ifPresent(this::declaredWeight);
            return self();
        }

        /**
         * The managed service implementation type name.
         *
         * @param type type of the service
         * @return updated builder instance
         */
        public BUILDER serviceTypeName(Class<?> type) {
            ServiceInfoBasicsBlueprint.CustomMethods.serviceTypeName(this, type);
            return self();
        }

        /**
         * Add contract implemented.
         *
         * @param type type of the service
         * @return updated builder instance
         */
        public BUILDER addContractImplemented(Class<?> type) {
            ServiceInfoBasicsBlueprint.CustomMethods.addContractImplemented(this, type);
            return self();
        }

        /**
         * The managed service implementation {@link Class}.
         *
         * @param serviceTypeName the service type name
         * @return updated builder instance
         * @see #serviceTypeName()
         */
        public BUILDER serviceTypeName(TypeName serviceTypeName) {
            Objects.requireNonNull(serviceTypeName);
            this.serviceTypeName = serviceTypeName;
            return self();
        }

        /**
         * The managed service implementation {@link Class}.
         *
         * @param consumer consumer of builder for
         *                 the service type name
         * @return updated builder instance
         * @see #serviceTypeName()
         */
        public BUILDER serviceTypeName(Consumer<TypeName.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = TypeName.builder();
            consumer.accept(builder);
            this.serviceTypeName(builder.build());
            return self();
        }

        /**
         * The managed service implementation {@link Class}.
         *
         * @param supplier supplier of
         *                 the service type name
         * @return updated builder instance
         * @see #serviceTypeName()
         */
        public BUILDER serviceTypeName(Supplier<? extends TypeName> supplier) {
            Objects.requireNonNull(supplier);
            this.serviceTypeName(supplier.get());
            return self();
        }

        /**
         * The managed service assigned Scope's.
         *
         * @param scopeTypeNames the service scope type name
         * @return updated builder instance
         * @see #scopeTypeNames()
         */
        public BUILDER scopeTypeNames(Set<? extends TypeName> scopeTypeNames) {
            Objects.requireNonNull(scopeTypeNames);
            this.scopeTypeNames.clear();
            this.scopeTypeNames.addAll(scopeTypeNames);
            return self();
        }

        /**
         * The managed service assigned Scope's.
         *
         * @param scopeTypeNames the service scope type name
         * @return updated builder instance
         * @see #scopeTypeNames()
         */
        public BUILDER addScopeTypeNames(Set<? extends TypeName> scopeTypeNames) {
            Objects.requireNonNull(scopeTypeNames);
            this.scopeTypeNames.addAll(scopeTypeNames);
            return self();
        }

        /**
         * The managed service assigned Scope's.
         *
         * @param scopeTypeName the service scope type name
         * @return updated builder instance
         * @see #scopeTypeNames()
         */
        public BUILDER addScopeTypeName(TypeName scopeTypeName) {
            Objects.requireNonNull(scopeTypeName);
            this.scopeTypeNames.add(scopeTypeName);
            return self();
        }

        /**
         * The managed service assigned Scope's.
         *
         * @param consumer the service scope type name
         * @return updated builder instance
         * @see #scopeTypeNames()
         */
        public BUILDER addScopeTypeName(Consumer<TypeName.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = TypeName.builder();
            consumer.accept(builder);
            this.scopeTypeNames.add(builder.build());
            return self();
        }

        /**
         * The managed service assigned Qualifier's.
         *
         * @param qualifiers the service qualifiers
         * @return updated builder instance
         * @see #qualifiers()
         */
        public BUILDER qualifiers(Set<? extends Qualifier> qualifiers) {
            Objects.requireNonNull(qualifiers);
            this.qualifiers.clear();
            this.qualifiers.addAll(qualifiers);
            return self();
        }

        /**
         * The managed service assigned Qualifier's.
         *
         * @param qualifiers the service qualifiers
         * @return updated builder instance
         * @see #qualifiers()
         */
        public BUILDER addQualifiers(Set<? extends Qualifier> qualifiers) {
            Objects.requireNonNull(qualifiers);
            this.qualifiers.addAll(qualifiers);
            return self();
        }

        /**
         * The managed service assigned Qualifier's.
         *
         * @param qualifier the service qualifiers
         * @return updated builder instance
         * @see #qualifiers()
         */
        public BUILDER addQualifier(Qualifier qualifier) {
            Objects.requireNonNull(qualifier);
            this.qualifiers.add(qualifier);
            return self();
        }

        /**
         * The managed service assigned Qualifier's.
         *
         * @param consumer the service qualifiers
         * @return updated builder instance
         * @see #qualifiers()
         */
        public BUILDER addQualifier(Consumer<Qualifier.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = Qualifier.builder();
            consumer.accept(builder);
            this.qualifiers.add(builder.build());
            return self();
        }

        /**
         * The managed services advertised types (i.e., typically its interfaces).
         *
         * @param contractsImplemented the service contracts implemented
         * @return updated builder instance
         * @see #contractsImplemented()
         */
        public BUILDER contractsImplemented(Set<? extends TypeName> contractsImplemented) {
            Objects.requireNonNull(contractsImplemented);
            this.contractsImplemented.clear();
            this.contractsImplemented.addAll(contractsImplemented);
            return self();
        }

        /**
         * The managed services advertised types (i.e., typically its interfaces).
         *
         * @param contractsImplemented the service contracts implemented
         * @return updated builder instance
         * @see #contractsImplemented()
         */
        public BUILDER addContractsImplemented(Set<? extends TypeName> contractsImplemented) {
            Objects.requireNonNull(contractsImplemented);
            this.contractsImplemented.addAll(contractsImplemented);
            return self();
        }

        /**
         * The managed services advertised types (i.e., typically its interfaces).
         *
         * @param contractImplemented the service contracts implemented
         * @return updated builder instance
         * @see ExternalContracts
         * @see #contractsImplemented()
         */
        public BUILDER addContractImplemented(TypeName contractImplemented) {
            Objects.requireNonNull(contractImplemented);
            this.contractsImplemented.add(contractImplemented);
            return self();
        }

        /**
         * The managed services advertised types (i.e., typically its interfaces).
         *
         * @param consumer the service contracts implemented
         * @return updated builder instance
         * @see ExternalContracts
         * @see #contractsImplemented()
         * @see #contractsImplemented()
         */
        public BUILDER addContractImplemented(Consumer<TypeName.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = TypeName.builder();
            consumer.accept(builder);
            this.contractsImplemented.add(builder.build());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #declaredRunLevel()
         */
        public BUILDER clearDeclaredRunLevel() {
            this.declaredRunLevel = null;
            return self();
        }

        /**
         * The optional {@link RunLevel} ascribed to the service.
         *
         * @param declaredRunLevel the service's run level
         * @return updated builder instance
         * @see #declaredRunLevel()
         */
        public BUILDER declaredRunLevel(int declaredRunLevel) {
            Objects.requireNonNull(declaredRunLevel);
            this.declaredRunLevel = declaredRunLevel;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #declaredWeight()
         */
        public BUILDER clearDeclaredWeight() {
            this.declaredWeight = null;
            return self();
        }

        /**
         * Weight that was declared on the type itself.
         *
         * @param declaredWeight the declared weight
         * @return updated builder instance
         * @see #declaredWeight()
         */
        public BUILDER declaredWeight(double declaredWeight) {
            Objects.requireNonNull(declaredWeight);
            this.declaredWeight = declaredWeight;
            return self();
        }

        /**
         * The managed service implementation {@link Class}.
         *
         * @return the service type name
         */
        public Optional<TypeName> serviceTypeName() {
            return Optional.ofNullable(serviceTypeName);
        }

        /**
         * The managed service assigned Scope's.
         *
         * @return the scope type names
         */
        public Set<TypeName> scopeTypeNames() {
            return scopeTypeNames;
        }

        /**
         * The managed service assigned Qualifier's.
         *
         * @return the qualifiers
         */
        public Set<Qualifier> qualifiers() {
            return qualifiers;
        }

        /**
         * The managed services advertised types (i.e., typically its interfaces).
         *
         * @return the contracts implemented
         * @see ExternalContracts
         * @see #contractsImplemented()
         * @see #contractsImplemented()
         */
        public Set<TypeName> contractsImplemented() {
            return contractsImplemented;
        }

        /**
         * The optional {@link RunLevel} ascribed to the service.
         *
         * @return the declared run level
         * @see #realizedRunLevel()
         */
        public Optional<Integer> declaredRunLevel() {
            return Optional.ofNullable(declaredRunLevel);
        }

        /**
         * Weight that was declared on the type itself.
         *
         * @return the declared weight
         * @see #realizedWeight()
         */
        public Optional<Double> declaredWeight() {
            return Optional.ofNullable(declaredWeight);
        }

        @Override
        public String toString() {
            return "ServiceInfoBasicsBuilder{"
                    + "serviceTypeName=" + serviceTypeName + ","
                    + "scopeTypeNames=" + scopeTypeNames + ","
                    + "qualifiers=" + qualifiers + ","
                    + "contractsImplemented=" + contractsImplemented + ","
                    + "declaredRunLevel=" + declaredRunLevel + ","
                    + "declaredWeight=" + declaredWeight
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (serviceTypeName == null) {
                collector.fatal(getClass(), "Property \"serviceTypeName\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * The optional {@link RunLevel} ascribed to the service.
         *
         * @param declaredRunLevel the service's run level
         * @return updated builder instance
         * @see #declaredRunLevel()
         */
        BUILDER declaredRunLevel(Optional<Integer> declaredRunLevel) {
            Objects.requireNonNull(declaredRunLevel);
            this.declaredRunLevel = declaredRunLevel.map(java.lang.Integer.class::cast).orElse(this.declaredRunLevel);
            return self();
        }

        /**
         * Weight that was declared on the type itself.
         *
         * @param declaredWeight the declared weight
         * @return updated builder instance
         * @see #declaredWeight()
         */
        BUILDER declaredWeight(Optional<Double> declaredWeight) {
            Objects.requireNonNull(declaredWeight);
            this.declaredWeight = declaredWeight.map(java.lang.Double.class::cast).orElse(this.declaredWeight);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class ServiceInfoBasicsImpl implements ServiceInfoBasics {

            private final Optional<Double> declaredWeight;
            private final Optional<Integer> declaredRunLevel;
            private final Set<Qualifier> qualifiers;
            private final Set<TypeName> contractsImplemented;
            private final Set<TypeName> scopeTypeNames;
            private final TypeName serviceTypeName;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected ServiceInfoBasicsImpl(ServiceInfoBasics.BuilderBase<?, ?> builder) {
                this.serviceTypeName = builder.serviceTypeName().get();
                this.scopeTypeNames = Collections.unmodifiableSet(new LinkedHashSet<>(builder.scopeTypeNames()));
                this.qualifiers = Collections.unmodifiableSet(new LinkedHashSet<>(builder.qualifiers()));
                this.contractsImplemented = Collections.unmodifiableSet(new LinkedHashSet<>(builder.contractsImplemented()));
                this.declaredRunLevel = builder.declaredRunLevel();
                this.declaredWeight = builder.declaredWeight();
            }

            @Override
            public TypeName serviceTypeName() {
                return serviceTypeName;
            }

            @Override
            public Set<TypeName> scopeTypeNames() {
                return scopeTypeNames;
            }

            @Override
            public Set<Qualifier> qualifiers() {
                return qualifiers;
            }

            @Override
            public Set<TypeName> contractsImplemented() {
                return contractsImplemented;
            }

            @Override
            public Optional<Integer> declaredRunLevel() {
                return declaredRunLevel;
            }

            @Override
            public Optional<Double> declaredWeight() {
                return declaredWeight;
            }

            @Override
            public String toString() {
                return "ServiceInfoBasics{"
                        + "serviceTypeName=" + serviceTypeName + ","
                        + "scopeTypeNames=" + scopeTypeNames + ","
                        + "qualifiers=" + qualifiers + ","
                        + "contractsImplemented=" + contractsImplemented + ","
                        + "declaredRunLevel=" + declaredRunLevel + ","
                        + "declaredWeight=" + declaredWeight
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof ServiceInfoBasics other)) {
                    return false;
                }
                return Objects.equals(serviceTypeName, other.serviceTypeName())
                        && Objects.equals(scopeTypeNames, other.scopeTypeNames())
                        && Objects.equals(qualifiers, other.qualifiers())
                        && Objects.equals(contractsImplemented, other.contractsImplemented())
                        && Objects.equals(declaredRunLevel, other.declaredRunLevel())
                        && Objects.equals(declaredWeight, other.declaredWeight());
            }

            @Override
            public int hashCode() {
                return Objects.hash(serviceTypeName, scopeTypeNames, qualifiers, contractsImplemented, declaredRunLevel, declaredWeight);
            }

        }

    }

    /**
     * Fluent API builder for {@link ServiceInfoBasics}.
     */
    class Builder extends ServiceInfoBasics.BuilderBase<ServiceInfoBasics.Builder, ServiceInfoBasics> implements io.helidon.common.Builder<ServiceInfoBasics.Builder, ServiceInfoBasics> {

        private Builder() {
        }

        @Override
        public ServiceInfoBasics buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new ServiceInfoBasicsImpl(this);
        }

        @Override
        public ServiceInfoBasics build() {
            return buildPrototype();
        }

    }

}
