/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.inject.spi;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;
import io.helidon.inject.api.InjectionPointInfo;
import io.helidon.inject.api.ServiceProvider;

/**
 * Represents the injection plan targeting a given {@link io.helidon.inject.api.ServiceProvider}.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.inject.spi.InjectionPlanBlueprint")
public interface InjectionPlan extends InjectionPlanBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static InjectionPlan.Builder builder() {
        return new InjectionPlan.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static InjectionPlan.Builder builder(InjectionPlan instance) {
        return InjectionPlan.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static InjectionPlan create() {
        return InjectionPlan.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link InjectionPlan}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends InjectionPlan.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends InjectionPlan> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private final List<ServiceProvider<?>> injectionPointQualifiedServiceProviders = new ArrayList<>();
        private boolean wasResolved = false;
        private InjectionPointInfo injectionPointInfo;
        private Object resolved;
        private ServiceProvider<?> serviceProvider;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(InjectionPlan prototype) {
            serviceProvider(prototype.serviceProvider());
            injectionPointInfo(prototype.injectionPointInfo());
            addInjectionPointQualifiedServiceProviders(prototype.injectionPointQualifiedServiceProviders());
            wasResolved(prototype.wasResolved());
            resolved(prototype.resolved());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(InjectionPlan.BuilderBase<?, ?> builder) {
            builder.serviceProvider().ifPresent(this::serviceProvider);
            builder.injectionPointInfo().ifPresent(this::injectionPointInfo);
            addInjectionPointQualifiedServiceProviders(builder.injectionPointQualifiedServiceProviders());
            wasResolved(builder.wasResolved());
            builder.resolved().ifPresent(this::resolved);
            return self();
        }

        /**
         * The service provider this plan pertains to.
         *
         * @param serviceProvider the service provider this plan pertains to
         * @return updated builder instance
         * @see #serviceProvider()
         */
        public BUILDER serviceProvider(ServiceProvider<?> serviceProvider) {
            Objects.requireNonNull(serviceProvider);
            this.serviceProvider = serviceProvider;
            return self();
        }

        /**
         * The injection point info for this element, which will also include its identity information.
         *
         * @param injectionPointInfo the injection point info for this element
         * @return updated builder instance
         * @see #injectionPointInfo()
         */
        public BUILDER injectionPointInfo(InjectionPointInfo injectionPointInfo) {
            Objects.requireNonNull(injectionPointInfo);
            this.injectionPointInfo = injectionPointInfo;
            return self();
        }

        /**
         * The injection point info for this element, which will also include its identity information.
         *
         * @param consumer consumer of builder for
         *                 the injection point info for this element
         * @return updated builder instance
         * @see #injectionPointInfo()
         */
        public BUILDER injectionPointInfo(Consumer<InjectionPointInfo.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = InjectionPointInfo.builder();
            consumer.accept(builder);
            this.injectionPointInfo(builder.build());
            return self();
        }

        /**
         * The injection point info for this element, which will also include its identity information.
         *
         * @param supplier supplier of
         *                 the injection point info for this element
         * @return updated builder instance
         * @see #injectionPointInfo()
         */
        public BUILDER injectionPointInfo(Supplier<? extends InjectionPointInfo> supplier) {
            Objects.requireNonNull(supplier);
            this.injectionPointInfo(supplier.get());
            return self();
        }

        /**
         * The list of service providers that are qualified to satisfy the given injection point for this service provider.
         *
         * @param injectionPointQualifiedServiceProviders the qualified service providers for this injection point
         * @return updated builder instance
         * @see #injectionPointQualifiedServiceProviders()
         */
        public BUILDER injectionPointQualifiedServiceProviders(List<ServiceProvider<?>> injectionPointQualifiedServiceProviders) {
            Objects.requireNonNull(injectionPointQualifiedServiceProviders);
            this.injectionPointQualifiedServiceProviders.clear();
            this.injectionPointQualifiedServiceProviders.addAll(injectionPointQualifiedServiceProviders);
            return self();
        }

        /**
         * The list of service providers that are qualified to satisfy the given injection point for this service provider.
         *
         * @param injectionPointQualifiedServiceProviders the qualified service providers for this injection point
         * @return updated builder instance
         * @see #injectionPointQualifiedServiceProviders()
         */
        public BUILDER addInjectionPointQualifiedServiceProviders(List<ServiceProvider<?>> injectionPointQualifiedServiceProviders) {
            Objects.requireNonNull(injectionPointQualifiedServiceProviders);
            this.injectionPointQualifiedServiceProviders.addAll(injectionPointQualifiedServiceProviders);
            return self();
        }

        /**
         * The list of service providers that are qualified to satisfy the given injection point for this service provider.
         *
         * @param injectionPointQualifiedServiceProvider the qualified service providers for this injection point
         * @return updated builder instance
         * @see #injectionPointQualifiedServiceProviders()
         */
        public BUILDER addInjectionPointQualifiedServiceProvider(ServiceProvider<?> injectionPointQualifiedServiceProvider) {
            Objects.requireNonNull(injectionPointQualifiedServiceProvider);
            this.injectionPointQualifiedServiceProviders.add(injectionPointQualifiedServiceProvider);
            return self();
        }

        /**
         * Flag indicating whether resolution occurred.
         *
         * @param wasResolved true if resolution occurred
         * @return updated builder instance
         * @see #wasResolved()
         */
        public BUILDER wasResolved(boolean wasResolved) {
            this.wasResolved = wasResolved;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #resolved()
         */
        public BUILDER clearResolved() {
            this.resolved = null;
            return self();
        }

        /**
         * The resolved value, set only if {@link #wasResolved()}.
         *
         * @param resolved any resolved value
         * @return updated builder instance
         * @see #resolved()
         */
        public BUILDER resolved(Object resolved) {
            Objects.requireNonNull(resolved);
            this.resolved = resolved;
            return self();
        }

        /**
         * The service provider this plan pertains to.
         *
         * @return the service provider
         */
        public Optional<ServiceProvider<?>> serviceProvider() {
            return Optional.ofNullable(serviceProvider);
        }

        /**
         * The injection point info for this element, which will also include its identity information.
         *
         * @return the injection point info
         */
        public Optional<InjectionPointInfo> injectionPointInfo() {
            return Optional.ofNullable(injectionPointInfo);
        }

        /**
         * The list of service providers that are qualified to satisfy the given injection point for this service provider.
         *
         * @return the injection point qualified service providers
         */
        public List<ServiceProvider<?>> injectionPointQualifiedServiceProviders() {
            return injectionPointQualifiedServiceProviders;
        }

        /**
         * Flag indicating whether resolution occurred.
         *
         * @return the was resolved
         */
        public boolean wasResolved() {
            return wasResolved;
        }

        /**
         * The resolved value, set only if {@link #wasResolved()}.
         *
         * @return the resolved
         */
        public Optional<Object> resolved() {
            return Optional.ofNullable(resolved);
        }

        @Override
        public String toString() {
            return "InjectionPlanBuilder{"
                    + "serviceProvider=" + serviceProvider + ","
                    + "injectionPointInfo=" + injectionPointInfo + ","
                    + "injectionPointQualifiedServiceProviders=" + injectionPointQualifiedServiceProviders + ","
                    + "wasResolved=" + wasResolved + ","
                    + "resolved=" + resolved
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (serviceProvider == null) {
                collector.fatal(getClass(), "Property \"serviceProvider\" must not be null, but not set");
            }
            if (injectionPointInfo == null) {
                collector.fatal(getClass(), "Property \"injectionPointInfo\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * The resolved value, set only if {@link #wasResolved()}.
         *
         * @param resolved any resolved value
         * @return updated builder instance
         * @see #resolved()
         */
        BUILDER resolved(Optional<?> resolved) {
            Objects.requireNonNull(resolved);
            this.resolved = resolved.map(java.lang.Object.class::cast).orElse(this.resolved);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class InjectionPlanImpl implements InjectionPlan {

            private final boolean wasResolved;
            private final InjectionPointInfo injectionPointInfo;
            private final List<ServiceProvider<?>> injectionPointQualifiedServiceProviders;
            private final Optional<Object> resolved;
            private final ServiceProvider<?> serviceProvider;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected InjectionPlanImpl(InjectionPlan.BuilderBase<?, ?> builder) {
                this.serviceProvider = builder.serviceProvider().get();
                this.injectionPointInfo = builder.injectionPointInfo().get();
                this.injectionPointQualifiedServiceProviders = List.copyOf(builder.injectionPointQualifiedServiceProviders());
                this.wasResolved = builder.wasResolved();
                this.resolved = builder.resolved();
            }

            @Override
            public ServiceProvider<?> serviceProvider() {
                return serviceProvider;
            }

            @Override
            public InjectionPointInfo injectionPointInfo() {
                return injectionPointInfo;
            }

            @Override
            public List<ServiceProvider<?>> injectionPointQualifiedServiceProviders() {
                return injectionPointQualifiedServiceProviders;
            }

            @Override
            public boolean wasResolved() {
                return wasResolved;
            }

            @Override
            public Optional<Object> resolved() {
                return resolved;
            }

            @Override
            public String toString() {
                return "InjectionPlan{"
                        + "serviceProvider=" + serviceProvider + ","
                        + "injectionPointInfo=" + injectionPointInfo + ","
                        + "injectionPointQualifiedServiceProviders=" + injectionPointQualifiedServiceProviders + ","
                        + "wasResolved=" + wasResolved + ","
                        + "resolved=" + resolved
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof InjectionPlan other)) {
                    return false;
                }
                return Objects.equals(serviceProvider, other.serviceProvider())
                        && Objects.equals(injectionPointInfo, other.injectionPointInfo())
                        && Objects.equals(injectionPointQualifiedServiceProviders, other.injectionPointQualifiedServiceProviders())
                        && wasResolved == other.wasResolved()
                        && Objects.equals(resolved, other.resolved());
            }

            @Override
            public int hashCode() {
                return Objects.hash(serviceProvider, injectionPointInfo, injectionPointQualifiedServiceProviders, wasResolved, resolved);
            }

        }

    }

    /**
     * Fluent API builder for {@link InjectionPlan}.
     */
    class Builder extends InjectionPlan.BuilderBase<InjectionPlan.Builder, InjectionPlan> implements io.helidon.common.Builder<InjectionPlan.Builder, InjectionPlan> {

        private Builder() {
        }

        @Override
        public InjectionPlan buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new InjectionPlanImpl(this);
        }

        @Override
        public InjectionPlan build() {
            return buildPrototype();
        }

    }

}
