/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.security.jwt;

import io.helidon.common.Errors;
import io.helidon.security.jwt.Jwt;
import io.helidon.security.jwt.JwtException;
import io.helidon.security.jwt.JwtHeaders;
import io.helidon.security.jwt.jwk.Jwk;
import io.helidon.security.jwt.jwk.JwkKeys;
import jakarta.json.Json;
import jakarta.json.JsonObject;
import jakarta.json.JsonReaderFactory;
import java.io.Reader;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class SignedJwt {
    private static final Pattern JWT_PATTERN = Pattern.compile("([a-zA-Z0-9-_=]+)\\.([a-zA-Z0-9-_=]+)\\.([a-zA-Z0-9-_=]*)");
    private static final Base64.Decoder URL_DECODER = Base64.getUrlDecoder();
    private static final Base64.Encoder URL_ENCODER = Base64.getUrlEncoder().withoutPadding();
    private static final JsonReaderFactory JSON = Json.createReaderFactory(Collections.emptyMap());
    private final String tokenContent;
    private final JwtHeaders headers;
    private final JsonObject payloadJson;
    private final byte[] signedBytes;
    private final byte[] signature;

    private SignedJwt(String tokenContent, JwtHeaders headers, JsonObject payloadJson, byte[] signedBytes, byte[] signature) {
        this.tokenContent = tokenContent;
        this.headers = headers;
        this.payloadJson = payloadJson;
        this.signedBytes = signedBytes;
        this.signature = signature;
    }

    public static SignedJwt sign(Jwt jwt, JwkKeys jwks) throws JwtException {
        return jwt.algorithm().map(alg -> SignedJwt.sign(jwt, jwks, alg)).orElseGet(() -> jwt.keyId().map(kid -> jwks.forKeyId((String)kid).map(jwk -> SignedJwt.sign(jwt, jwk)).orElseThrow(() -> new JwtException("Could not find JWK based on key id. JWT: " + jwt + ", kid: " + kid))).orElseGet(() -> SignedJwt.sign(jwt, Jwk.NONE_JWK)));
    }

    public static SignedJwt sign(Jwt jwt, Jwk jwk) throws JwtException {
        JsonObject headerJson = jwt.headerJson();
        JsonObject payloadJson = jwt.payloadJson();
        String headerJsonString = headerJson.toString();
        String payloadJsonString = payloadJson.toString();
        String headerBase64 = SignedJwt.encode(headerJsonString);
        String payloadBase64 = SignedJwt.encode(payloadJsonString);
        String signedString = headerBase64 + "." + payloadBase64;
        byte[] signedBytes = signedString.getBytes(StandardCharsets.UTF_8);
        byte[] signature = jwk.sign(signedBytes);
        String signatureBase64 = SignedJwt.encode(signature);
        String tokenContent = signedString + "." + signatureBase64;
        return new SignedJwt(tokenContent, jwt.headers(), payloadJson, signedBytes, signature);
    }

    private static SignedJwt sign(Jwt jwt, JwkKeys jwks, String alg) {
        Jwk jwk = jwt.keyId().map(kid -> jwks.forKeyId((String)kid).orElseThrow(() -> new JwtException("Could not find JWK for kid: " + kid))).orElseGet(() -> {
            if ("none".equals(alg)) {
                return Jwk.NONE_JWK;
            }
            throw new JwtException("JWT defined with signature algorithm " + alg + ", yet no key id (kid): " + jwt);
        });
        return SignedJwt.sign(jwt, jwk);
    }

    public static SignedJwt parseToken(String tokenContent) {
        Errors.Collector collector = Errors.collector();
        Matcher matcher = JWT_PATTERN.matcher(tokenContent);
        if (matcher.matches()) {
            String headerBase64 = matcher.group(1);
            String payloadBase64 = matcher.group(2);
            String signatureBase64 = matcher.group(3);
            JwtHeaders headers = JwtHeaders.parseBase64(headerBase64, collector);
            return SignedJwt.parse(tokenContent, collector, headers, headerBase64, payloadBase64, signatureBase64);
        }
        throw new JwtException("Not a JWT token: " + tokenContent);
    }

    public static SignedJwt parseToken(JwtHeaders headers, String tokenContent) {
        Matcher matcher = JWT_PATTERN.matcher(tokenContent);
        if (matcher.matches()) {
            String headerBase64 = matcher.group(1);
            String payloadBase64 = matcher.group(2);
            String signatureBase64 = matcher.group(3);
            return SignedJwt.parse(tokenContent, Errors.collector(), headers, headerBase64, payloadBase64, signatureBase64);
        }
        throw new JwtException("Not a JWT token: " + tokenContent);
    }

    private static SignedJwt parse(String tokenContent, Errors.Collector collector, JwtHeaders headers, String headerBase64, String payloadBase64, String signatureBase64) {
        String payloadJsonString = SignedJwt.decode(payloadBase64, collector, "JWT payload");
        byte[] signatureBytes = SignedJwt.decodeBytes(signatureBase64, collector, "JWT signature");
        collector.collect().checkValid();
        String signedContent = headerBase64 + "." + payloadBase64;
        JsonObject contentJson = SignedJwt.parseJson(payloadJsonString, collector, payloadBase64, "JWT payload");
        collector.collect().checkValid();
        return new SignedJwt(tokenContent, headers, contentJson, signedContent.getBytes(StandardCharsets.UTF_8), signatureBytes);
    }

    private static JsonObject parseJson(String jsonString, Errors.Collector collector, String base64, String description) {
        try {
            return JSON.createReader((Reader)new StringReader(jsonString)).readObject();
        }
        catch (Exception e) {
            collector.fatal((Object)base64, description + " is not a valid JSON object (value is base64 encoded)");
            return null;
        }
    }

    private static String encode(String string) {
        return SignedJwt.encode(string.getBytes(StandardCharsets.UTF_8));
    }

    private static String encode(byte[] bytes) {
        return URL_ENCODER.encodeToString(bytes);
    }

    private static String decode(String base64, Errors.Collector collector, String description) {
        try {
            return new String(URL_DECODER.decode(base64), StandardCharsets.UTF_8);
        }
        catch (Exception e) {
            collector.fatal((Object)base64, description + " is not a base64 encoded string.");
            return null;
        }
    }

    private static byte[] decodeBytes(String base64, Errors.Collector collector, String description) {
        try {
            return URL_DECODER.decode(base64);
        }
        catch (Exception e) {
            collector.fatal((Object)base64, description + " is not a base64 encoded string.");
            return null;
        }
    }

    public String tokenContent() {
        return this.tokenContent;
    }

    JsonObject headerJson() {
        return this.headers.headerJson();
    }

    JsonObject payloadJson() {
        return this.payloadJson;
    }

    public byte[] getSignedBytes() {
        return Arrays.copyOf(this.signedBytes, this.signedBytes.length);
    }

    public byte[] getSignature() {
        return Arrays.copyOf(this.signature, this.signature.length);
    }

    public Jwt getJwt() {
        return new Jwt(this.headers, this.payloadJson);
    }

    public Errors verifySignature(JwkKeys keys) {
        return this.verifySignature(keys, null);
    }

    public Errors verifySignature(JwkKeys keys, Jwk defaultJwk) {
        Errors.Collector collector = Errors.collector();
        String alg = this.headers.algorithm().orElse(null);
        String kid = this.headers.keyId().orElse(null);
        Jwk jwk = null;
        boolean jwtWithoutKidAndNoneAlg = false;
        if (null == alg) {
            if (null == kid) {
                if (defaultJwk == null) {
                    jwtWithoutKidAndNoneAlg = true;
                    jwk = Jwk.NONE_JWK;
                } else {
                    jwk = defaultJwk;
                }
                alg = jwk.algorithm();
            } else {
                jwk = keys.forKeyId(kid).orElse(null);
                if (null == jwk) {
                    if (null == defaultJwk) {
                        collector.fatal((Object)keys, "Key for key id: " + kid + " not found");
                    } else {
                        jwk = defaultJwk;
                    }
                }
                if (null != jwk) {
                    alg = jwk.algorithm();
                }
            }
        } else if (null == kid) {
            if ("none".equals(alg)) {
                if (null != defaultJwk) {
                    if (!defaultJwk.algorithm().equals(alg)) {
                        collector.fatal("Algorithm is " + alg + ", default jwk requires " + defaultJwk.algorithm());
                    }
                } else {
                    jwk = Jwk.NONE_JWK;
                    jwtWithoutKidAndNoneAlg = true;
                }
            } else {
                jwk = defaultJwk;
                if (null == jwk) {
                    collector.fatal("Algorithm is " + alg + ", yet no kid is defined in JWT header, cannot validate");
                }
            }
        } else {
            jwk = keys.forKeyId(kid).orElse(null);
            if (null == jwk) {
                if (null != defaultJwk && alg.equals(defaultJwk.algorithm())) {
                    jwk = defaultJwk;
                }
                if (null == jwk) {
                    collector.fatal((Object)keys, "Key for key id: " + kid + " not found");
                }
            }
        }
        if (null == jwk) {
            return collector.collect();
        }
        if (jwtWithoutKidAndNoneAlg) {
            collector.fatal((Object)jwk, "None algorithm not allowed, unless specified as the default JWK");
        }
        if (jwk.algorithm().equals(alg)) {
            try {
                if (!jwk.verifySignature(this.signedBytes, this.signature)) {
                    collector.fatal((Object)jwk, "Signature of JWT token is not valid, based on alg: " + alg + ", kid: " + kid);
                }
            }
            catch (Exception e) {
                collector.fatal((Object)jwk, "Failed to verify signature due to an exception: " + e.getClass().getName() + ": " + e.getMessage());
            }
        } else {
            collector.fatal((Object)jwk, "Algorithm of JWK (" + jwk.algorithm() + ") does not match algorithm of this JWT (" + alg + ") for kid: " + kid);
        }
        return collector.collect();
    }
}

