/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.security.providers.httpauth;

import io.helidon.config.Config;
import io.helidon.security.AuthenticationResponse;
import io.helidon.security.Grant;
import io.helidon.security.Principal;
import io.helidon.security.ProviderRequest;
import io.helidon.security.Role;
import io.helidon.security.SecurityEnvironment;
import io.helidon.security.SecurityResponse;
import io.helidon.security.Subject;
import io.helidon.security.SubjectType;
import io.helidon.security.providers.httpauth.ConfigUserStore;
import io.helidon.security.providers.httpauth.DigestToken;
import io.helidon.security.providers.httpauth.HttpAuthException;
import io.helidon.security.providers.httpauth.HttpAuthUtil;
import io.helidon.security.providers.httpauth.HttpDigest;
import io.helidon.security.providers.httpauth.SecureUserStore;
import io.helidon.security.spi.AuthenticationProvider;
import io.helidon.security.spi.SynchronousProvider;
import java.math.BigInteger;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Base64;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.crypto.Cipher;

public final class HttpDigestAuthProvider
extends SynchronousProvider
implements AuthenticationProvider {
    static final String HEADER_AUTHENTICATION_REQUIRED = "WWW-Authenticate";
    static final String HEADER_AUTHENTICATION = "authorization";
    static final String DIGEST_PREFIX = "digest ";
    private static final int UNAUTHORIZED_STATUS_CODE = 401;
    private static final int SALT_LENGTH = 16;
    private static final int AES_NONCE_LENGTH = 12;
    private static final Logger LOGGER = Logger.getLogger(HttpDigestAuthProvider.class.getName());
    private final List<HttpDigest.Qop> digestQopOptions = new LinkedList<HttpDigest.Qop>();
    private final SecureUserStore userStore;
    private final String realm;
    private final SubjectType subjectType;
    private final HttpDigest.Algorithm digestAlgorithm;
    private final SecureRandom random;
    private final long digestNonceTimeoutMillis;
    private final char[] digestServerSecret;

    private HttpDigestAuthProvider(Builder builder) {
        this.userStore = builder.userStore;
        this.realm = builder.realm;
        this.subjectType = builder.subjectType;
        this.digestAlgorithm = builder.digestAlgorithm;
        this.digestQopOptions.addAll(builder.digestQopOptions);
        this.digestNonceTimeoutMillis = builder.digestNonceTimeoutMillis;
        this.digestServerSecret = builder.digestServerSecret;
        this.random = new SecureRandom();
    }

    public static Builder builder() {
        return new Builder();
    }

    public static HttpDigestAuthProvider create(Config config) {
        return HttpDigestAuthProvider.builder().config(config).build();
    }

    static String nonce(long timeInMillis, Random random, char[] serverSecret) {
        byte[] salt = new byte[16];
        random.nextBytes(salt);
        byte[] aesNonce = new byte[12];
        random.nextBytes(aesNonce);
        byte[] timestamp = HttpAuthUtil.toBytes(timeInMillis);
        Cipher cipher = HttpAuthUtil.cipher(serverSecret, salt, aesNonce, 1);
        try {
            timestamp = cipher.doFinal(timestamp);
            byte[] result = new byte[salt.length + aesNonce.length + timestamp.length];
            System.arraycopy(salt, 0, result, 0, salt.length);
            System.arraycopy(aesNonce, 0, result, salt.length, aesNonce.length);
            System.arraycopy(timestamp, 0, result, aesNonce.length + salt.length, timestamp.length);
            return Base64.getEncoder().encodeToString(result);
        }
        catch (Exception e) {
            LOGGER.log(Level.SEVERE, "Encryption failed, though this should not happen. This is a bug.", e);
            return "failed_nonce_value";
        }
    }

    protected AuthenticationResponse syncAuthenticate(ProviderRequest providerRequest) {
        Map headers = providerRequest.env().headers();
        List authorizationHeader = (List)headers.get(HEADER_AUTHENTICATION);
        if (null == authorizationHeader) {
            return this.fail("No authorization header");
        }
        return authorizationHeader.stream().filter(header -> header.toLowerCase().startsWith(DIGEST_PREFIX)).findFirst().map(value -> this.validateDigestAuth((String)value, providerRequest.env())).orElseGet(() -> this.fail("Authorization header does not contain digest authentication: " + authorizationHeader));
    }

    private AuthenticationResponse validateDigestAuth(String headerValue, SecurityEnvironment env) {
        byte[] bytes;
        DigestToken token;
        try {
            token = DigestToken.fromAuthorizationHeader(headerValue.substring(DIGEST_PREFIX.length()), env.method().toLowerCase());
        }
        catch (HttpAuthException e) {
            LOGGER.log(Level.FINEST, "Failed to process digest token", e);
            return this.fail(e.getMessage());
        }
        try {
            bytes = Base64.getDecoder().decode(token.getNonce());
        }
        catch (IllegalArgumentException e) {
            LOGGER.log(Level.FINEST, "Failed to base64 decode nonce", e);
            return this.fail("Nonce must be base64 encoded");
        }
        if (bytes.length < 17) {
            return this.fail("Invalid nonce length");
        }
        byte[] salt = new byte[16];
        byte[] aesNonce = new byte[12];
        byte[] encryptedBytes = new byte[bytes.length - 16 - 12];
        System.arraycopy(bytes, 0, salt, 0, salt.length);
        System.arraycopy(bytes, 16, aesNonce, 0, aesNonce.length);
        System.arraycopy(bytes, 28, encryptedBytes, 0, encryptedBytes.length);
        Cipher cipher = HttpAuthUtil.cipher(this.digestServerSecret, salt, aesNonce, 2);
        try {
            byte[] timestampBytes = cipher.doFinal(encryptedBytes);
            long nonceTimestamp = HttpAuthUtil.toLong(timestampBytes, 0, timestampBytes.length);
            if (System.currentTimeMillis() - nonceTimestamp > this.digestNonceTimeoutMillis) {
                return this.fail("Nonce timeout");
            }
        }
        catch (Exception e) {
            LOGGER.log(Level.FINEST, "Failed to validate nonce", e);
            return this.fail("Invalid nonce value");
        }
        if (!this.realm.equals(token.getRealm())) {
            return this.fail("Invalid realm");
        }
        return this.userStore.user(token.getUsername()).map(user -> {
            if (token.validateLogin((SecureUserStore.User)user)) {
                if (this.subjectType == SubjectType.USER) {
                    return AuthenticationResponse.success((Subject)this.buildSubject((SecureUserStore.User)user));
                }
                return AuthenticationResponse.successService((Subject)this.buildSubject((SecureUserStore.User)user));
            }
            return this.fail("Invalid username or password");
        }).orElse(this.fail("Invalid username or password"));
    }

    private AuthenticationResponse fail(String message) {
        return ((AuthenticationResponse.Builder)((AuthenticationResponse.Builder)((AuthenticationResponse.Builder)((AuthenticationResponse.Builder)AuthenticationResponse.builder().statusCode(401)).responseHeader(HEADER_AUTHENTICATION_REQUIRED, this.buildChallenge())).status(SecurityResponse.SecurityStatus.FAILURE)).description(message)).build();
    }

    private String buildChallenge() {
        StringBuilder challenge = new StringBuilder();
        challenge.append("Digest realm=\"").append(this.realm).append("\"");
        if (!this.digestQopOptions.isEmpty()) {
            challenge.append(", qop=\"").append(this.join(this.digestQopOptions)).append("\"");
        }
        challenge.append(", algorithm=\"").append(this.digestAlgorithm.getAlgorithm()).append("\"");
        challenge.append(", nonce=\"").append(HttpDigestAuthProvider.nonce(System.currentTimeMillis(), this.random, this.digestServerSecret)).append("\"");
        challenge.append(", opaque=\"").append(this.opaque()).append("\"");
        return challenge.toString();
    }

    private String opaque() {
        byte[] bytes = new byte[32];
        this.random.nextBytes(bytes);
        return Base64.getEncoder().encodeToString(bytes);
    }

    private String join(List<HttpDigest.Qop> digestQopOptions) {
        return digestQopOptions.stream().map(HttpDigest.Qop::getQop).collect(Collectors.joining(","));
    }

    private Subject buildSubject(SecureUserStore.User user) {
        Subject.Builder builder = Subject.builder().principal(Principal.builder().name(user.login()).build()).addPrivateCredential(SecureUserStore.User.class, (Object)user);
        user.roles().forEach(role -> builder.addGrant((Grant)Role.create((String)role)));
        return builder.build();
    }

    public static final class Builder
    implements io.helidon.common.Builder<HttpDigestAuthProvider> {
        private static final SecureUserStore EMPTY_STORE = login -> Optional.empty();
        public static final long DEFAULT_DIGEST_NONCE_TIMEOUT = 86400000L;
        private final List<HttpDigest.Qop> digestQopOptions = new LinkedList<HttpDigest.Qop>();
        private SecureUserStore userStore = EMPTY_STORE;
        private String realm = "Helidon";
        private SubjectType subjectType = SubjectType.USER;
        private HttpDigest.Algorithm digestAlgorithm = HttpDigest.Algorithm.MD5;
        private boolean noDigestQop = false;
        private long digestNonceTimeoutMillis = 86400000L;
        private char[] digestServerSecret = Builder.randomSecret();

        private Builder() {
        }

        public Builder config(Config config) {
            config.get("realm").asString().ifPresent(this::realm);
            config.get("users").as(ConfigUserStore::create).ifPresent(this::userStore);
            config.get("algorithm").asString().as(HttpDigest.Algorithm::valueOf).ifPresent(this::digestAlgorithm);
            config.get("nonce-timeout-millis").asLong().ifPresent(timeout -> this.digestNonceTimeout((long)timeout, TimeUnit.MILLISECONDS));
            config.get("principal-type").asString().as(SubjectType::valueOf).ifPresent(this::subjectType);
            config.get("server-secret").asString().map(String::toCharArray).ifPresent(this::digestServerSecret);
            config.get("qop").asList(HttpDigest.Qop::create).ifPresent(qop -> {
                if (qop.isEmpty()) {
                    this.noDigestQop();
                } else {
                    qop.forEach(this::addDigestQop);
                }
            });
            return this;
        }

        private static char[] randomSecret() {
            Random random = new Random();
            String pwd = new BigInteger(130, random).toString(32);
            return pwd.toCharArray();
        }

        public HttpDigestAuthProvider build() {
            if (this.digestQopOptions.isEmpty() && !this.noDigestQop) {
                this.digestQopOptions.add(HttpDigest.Qop.AUTH);
            }
            Objects.requireNonNull(this.userStore, "User store must be configured");
            return new HttpDigestAuthProvider(this);
        }

        public Builder subjectType(SubjectType subjectType) {
            this.subjectType = subjectType;
            switch (subjectType) {
                case USER: 
                case SERVICE: {
                    break;
                }
                default: {
                    throw new SecurityException("Invalid configuration. Principal type not supported: " + subjectType);
                }
            }
            return this;
        }

        public Builder userStore(SecureUserStore store) {
            this.userStore = store;
            return this;
        }

        public Builder realm(String realm) {
            this.realm = realm;
            return this;
        }

        public Builder digestAlgorithm(HttpDigest.Algorithm algorithm) {
            this.digestAlgorithm = algorithm;
            return this;
        }

        public Builder digestNonceTimeout(long duration, TimeUnit unit) {
            this.digestNonceTimeoutMillis = unit.toMillis(duration);
            return this;
        }

        public Builder digestServerSecret(char[] serverSecret) {
            this.digestServerSecret = Arrays.copyOf(serverSecret, serverSecret.length);
            return this;
        }

        public Builder addDigestQop(HttpDigest.Qop qop) {
            this.digestQopOptions.add(qop);
            return this;
        }

        public Builder noDigestQop() {
            this.noDigestQop = true;
            this.digestQopOptions.clear();
            return this;
        }
    }
}

