/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.security.providers.oidc;

import io.helidon.common.http.FormParams;
import io.helidon.common.http.Http;
import io.helidon.common.http.MediaType;
import io.helidon.config.Config;
import io.helidon.security.providers.oidc.common.OidcConfig;
import io.helidon.security.providers.oidc.common.OidcCookieHandler;
import io.helidon.webclient.WebClient;
import io.helidon.webclient.WebClientRequestBuilder;
import io.helidon.webserver.Handler;
import io.helidon.webserver.ResponseHeaders;
import io.helidon.webserver.Routing;
import io.helidon.webserver.ServerRequest;
import io.helidon.webserver.ServerResponse;
import io.helidon.webserver.Service;
import io.helidon.webserver.cors.CorsSupport;
import io.helidon.webserver.cors.CrossOriginConfig;
import jakarta.json.JsonObject;
import java.net.URI;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class OidcSupport
implements Service {
    private static final Logger LOGGER = Logger.getLogger(OidcSupport.class.getName());
    private static final String CODE_PARAM_NAME = "code";
    private static final String STATE_PARAM_NAME = "state";
    private static final String DEFAULT_REDIRECT = "/index.html";
    private final OidcConfig oidcConfig;
    private final OidcCookieHandler tokenCookieHandler;
    private final OidcCookieHandler idTokenCookieHandler;
    private final boolean enabled;
    private final CorsSupport corsSupport;

    private OidcSupport(Builder builder) {
        this.oidcConfig = builder.oidcConfig;
        this.enabled = builder.enabled;
        this.tokenCookieHandler = this.oidcConfig.tokenCookieHandler();
        this.idTokenCookieHandler = this.oidcConfig.idTokenCookieHandler();
        this.corsSupport = this.prepareCrossOriginSupport(this.oidcConfig.redirectUri(), this.oidcConfig.crossOriginConfig());
    }

    public static OidcSupport create(Config config, String providerName) {
        return OidcSupport.builder().config(config, providerName).build();
    }

    public static OidcSupport create(Config config) {
        return OidcSupport.builder().config(config, "oidc").build();
    }

    public static OidcSupport create(OidcConfig oidcConfig) {
        return OidcSupport.builder().config(oidcConfig).build();
    }

    public static Builder builder() {
        return new Builder();
    }

    public void update(Routing.Rules rules) {
        if (this.enabled) {
            if (this.corsSupport != null) {
                rules.any(this.oidcConfig.redirectUri(), new Handler[]{this.corsSupport});
            }
            rules.get(this.oidcConfig.redirectUri(), new Handler[]{this::processOidcRedirect});
            if (this.oidcConfig.logoutEnabled()) {
                if (this.corsSupport != null) {
                    rules.any(this.oidcConfig.logoutUri(), new Handler[]{this.corsSupport});
                }
                rules.get(this.oidcConfig.logoutUri(), new Handler[]{this::processLogout});
            }
            rules.any(new Handler[]{this::addRequestAsHeader});
        }
    }

    private void processLogout(ServerRequest req, ServerResponse res) {
        Optional idTokenCookie = req.headers().cookies().first(this.idTokenCookieHandler.cookieName());
        if (idTokenCookie.isEmpty()) {
            LOGGER.finest("Logout request invoked without ID Token cookie");
            res.status((Http.ResponseStatus)Http.Status.FORBIDDEN_403).send();
            return;
        }
        String encryptedIdToken = (String)idTokenCookie.get();
        this.idTokenCookieHandler.decrypt(encryptedIdToken).forSingle(idToken -> {
            StringBuilder sb = new StringBuilder(this.oidcConfig.logoutEndpointUri() + "?id_token_hint=" + idToken + "&post_logout_redirect_uri=" + this.postLogoutUri(req));
            req.queryParams().first(STATE_PARAM_NAME).ifPresent(it -> sb.append("&state=").append((String)it));
            ResponseHeaders headers = res.headers();
            headers.addCookie(this.tokenCookieHandler.removeCookie().build());
            headers.addCookie(this.idTokenCookieHandler.removeCookie().build());
            res.status((Http.ResponseStatus)Http.Status.TEMPORARY_REDIRECT_307).addHeader("Location", new String[]{sb.toString()}).send();
        }).exceptionallyAccept(t -> this.sendError(res, (Throwable)t));
    }

    private void addRequestAsHeader(ServerRequest req, ServerResponse res) {
        Map newHeaders = req.context().get((Object)"security.addHeaders", Map.class).map(theMap -> theMap).orElseGet(() -> {
            HashMap newMap = new HashMap();
            req.context().register((Object)"security.addHeaders", newMap);
            return newMap;
        });
        String query = req.query();
        if (null == query || query.isEmpty()) {
            newHeaders.put("X_ORIG_URI_HEADER", List.of(req.uri().getPath()));
        } else {
            newHeaders.put("X_ORIG_URI_HEADER", List.of(req.uri().getPath() + "?" + query));
        }
        req.next();
    }

    private void processOidcRedirect(ServerRequest req, ServerResponse res) {
        Optional codeParam = req.queryParams().first(CODE_PARAM_NAME);
        codeParam.ifPresentOrElse(code -> this.processCode((String)code, req, res), () -> this.processError(req, res));
    }

    private void processCode(String code, ServerRequest req, ServerResponse res) {
        WebClient webClient = this.oidcConfig.appWebClient();
        FormParams.Builder form = FormParams.builder().add("grant_type", new String[]{"authorization_code"}).add(CODE_PARAM_NAME, new String[]{code}).add("redirect_uri", new String[]{this.redirectUri(req)});
        WebClientRequestBuilder post = webClient.post().uri(this.oidcConfig.tokenEndpointUri()).accept(new MediaType[]{MediaType.APPLICATION_JSON});
        this.oidcConfig.updateRequest(OidcConfig.RequestType.CODE_TO_TOKEN, post, form);
        OidcConfig.postJsonResponse((WebClientRequestBuilder)post, (Object)form.build(), json -> this.processJsonResponse(req, res, (JsonObject)json), (status, errorEntity) -> this.processError(res, (Http.ResponseStatus)status, (String)errorEntity), (t, message) -> this.processError(res, (Throwable)t, (String)message)).ignoreElement();
    }

    private Object postLogoutUri(ServerRequest req) {
        URI uri = this.oidcConfig.postLogoutUri();
        if (uri.getHost() != null) {
            return uri.toString();
        }
        Object path = uri.getPath();
        path = ((String)path).startsWith("/") ? path : "/" + (String)path;
        Optional host = req.headers().first("host");
        if (host.isPresent()) {
            String scheme = this.oidcConfig.forceHttpsRedirects() || req.isSecure() ? "https" : "http";
            return scheme + "://" + (String)host.get() + (String)path;
        }
        LOGGER.warning("Request without Host header received, yet post logout URI does not define a host");
        return this.oidcConfig.toString();
    }

    private String redirectUri(ServerRequest req) {
        Optional host = req.headers().first("host");
        if (host.isPresent()) {
            String scheme = req.isSecure() ? "https" : "http";
            return this.oidcConfig.redirectUriWithHost(scheme + "://" + (String)host.get());
        }
        return this.oidcConfig.redirectUriWithHost();
    }

    private String processJsonResponse(ServerRequest req, ServerResponse res, JsonObject json) {
        String tokenValue = json.getString("access_token");
        String idToken = json.getString("id_token", null);
        Object state = req.queryParams().first(STATE_PARAM_NAME).orElse(DEFAULT_REDIRECT);
        res.status((Http.ResponseStatus)Http.Status.TEMPORARY_REDIRECT_307);
        if (this.oidcConfig.useParam()) {
            state = (((String)state).contains("?") ? "&" : "?") + this.oidcConfig.paramName() + "=" + tokenValue;
        }
        state = this.increaseRedirectCounter((String)state);
        res.headers().add("Location", new String[]{state});
        if (this.oidcConfig.useCookie()) {
            ResponseHeaders headers = res.headers();
            this.tokenCookieHandler.createCookie(tokenValue).forSingle(builder -> {
                headers.addCookie(builder.build());
                if (idToken != null && this.oidcConfig.logoutEnabled()) {
                    this.idTokenCookieHandler.createCookie(idToken).forSingle(it -> {
                        headers.addCookie(it.build());
                        res.send();
                    }).exceptionallyAccept(t -> this.sendError(res, (Throwable)t));
                } else {
                    res.send();
                }
            }).exceptionallyAccept(t -> this.sendError(res, (Throwable)t));
        } else {
            res.send();
        }
        return "done";
    }

    private void sendError(ServerResponse response, Throwable t) {
        if (LOGGER.isLoggable(Level.FINEST)) {
            LOGGER.log(Level.FINEST, "Failed to process OIDC request", t);
        }
        response.status((Http.ResponseStatus)Http.Status.INTERNAL_SERVER_ERROR_500).send();
    }

    private Optional<String> processError(ServerResponse serverResponse, Http.ResponseStatus status, String entity) {
        LOGGER.log(Level.FINE, "Invalid token or failed request when connecting to OIDC Token Endpoint. Response: " + entity + ", response status: " + status);
        this.sendErrorResponse(serverResponse);
        return Optional.empty();
    }

    private Optional<String> processError(ServerResponse res, Throwable t, String message) {
        LOGGER.log(Level.FINE, message, t);
        this.sendErrorResponse(res);
        return Optional.empty();
    }

    private void sendErrorResponse(ServerResponse serverResponse) {
        serverResponse.status((Http.ResponseStatus)Http.Status.UNAUTHORIZED_401);
        serverResponse.send((Object)"Not a valid authorization code");
    }

    String increaseRedirectCounter(String state) {
        if (state.contains("?")) {
            Pattern attemptPattern = Pattern.compile(".*?(" + this.oidcConfig.redirectAttemptParam() + "=\\d+).*");
            Matcher matcher = attemptPattern.matcher(state);
            if (matcher.matches()) {
                String attempts = matcher.group(1);
                int equals = attempts.lastIndexOf(61);
                String count = attempts.substring(equals + 1);
                int countNumber = Integer.parseInt(count);
                return state.replace(attempts, this.oidcConfig.redirectAttemptParam() + "=" + ++countNumber);
            }
            return state + "&" + this.oidcConfig.redirectAttemptParam() + "=1";
        }
        return state + "?" + this.oidcConfig.redirectAttemptParam() + "=1";
    }

    private void processError(ServerRequest req, ServerResponse res) {
        String error = req.queryParams().first("error").orElse("invalid_request");
        String errorDescription = req.queryParams().first("error_description").orElseGet(() -> "Failed to process authorization request. Expected redirect from OIDC server with code parameter, but got: " + req.query());
        LOGGER.log(Level.WARNING, () -> "Received request on OIDC endpoint with no code. Error: " + error + " Error description: " + errorDescription);
        res.status((Http.ResponseStatus)Http.Status.BAD_REQUEST_400);
        res.send((Object)("{\"error\": \"" + error + "\", \"error_description\": \"" + errorDescription + "\"}"));
    }

    private CorsSupport prepareCrossOriginSupport(String path, CrossOriginConfig crossOriginConfig) {
        return crossOriginConfig == null ? null : ((CorsSupport.Builder)CorsSupport.builder().addCrossOrigin(path, crossOriginConfig)).build();
    }

    public static class Builder
    implements io.helidon.common.Builder<Builder, OidcSupport> {
        private boolean enabled = true;
        private OidcConfig oidcConfig;

        private Builder() {
        }

        private static Config findMyKey(Config rootConfig, String providerName) {
            if (rootConfig.key().name().equals(providerName)) {
                return rootConfig;
            }
            return ((List)rootConfig.get("security.providers").asNodeList().get()).stream().filter(it -> it.get(providerName).exists()).findFirst().map(it -> it.get(providerName)).orElseThrow(() -> new SecurityException("No configuration found for provider named: " + providerName));
        }

        public OidcSupport build() {
            if (this.enabled && this.oidcConfig == null) {
                throw new IllegalStateException("When OIDC and security is enabled, OIDC configuration must be provided");
            }
            return new OidcSupport(this);
        }

        public Builder config(Config config) {
            config.get("enabled").asBoolean().ifPresent(this::enabled);
            if (this.enabled) {
                this.oidcConfig = OidcConfig.create((Config)config);
            }
            return this;
        }

        public Builder config(OidcConfig config) {
            this.oidcConfig = config;
            return this;
        }

        public Builder config(Config config, String providerName) {
            config.get("security.enabled").asBoolean().ifPresent(this::enabled);
            this.config(Builder.findMyKey(config, providerName));
            return this;
        }

        public Builder enabled(boolean enabled) {
            this.enabled = enabled;
            return this;
        }
    }
}

