/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.security.providers.oidc;

import io.helidon.common.Errors;
import io.helidon.common.http.FormParams;
import io.helidon.common.http.Http;
import io.helidon.common.http.MediaType;
import io.helidon.common.reactive.Single;
import io.helidon.security.AuthenticationResponse;
import io.helidon.security.EndpointConfig;
import io.helidon.security.Grant;
import io.helidon.security.Principal;
import io.helidon.security.ProviderRequest;
import io.helidon.security.Role;
import io.helidon.security.SecurityEnvironment;
import io.helidon.security.SecurityLevel;
import io.helidon.security.SecurityResponse;
import io.helidon.security.Subject;
import io.helidon.security.abac.scope.ScopeValidator;
import io.helidon.security.jwt.Jwt;
import io.helidon.security.jwt.JwtException;
import io.helidon.security.jwt.JwtUtil;
import io.helidon.security.jwt.SignedJwt;
import io.helidon.security.jwt.jwk.JwkKeys;
import io.helidon.security.providers.common.TokenCredential;
import io.helidon.security.providers.oidc.OidcUtil;
import io.helidon.security.providers.oidc.common.OidcConfig;
import io.helidon.security.providers.oidc.common.Tenant;
import io.helidon.security.providers.oidc.common.TenantConfig;
import io.helidon.security.util.TokenHandler;
import io.helidon.webclient.WebClientRequestBuilder;
import jakarta.json.JsonValue;
import java.net.URI;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

class TenantAuthenticationHandler {
    private static final Logger LOGGER = Logger.getLogger(TenantAuthenticationHandler.class.getName());
    private static final TokenHandler PARAM_HEADER_HANDLER = TokenHandler.forHeader((String)"X_OIDC_TOKEN_HEADER");
    private final boolean optional;
    private final OidcConfig oidcConfig;
    private final TenantConfig tenantConfig;
    private final Tenant tenant;
    private final boolean useJwtGroups;
    private final BiFunction<SignedJwt, Errors.Collector, Single<Errors.Collector>> jwtValidator;
    private final BiConsumer<StringBuilder, String> scopeAppender;
    private final Pattern attemptPattern;

    TenantAuthenticationHandler(OidcConfig oidcConfig, Tenant tenant, boolean useJwtGroups, boolean optional) {
        this.oidcConfig = oidcConfig;
        this.tenant = tenant;
        this.tenantConfig = tenant.tenantConfig();
        this.useJwtGroups = useJwtGroups;
        this.optional = optional;
        this.attemptPattern = Pattern.compile(".*?" + oidcConfig.redirectAttemptParam() + "=(\\d+).*");
        this.jwtValidator = this.tenantConfig.validateJwtWithJwk() ? (signedJwt, collector) -> {
            JwkKeys jwk = tenant.signJwk();
            Errors errors = signedJwt.verifySignature(jwk);
            errors.forEach(errorMessage -> {
                switch (errorMessage.getSeverity()) {
                    case FATAL: {
                        collector.fatal(errorMessage.getSource(), errorMessage.getMessage());
                        break;
                    }
                    case WARN: {
                        collector.warn(errorMessage.getSource(), errorMessage.getMessage());
                        break;
                    }
                    default: {
                        collector.hint(errorMessage.getSource(), errorMessage.getMessage());
                    }
                }
            });
            return Single.just((Object)collector);
        } : (signedJwt, collector) -> {
            FormParams.Builder form = FormParams.builder().add("token", new String[]{signedJwt.tokenContent()});
            WebClientRequestBuilder post = tenant.appWebClient().post().uri(tenant.introspectUri()).accept(new MediaType[]{MediaType.APPLICATION_JSON}).headers(it -> {
                it.add("Cache-Control", new String[]{"no-cache, no-store, must-revalidate"});
                return it;
            });
            OidcUtil.updateRequest(OidcConfig.RequestType.INTROSPECT_JWT, this.tenantConfig, form);
            return OidcConfig.postJsonResponse((WebClientRequestBuilder)post, (Object)form.build(), json -> {
                if (!json.getBoolean("active")) {
                    collector.fatal(json, "Token is not active");
                }
                return collector;
            }, (status, message) -> Optional.of(collector.fatal(status, "Failed to validate token, response status: " + status + ", entity: " + message)), (t, message) -> Optional.of(collector.fatal(t, "Failed to validate token, request failed: " + message)));
        };
        String configuredScopeAudience = this.tenantConfig.scopeAudience();
        this.scopeAppender = configuredScopeAudience == null || configuredScopeAudience.isEmpty() ? StringBuilder::append : (configuredScopeAudience.endsWith("/") ? (stringBuilder, scope) -> stringBuilder.append(configuredScopeAudience).append((String)scope) : (stringBuilder, scope) -> stringBuilder.append(configuredScopeAudience).append("/").append((String)scope));
    }

    CompletionStage<AuthenticationResponse> authenticate(String tenantId, ProviderRequest providerRequest) {
        Optional token;
        LinkedList<String> missingLocations;
        block8: {
            missingLocations = new LinkedList<String>();
            token = Optional.empty();
            try {
                if (this.oidcConfig.useHeader() && (token = token.or(() -> this.oidcConfig.headerHandler().extractToken(providerRequest.env().headers()))).isEmpty()) {
                    missingLocations.add("header");
                }
                if (this.oidcConfig.useParam()) {
                    if ((token = token.or(() -> PARAM_HEADER_HANDLER.extractToken(providerRequest.env().headers()))).isEmpty()) {
                        token = token.or(() -> providerRequest.env().queryParams().first(this.oidcConfig.paramName()));
                    }
                    if (token.isEmpty()) {
                        missingLocations.add("query-param");
                    }
                }
                if (!this.oidcConfig.useCookie() || !token.isEmpty()) break block8;
                Optional cookie = this.oidcConfig.tokenCookieHandler().findCookie(providerRequest.env().headers());
                if (cookie.isEmpty()) {
                    missingLocations.add("cookie");
                    break block8;
                }
                return ((Single)cookie.get()).flatMapSingle(it -> this.validateToken(tenantId, providerRequest, (String)it)).onErrorResumeWithSingle(throwable -> {
                    if (LOGGER.isLoggable(Level.FINEST)) {
                        LOGGER.log(Level.FINEST, "Invalid token in cookie", (Throwable)throwable);
                    }
                    return Single.just((Object)this.errorResponse(providerRequest, Http.Status.UNAUTHORIZED_401, null, "Invalid token", tenantId));
                });
            }
            catch (SecurityException e) {
                LOGGER.log(Level.FINEST, "Failed to extract token from one of the configured locations", e);
                return this.failOrAbstain("Failed to extract one of the configured tokens" + e);
            }
        }
        if (token.isPresent()) {
            return this.validateToken(tenantId, providerRequest, (String)token.get());
        }
        LOGGER.finest(() -> "Missing token, could not find in either of: " + missingLocations);
        return CompletableFuture.completedFuture(this.errorResponse(providerRequest, Http.Status.UNAUTHORIZED_401, null, "Missing token, could not find in either of: " + missingLocations, tenantId));
    }

    private Set<String> expectedScopes(ProviderRequest request) {
        HashSet<String> result = new HashSet<String>();
        for (SecurityLevel securityLevel : request.endpointConfig().securityLevels()) {
            List expectedScopes = securityLevel.combineAnnotations(ScopeValidator.Scopes.class, EndpointConfig.AnnotationScope.values());
            expectedScopes.stream().map(ScopeValidator.Scopes::value).map(Arrays::asList).map(Collection::stream).forEach(stream -> stream.map(ScopeValidator.Scope::value).forEach(result::add));
            List expectedScopeAnnotations = securityLevel.combineAnnotations(ScopeValidator.Scope.class, EndpointConfig.AnnotationScope.values());
            expectedScopeAnnotations.stream().map(ScopeValidator.Scope::value).forEach(result::add);
        }
        return result;
    }

    private AuthenticationResponse errorResponse(ProviderRequest providerRequest, Http.Status status, String code, String description, String tenantId) {
        if (this.oidcConfig.shouldRedirect()) {
            String state = this.origUri(providerRequest);
            int redirectAttempt = this.redirectAttempt(state);
            if (redirectAttempt >= this.oidcConfig.maxRedirects()) {
                return this.errorResponseNoRedirect(code, description, status);
            }
            Set<String> expectedScopes = this.expectedScopes(providerRequest);
            StringBuilder scopes = new StringBuilder(this.tenantConfig.baseScopes());
            for (String expectedScope : expectedScopes) {
                String scope;
                if (scopes.length() > 0) {
                    scopes.append(' ');
                }
                if ((scope = expectedScope).startsWith("/")) {
                    scope = scope.substring(1);
                }
                this.scopeAppender.accept(scopes, scope);
            }
            String scopeString = URLEncoder.encode(scopes.toString(), StandardCharsets.UTF_8);
            String authorizationEndpoint = this.tenant.authorizationEndpointUri();
            String nonce = UUID.randomUUID().toString();
            String redirectUri = "@default".equals(tenantId) ? this.encode(this.redirectUri(providerRequest.env())) : this.encode(this.redirectUri(providerRequest.env()) + "?" + this.encode(this.oidcConfig.tenantParamName()) + "=" + this.encode(tenantId));
            StringBuilder queryString = new StringBuilder("?");
            queryString.append("client_id=").append(this.tenantConfig.clientId()).append("&");
            queryString.append("response_type=code&");
            queryString.append("redirect_uri=").append(redirectUri).append("&");
            queryString.append("scope=").append(scopeString).append("&");
            queryString.append("nonce=").append(nonce).append("&");
            queryString.append("state=").append(this.encode(state));
            return ((AuthenticationResponse.Builder)((AuthenticationResponse.Builder)((AuthenticationResponse.Builder)((AuthenticationResponse.Builder)AuthenticationResponse.builder().status(SecurityResponse.SecurityStatus.FAILURE_FINISH)).statusCode(Http.Status.TEMPORARY_REDIRECT_307.code())).description("Redirecting to identity server: " + description)).responseHeader("Location", authorizationEndpoint + queryString)).build();
        }
        return this.errorResponseNoRedirect(code, description, status);
    }

    private String redirectUri(SecurityEnvironment env) {
        for (Map.Entry entry : env.headers().entrySet()) {
            if (!((String)entry.getKey()).equalsIgnoreCase("host") || ((List)entry.getValue()).isEmpty()) continue;
            String firstHost = (String)((List)entry.getValue()).get(0);
            return this.oidcConfig.redirectUriWithHost((String)(this.oidcConfig.forceHttpsRedirects() ? "https" : env.transport() + "://" + firstHost));
        }
        return this.oidcConfig.redirectUriWithHost();
    }

    private CompletionStage<AuthenticationResponse> failOrAbstain(String message) {
        if (this.optional) {
            return CompletableFuture.completedFuture(((AuthenticationResponse.Builder)((AuthenticationResponse.Builder)AuthenticationResponse.builder().status(SecurityResponse.SecurityStatus.ABSTAIN)).description(message)).build());
        }
        return CompletableFuture.completedFuture(((AuthenticationResponse.Builder)((AuthenticationResponse.Builder)AuthenticationResponse.builder().status(SecurityResponse.SecurityStatus.FAILURE)).description(message)).build());
    }

    private AuthenticationResponse errorResponseNoRedirect(String code, String description, Http.Status status) {
        if (this.optional) {
            return ((AuthenticationResponse.Builder)((AuthenticationResponse.Builder)AuthenticationResponse.builder().status(SecurityResponse.SecurityStatus.ABSTAIN)).description(description)).build();
        }
        if (null == code) {
            return ((AuthenticationResponse.Builder)((AuthenticationResponse.Builder)((AuthenticationResponse.Builder)((AuthenticationResponse.Builder)AuthenticationResponse.builder().status(SecurityResponse.SecurityStatus.FAILURE)).statusCode(Http.Status.UNAUTHORIZED_401.code())).responseHeader("WWW-Authenticate", "Bearer realm=\"" + this.tenantConfig.realm() + "\"")).description(description)).build();
        }
        return ((AuthenticationResponse.Builder)((AuthenticationResponse.Builder)((AuthenticationResponse.Builder)((AuthenticationResponse.Builder)AuthenticationResponse.builder().status(SecurityResponse.SecurityStatus.FAILURE)).statusCode(status.code())).responseHeader("WWW-Authenticate", this.errorHeader(code, description))).description(description)).build();
    }

    private int redirectAttempt(String state) {
        Matcher matcher;
        if (state.contains("?") && (matcher = this.attemptPattern.matcher(state)).matches()) {
            return Integer.parseInt(matcher.group(1));
        }
        return 1;
    }

    private String errorHeader(String code, String description) {
        return "Bearer realm=\"" + this.tenantConfig.realm() + "\", error=\"" + code + "\", error_description=\"" + description + "\"";
    }

    String origUri(ProviderRequest providerRequest) {
        List origUri = providerRequest.env().headers().getOrDefault("X_ORIG_URI_HEADER", List.of());
        if (origUri.isEmpty()) {
            URI targetUri = providerRequest.env().targetUri();
            String query = targetUri.getQuery();
            String path = targetUri.getPath();
            if (query == null || query.isEmpty()) {
                return path;
            }
            return path + "?" + query;
        }
        return (String)origUri.get(0);
    }

    private String encode(String state) {
        return URLEncoder.encode(state, StandardCharsets.UTF_8);
    }

    private Single<AuthenticationResponse> validateToken(String tenantId, ProviderRequest providerRequest, String token) {
        SignedJwt signedJwt;
        try {
            signedJwt = SignedJwt.parseToken((String)token);
        }
        catch (Exception e) {
            LOGGER.log(Level.FINEST, "Could not parse inbound token", e);
            return Single.just((Object)AuthenticationResponse.failed((String)"Invalid token", (Throwable)e));
        }
        return this.jwtValidator.apply(signedJwt, Errors.collector()).map(it -> this.processValidationResult(providerRequest, signedJwt, tenantId, (Errors.Collector)it)).onErrorResume(t -> {
            LOGGER.log(Level.FINEST, "Failed to validate request", (Throwable)t);
            return AuthenticationResponse.failed((String)"Failed to validate JWT", (Throwable)t);
        });
    }

    private AuthenticationResponse processValidationResult(ProviderRequest providerRequest, SignedJwt signedJwt, String tenantId, Errors.Collector collector) {
        Jwt jwt = signedJwt.getJwt();
        Errors errors = collector.collect();
        Errors validationErrors = jwt.validate(this.tenant.issuer(), this.tenantConfig.audience());
        if (errors.isValid() && validationErrors.isValid()) {
            errors.log(LOGGER);
            Subject subject = this.buildSubject(jwt, signedJwt);
            Set scopes = subject.grantsByType("scope").stream().map(Grant::getName).collect(Collectors.toSet());
            Set<String> expectedScopes = this.expectedScopes(providerRequest);
            LinkedList<String> missingScopes = new LinkedList<String>();
            for (String expectedScope : expectedScopes) {
                if (scopes.contains(expectedScope)) continue;
                missingScopes.add(expectedScope);
            }
            if (missingScopes.isEmpty()) {
                return AuthenticationResponse.success((Subject)subject);
            }
            return this.errorResponse(providerRequest, Http.Status.FORBIDDEN_403, "insufficient_scope", "Scopes " + missingScopes + " are missing", tenantId);
        }
        if (LOGGER.isLoggable(Level.FINEST)) {
            errors.log(LOGGER);
            validationErrors.log(LOGGER);
        }
        return this.errorResponse(providerRequest, Http.Status.UNAUTHORIZED_401, "invalid_token", "Token not valid", tenantId);
    }

    private Subject buildSubject(Jwt jwt, SignedJwt signedJwt) {
        Principal principal = this.buildPrincipal(jwt);
        TokenCredential.Builder builder = TokenCredential.builder();
        jwt.issueTime().ifPresent(arg_0 -> ((TokenCredential.Builder)builder).issueTime(arg_0));
        jwt.expirationTime().ifPresent(arg_0 -> ((TokenCredential.Builder)builder).expTime(arg_0));
        jwt.issuer().ifPresent(arg_0 -> ((TokenCredential.Builder)builder).issuer(arg_0));
        builder.token(signedJwt.tokenContent());
        builder.addToken(Jwt.class, (Object)jwt);
        builder.addToken(SignedJwt.class, (Object)signedJwt);
        Subject.Builder subjectBuilder = Subject.builder().principal(principal).addPublicCredential(TokenCredential.class, (Object)builder.build());
        if (this.useJwtGroups) {
            Optional userGroups = jwt.userGroups();
            userGroups.ifPresent(groups -> groups.forEach(group -> subjectBuilder.addGrant((Grant)Role.create((String)group))));
        }
        Optional scopes = jwt.scopes();
        scopes.ifPresent(scopeList -> scopeList.forEach(scope -> subjectBuilder.addGrant(Grant.builder().name(scope).type("scope").build())));
        return subjectBuilder.build();
    }

    private Principal buildPrincipal(Jwt jwt) {
        String subject = (String)jwt.subject().orElseThrow(() -> new JwtException("JWT does not contain subject claim, cannot create principal."));
        String name = jwt.preferredUsername().orElse(subject);
        Principal.Builder builder = Principal.builder();
        builder.name(name).id(subject);
        jwt.payloadClaims().forEach((key, jsonValue) -> builder.addAttribute(key, JwtUtil.toObject((JsonValue)jsonValue)));
        jwt.email().ifPresent(value -> builder.addAttribute("email", value));
        jwt.emailVerified().ifPresent(value -> builder.addAttribute("email_verified", value));
        jwt.locale().ifPresent(value -> builder.addAttribute("locale", value));
        jwt.familyName().ifPresent(value -> builder.addAttribute("family_name", value));
        jwt.givenName().ifPresent(value -> builder.addAttribute("given_name", value));
        jwt.fullName().ifPresent(value -> builder.addAttribute("full_name", value));
        return builder.build();
    }
}

