/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webserver.http1;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;
import io.helidon.http.RequestedUriDiscoveryContext;

/**
 * HTTP/1.1 server configuration.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.webserver.http1.Http1ConfigBlueprint")
public interface Http1Config extends Http1ConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static Http1Config.Builder builder() {
        return new Http1Config.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static Http1Config.Builder builder(Http1Config instance) {
        return Http1Config.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static Http1Config create(Config config) {
        return Http1Config.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static Http1Config create() {
        return Http1Config.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link Http1Config}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends Http1Config.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends Http1Config> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final List<Http1ConnectionListener> receiveListeners = new ArrayList<>();
        private final List<Http1ConnectionListener> sendListeners = new ArrayList<>();
        private boolean continueImmediately = false;
        private boolean isReceiveListenersMutated;
        private boolean isSendListenersMutated;
        private boolean receiveLog = true;
        private boolean sendLog = true;
        private boolean validatePath = true;
        private boolean validatePrologue = true;
        private boolean validateRequestHeaders = true;
        private boolean validateRequestHostHeader = true;
        private boolean validateResponseHeaders = false;
        private Config config;
        private Http1ConnectionListener compositeReceiveListener;
        private Http1ConnectionListener compositeSendListener;
        private int maxHeadersSize = 16384;
        private int maxPrologueLength = 4096;
        private RequestedUriDiscoveryContext requestedUriDiscovery;
        private String name;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance. This method disables automatic service discovery.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(Http1Config prototype) {
            name(prototype.name());
            maxPrologueLength(prototype.maxPrologueLength());
            maxHeadersSize(prototype.maxHeadersSize());
            validateRequestHeaders(prototype.validateRequestHeaders());
            validateRequestHostHeader(prototype.validateRequestHostHeader());
            validateResponseHeaders(prototype.validateResponseHeaders());
            validatePrologue(prototype.validatePrologue());
            validatePath(prototype.validatePath());
            receiveLog(prototype.receiveLog());
            sendLog(prototype.sendLog());
            continueImmediately(prototype.continueImmediately());
            requestedUriDiscovery(prototype.requestedUriDiscovery());
            if (!isSendListenersMutated) {
                sendListeners.clear();
            }
            addSendListeners(prototype.sendListeners());
            if (!isReceiveListenersMutated) {
                receiveListeners.clear();
            }
            addReceiveListeners(prototype.receiveListeners());
            compositeSendListener(prototype.compositeSendListener());
            compositeReceiveListener(prototype.compositeReceiveListener());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(Http1Config.BuilderBase<?, ?> builder) {
            builder.name().ifPresent(this::name);
            maxPrologueLength(builder.maxPrologueLength());
            maxHeadersSize(builder.maxHeadersSize());
            validateRequestHeaders(builder.validateRequestHeaders());
            validateRequestHostHeader(builder.validateRequestHostHeader());
            validateResponseHeaders(builder.validateResponseHeaders());
            validatePrologue(builder.validatePrologue());
            validatePath(builder.validatePath());
            receiveLog(builder.receiveLog());
            sendLog(builder.sendLog());
            continueImmediately(builder.continueImmediately());
            builder.requestedUriDiscovery().ifPresent(this::requestedUriDiscovery);
            if (isSendListenersMutated) {
                if (builder.isSendListenersMutated) {
                    addSendListeners(builder.sendListeners);
                }
            } else {
                sendListeners.clear();
                addSendListeners(builder.sendListeners);
            }
            if (isReceiveListenersMutated) {
                if (builder.isReceiveListenersMutated) {
                    addReceiveListeners(builder.receiveListeners);
                }
            } else {
                receiveListeners.clear();
                addReceiveListeners(builder.receiveListeners);
            }
            builder.compositeSendListener().ifPresent(this::compositeSendListener);
            builder.compositeReceiveListener().ifPresent(this::compositeReceiveListener);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("max-prologue-length").as(Integer.class).ifPresent(this::maxPrologueLength);
            config.get("max-headers-size").as(Integer.class).ifPresent(this::maxHeadersSize);
            config.get("validate-request-headers").as(Boolean.class).ifPresent(this::validateRequestHeaders);
            config.get("validate-request-host-header").as(Boolean.class).ifPresent(this::validateRequestHostHeader);
            config.get("validate-response-headers").as(Boolean.class).ifPresent(this::validateResponseHeaders);
            config.get("validate-prologue").as(Boolean.class).ifPresent(this::validatePrologue);
            config.get("validate-path").as(Boolean.class).ifPresent(this::validatePath);
            config.get("recv-log").as(Boolean.class).ifPresent(this::receiveLog);
            config.get("send-log").as(Boolean.class).ifPresent(this::sendLog);
            config.get("continue-immediately").as(Boolean.class).ifPresent(this::continueImmediately);
            config.get("requested-uri-discovery").map(RequestedUriDiscoveryContext::create).ifPresent(this::requestedUriDiscovery);
            return self();
        }

        /**
         * Name of this configuration, in most cases the same as {@link #type()}.
         *
         * @param name name of this configuration
         * @return updated builder instance
         * @see #name()
         */
        public BUILDER name(String name) {
            Objects.requireNonNull(name);
            this.name = name;
            return self();
        }

        /**
         * Maximal size of received HTTP prologue (GET /path HTTP/1.1).
         *
         * @param maxPrologueLength maximal size in bytes
         * @return updated builder instance
         * @see #maxPrologueLength()
         */
        public BUILDER maxPrologueLength(int maxPrologueLength) {
            this.maxPrologueLength = maxPrologueLength;
            return self();
        }

        /**
         * Maximal size of received headers in bytes.
         *
         * @param maxHeadersSize maximal header size
         * @return updated builder instance
         * @see #maxHeadersSize()
         */
        public BUILDER maxHeadersSize(int maxHeadersSize) {
            this.maxHeadersSize = maxHeadersSize;
            return self();
        }

        /**
         * Whether to validate headers.
         * If set to false, any value is accepted, otherwise validates headers + known headers
         * are validated by format
         * (content length is always validated as it is part of protocol processing (other headers may be validated if
         * features use them)).
         * <p>
         *     Defaults to {@code true}.
         * </p>
         *
         * @param validateRequestHeaders whether to validate headers
         * @return updated builder instance
         * @see #validateRequestHeaders()
         */
        public BUILDER validateRequestHeaders(boolean validateRequestHeaders) {
            this.validateRequestHeaders = validateRequestHeaders;
            return self();
        }

        /**
         * Request host header validation.
         * When host header is invalid, we return {@link io.helidon.http.Status#BAD_REQUEST_400}.
         * <p>
         * The validation is done according to RFC-3986 (see {@link io.helidon.common.uri.UriValidator}). This is a requirement of
         * the HTTP specification.
         * <p>
         * This option allows you to disable the "full-blown" validation ("simple" validation is still in - the port must be
         * parseable to integer).
         *
         * @param validateRequestHostHeader whether to do a full validation of {@code Host} header according to the specification
         * @return updated builder instance
         * @deprecated this switch exists for temporary backward compatible behavior, and will be removed in a future Helidon
         *             version
         * @see #validateRequestHostHeader()
         */
        @Deprecated(since = "4.1.3", forRemoval = true)
        public BUILDER validateRequestHostHeader(boolean validateRequestHostHeader) {
            this.validateRequestHostHeader = validateRequestHostHeader;
            return self();
        }

        /**
         * Whether to validate headers.
         * If set to false, any value is accepted, otherwise validates headers + known headers
         * are validated by format
         * (content length is always validated as it is part of protocol processing (other headers may be validated if
         * features use them)).
         * <p>
         *     Defaults to {@code false} as user has control on the header creation.
         * </p>
         *
         * @param validateResponseHeaders whether to validate headers
         * @return updated builder instance
         * @see #validateResponseHeaders()
         */
        public BUILDER validateResponseHeaders(boolean validateResponseHeaders) {
            this.validateResponseHeaders = validateResponseHeaders;
            return self();
        }

        /**
         * If set to false, any query and fragment is accepted (even containing illegal characters).
         * Validation of path is controlled by {@link #validatePath()}.
         *
         * @param validatePrologue whether to validate prologue query and fragment
         * @return updated builder instance
         * @see #validatePrologue()
         */
        public BUILDER validatePrologue(boolean validatePrologue) {
            this.validatePrologue = validatePrologue;
            return self();
        }

        /**
         * If set to false, any path is accepted (even containing illegal characters).
         *
         * @param validatePath whether to validate path
         * @return updated builder instance
         * @see #validatePath()
         */
        public BUILDER validatePath(boolean validatePath) {
            this.validatePath = validatePath;
            return self();
        }

        /**
         * Logging of received packets. Uses trace and debug levels on logger of
         * {@link Http1LoggingConnectionListener} with suffix of {@code .recv`}.
         *
         * @param receiveLog {@code true} if logging should be enabled for received packets, {@code false} if no logging should be done
         * @return updated builder instance
         * @see #receiveLog()
         */
        public BUILDER receiveLog(boolean receiveLog) {
            this.receiveLog = receiveLog;
            return self();
        }

        /**
         * Logging of sent packets. Uses trace and debug levels on logger of
         * {@link Http1LoggingConnectionListener} with suffix of {@code .send`}.
         *
         * @param sendLog {@code true} if logging should be enabled for sent packets, {@code false} if no logging should be done
         * @return updated builder instance
         * @see #sendLog()
         */
        public BUILDER sendLog(boolean sendLog) {
            this.sendLog = sendLog;
            return self();
        }

        /**
         * When true WebServer answers to expect continue with 100 continue immediately,
         * not waiting for user to actually request the data.
         *
         * @param continueImmediately if {@code true} answer with 100 continue immediately after expect continue
         * @return updated builder instance
         * @see #continueImmediately()
         */
        public BUILDER continueImmediately(boolean continueImmediately) {
            this.continueImmediately = continueImmediately;
            return self();
        }

        /**
         * Requested URI discovery settings.
         *
         * @param requestedUriDiscovery settings for computing the requested URI
         * @return updated builder instance
         * @see #requestedUriDiscovery()
         */
        public BUILDER requestedUriDiscovery(RequestedUriDiscoveryContext requestedUriDiscovery) {
            Objects.requireNonNull(requestedUriDiscovery);
            this.requestedUriDiscovery = requestedUriDiscovery;
            return self();
        }

        /**
         * Requested URI discovery settings.
         *
         * @param consumer consumer of builder for
         *                 settings for computing the requested URI
         * @return updated builder instance
         * @see #requestedUriDiscovery()
         */
        public BUILDER requestedUriDiscovery(Consumer<RequestedUriDiscoveryContext.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = RequestedUriDiscoveryContext.builder();
            consumer.accept(builder);
            this.requestedUriDiscovery(builder.build());
            return self();
        }

        /**
         * Requested URI discovery settings.
         *
         * @param supplier supplier of
         *                 settings for computing the requested URI
         * @return updated builder instance
         * @see #requestedUriDiscovery()
         */
        public BUILDER requestedUriDiscovery(Supplier<? extends RequestedUriDiscoveryContext> supplier) {
            Objects.requireNonNull(supplier);
            this.requestedUriDiscovery(supplier.get());
            return self();
        }

        /**
         * Connection send event listeners for HTTP/1.1.
         *
         * @param sendListeners send event listeners
         * @return updated builder instance
         * @see #sendListeners()
         */
        public BUILDER sendListeners(List<? extends Http1ConnectionListener> sendListeners) {
            Objects.requireNonNull(sendListeners);
            isSendListenersMutated = true;
            this.sendListeners.clear();
            this.sendListeners.addAll(sendListeners);
            return self();
        }

        /**
         * Connection send event listeners for HTTP/1.1.
         *
         * @param sendListeners send event listeners
         * @return updated builder instance
         * @see #sendListeners()
         */
        public BUILDER addSendListeners(List<? extends Http1ConnectionListener> sendListeners) {
            Objects.requireNonNull(sendListeners);
            isSendListenersMutated = true;
            this.sendListeners.addAll(sendListeners);
            return self();
        }

        /**
         * Connection send event listeners for HTTP/1.1.
         *
         * @param sendListener send event listeners
         * @return updated builder instance
         * @see #sendListeners()
         */
        public BUILDER addSendListener(Http1ConnectionListener sendListener) {
            Objects.requireNonNull(sendListener);
            this.sendListeners.add(sendListener);
            isSendListenersMutated = true;
            return self();
        }

        /**
         * Connection receive event listeners for HTTP/1.1.
         *
         * @param receiveListeners receive event listeners
         * @return updated builder instance
         * @see #receiveListeners()
         */
        public BUILDER receiveListeners(List<? extends Http1ConnectionListener> receiveListeners) {
            Objects.requireNonNull(receiveListeners);
            isReceiveListenersMutated = true;
            this.receiveListeners.clear();
            this.receiveListeners.addAll(receiveListeners);
            return self();
        }

        /**
         * Connection receive event listeners for HTTP/1.1.
         *
         * @param receiveListeners receive event listeners
         * @return updated builder instance
         * @see #receiveListeners()
         */
        public BUILDER addReceiveListeners(List<? extends Http1ConnectionListener> receiveListeners) {
            Objects.requireNonNull(receiveListeners);
            isReceiveListenersMutated = true;
            this.receiveListeners.addAll(receiveListeners);
            return self();
        }

        /**
         * Connection receive event listeners for HTTP/1.1.
         *
         * @param receiveListener receive event listeners
         * @return updated builder instance
         * @see #receiveListeners()
         */
        public BUILDER addReceiveListener(Http1ConnectionListener receiveListener) {
            Objects.requireNonNull(receiveListener);
            this.receiveListeners.add(receiveListener);
            isReceiveListenersMutated = true;
            return self();
        }

        /**
         * A single send listener, this value is computed.
         *
         * @param compositeSendListener send listener
         * @return updated builder instance
         * @see #compositeSendListener()
         */
        public BUILDER compositeSendListener(Http1ConnectionListener compositeSendListener) {
            Objects.requireNonNull(compositeSendListener);
            this.compositeSendListener = compositeSendListener;
            return self();
        }

        /**
         * A single receive listener, this value is computed.
         *
         * @param compositeReceiveListener receive listener
         * @return updated builder instance
         * @see #compositeReceiveListener()
         */
        public BUILDER compositeReceiveListener(Http1ConnectionListener compositeReceiveListener) {
            Objects.requireNonNull(compositeReceiveListener);
            this.compositeReceiveListener = compositeReceiveListener;
            return self();
        }

        /**
         * Name of this configuration, in most cases the same as {@link #type()}.
         *
         * @return the name
         */
        public Optional<String> name() {
            return Optional.ofNullable(name);
        }

        /**
         * Maximal size of received HTTP prologue (GET /path HTTP/1.1).
         *
         * @return the max prologue length
         */
        public int maxPrologueLength() {
            return maxPrologueLength;
        }

        /**
         * Maximal size of received headers in bytes.
         *
         * @return the max headers size
         */
        public int maxHeadersSize() {
            return maxHeadersSize;
        }

        /**
         * Whether to validate headers.
         * If set to false, any value is accepted, otherwise validates headers + known headers
         * are validated by format
         * (content length is always validated as it is part of protocol processing (other headers may be validated if
         * features use them)).
         * <p>
         *     Defaults to {@code true}.
         * </p>
         *
         * @return the validate request headers
         */
        public boolean validateRequestHeaders() {
            return validateRequestHeaders;
        }

        /**
         * Request host header validation.
         * When host header is invalid, we return {@link io.helidon.http.Status#BAD_REQUEST_400}.
         * <p>
         * The validation is done according to RFC-3986 (see {@link io.helidon.common.uri.UriValidator}). This is a requirement of
         * the HTTP specification.
         * <p>
         * This option allows you to disable the "full-blown" validation ("simple" validation is still in - the port must be
         * parseable to integer).
         *
         * @return the validate request host header
         * @deprecated this switch exists for temporary backward compatible behavior, and will be removed in a future Helidon
         *             version
         */
        @Deprecated(since = "4.1.3", forRemoval = true)
        public boolean validateRequestHostHeader() {
            return validateRequestHostHeader;
        }

        /**
         * Whether to validate headers.
         * If set to false, any value is accepted, otherwise validates headers + known headers
         * are validated by format
         * (content length is always validated as it is part of protocol processing (other headers may be validated if
         * features use them)).
         * <p>
         *     Defaults to {@code false} as user has control on the header creation.
         * </p>
         *
         * @return the validate response headers
         */
        public boolean validateResponseHeaders() {
            return validateResponseHeaders;
        }

        /**
         * If set to false, any query and fragment is accepted (even containing illegal characters).
         * Validation of path is controlled by {@link #validatePath()}.
         *
         * @return the validate prologue
         */
        public boolean validatePrologue() {
            return validatePrologue;
        }

        /**
         * If set to false, any path is accepted (even containing illegal characters).
         *
         * @return the validate path
         */
        public boolean validatePath() {
            return validatePath;
        }

        /**
         * Logging of received packets. Uses trace and debug levels on logger of
         * {@link Http1LoggingConnectionListener} with suffix of {@code .recv`}.
         *
         * @return the receive log
         */
        public boolean receiveLog() {
            return receiveLog;
        }

        /**
         * Logging of sent packets. Uses trace and debug levels on logger of
         * {@link Http1LoggingConnectionListener} with suffix of {@code .send`}.
         *
         * @return the send log
         */
        public boolean sendLog() {
            return sendLog;
        }

        /**
         * When true WebServer answers to expect continue with 100 continue immediately,
         * not waiting for user to actually request the data.
         *
         * @return the continue immediately
         */
        public boolean continueImmediately() {
            return continueImmediately;
        }

        /**
         * Requested URI discovery settings.
         *
         * @return the requested uri discovery
         */
        public Optional<RequestedUriDiscoveryContext> requestedUriDiscovery() {
            return Optional.ofNullable(requestedUriDiscovery);
        }

        /**
         * Connection send event listeners for HTTP/1.1.
         *
         * @return the send listeners
         */
        public List<Http1ConnectionListener> sendListeners() {
            return sendListeners;
        }

        /**
         * Connection receive event listeners for HTTP/1.1.
         *
         * @return the receive listeners
         */
        public List<Http1ConnectionListener> receiveListeners() {
            return receiveListeners;
        }

        /**
         * A single send listener, this value is computed.
         *
         * @return the composite send listener
         */
        public Optional<Http1ConnectionListener> compositeSendListener() {
            return Optional.ofNullable(compositeSendListener);
        }

        /**
         * A single receive listener, this value is computed.
         *
         * @return the composite receive listener
         */
        public Optional<Http1ConnectionListener> compositeReceiveListener() {
            return Optional.ofNullable(compositeReceiveListener);
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "Http1ConfigBuilder{"
                    + "name=" + name + ","
                    + "maxPrologueLength=" + maxPrologueLength + ","
                    + "maxHeadersSize=" + maxHeadersSize + ","
                    + "validateRequestHeaders=" + validateRequestHeaders + ","
                    + "validateRequestHostHeader=" + validateRequestHostHeader + ","
                    + "validateResponseHeaders=" + validateResponseHeaders + ","
                    + "validatePrologue=" + validatePrologue + ","
                    + "validatePath=" + validatePath + ","
                    + "receiveLog=" + receiveLog + ","
                    + "sendLog=" + sendLog + ","
                    + "continueImmediately=" + continueImmediately + ","
                    + "requestedUriDiscovery=" + requestedUriDiscovery + ","
                    + "sendListeners=" + sendListeners + ","
                    + "receiveListeners=" + receiveListeners + ","
                    + "compositeSendListener=" + compositeSendListener + ","
                    + "compositeReceiveListener=" + compositeReceiveListener
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            new Http1BuilderDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (name == null) {
                collector.fatal(getClass(), "Property \"name\" must not be null, but not set");
            }
            if (requestedUriDiscovery == null) {
                collector.fatal(getClass(), "Property \"requested-uri-discovery\" must not be null, but not set");
            }
            if (compositeSendListener == null) {
                collector.fatal(getClass(), "Property \"compositeSendListener\" must not be null, but not set");
            }
            if (compositeReceiveListener == null) {
                collector.fatal(getClass(), "Property \"compositeReceiveListener\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class Http1ConfigImpl implements Http1Config {

            private final boolean continueImmediately;
            private final boolean receiveLog;
            private final boolean sendLog;
            private final boolean validatePath;
            private final boolean validatePrologue;
            private final boolean validateRequestHeaders;
            private final boolean validateRequestHostHeader;
            private final boolean validateResponseHeaders;
            private final Http1ConnectionListener compositeReceiveListener;
            private final Http1ConnectionListener compositeSendListener;
            private final int maxHeadersSize;
            private final int maxPrologueLength;
            private final List<Http1ConnectionListener> receiveListeners;
            private final List<Http1ConnectionListener> sendListeners;
            private final RequestedUriDiscoveryContext requestedUriDiscovery;
            private final String name;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected Http1ConfigImpl(Http1Config.BuilderBase<?, ?> builder) {
                this.name = builder.name().get();
                this.maxPrologueLength = builder.maxPrologueLength();
                this.maxHeadersSize = builder.maxHeadersSize();
                this.validateRequestHeaders = builder.validateRequestHeaders();
                this.validateRequestHostHeader = builder.validateRequestHostHeader();
                this.validateResponseHeaders = builder.validateResponseHeaders();
                this.validatePrologue = builder.validatePrologue();
                this.validatePath = builder.validatePath();
                this.receiveLog = builder.receiveLog();
                this.sendLog = builder.sendLog();
                this.continueImmediately = builder.continueImmediately();
                this.requestedUriDiscovery = builder.requestedUriDiscovery().get();
                this.sendListeners = List.copyOf(builder.sendListeners());
                this.receiveListeners = List.copyOf(builder.receiveListeners());
                this.compositeSendListener = builder.compositeSendListener().get();
                this.compositeReceiveListener = builder.compositeReceiveListener().get();
            }

            @Override
            public String name() {
                return name;
            }

            @Override
            public int maxPrologueLength() {
                return maxPrologueLength;
            }

            @Override
            public int maxHeadersSize() {
                return maxHeadersSize;
            }

            @Override
            public boolean validateRequestHeaders() {
                return validateRequestHeaders;
            }

            @Override
            public boolean validateRequestHostHeader() {
                return validateRequestHostHeader;
            }

            @Override
            public boolean validateResponseHeaders() {
                return validateResponseHeaders;
            }

            @Override
            public boolean validatePrologue() {
                return validatePrologue;
            }

            @Override
            public boolean validatePath() {
                return validatePath;
            }

            @Override
            public boolean receiveLog() {
                return receiveLog;
            }

            @Override
            public boolean sendLog() {
                return sendLog;
            }

            @Override
            public boolean continueImmediately() {
                return continueImmediately;
            }

            @Override
            public RequestedUriDiscoveryContext requestedUriDiscovery() {
                return requestedUriDiscovery;
            }

            @Override
            public List<Http1ConnectionListener> sendListeners() {
                return sendListeners;
            }

            @Override
            public List<Http1ConnectionListener> receiveListeners() {
                return receiveListeners;
            }

            @Override
            public Http1ConnectionListener compositeSendListener() {
                return compositeSendListener;
            }

            @Override
            public Http1ConnectionListener compositeReceiveListener() {
                return compositeReceiveListener;
            }

            @Override
            public String toString() {
                return "Http1Config{"
                        + "name=" + name + ","
                        + "maxPrologueLength=" + maxPrologueLength + ","
                        + "maxHeadersSize=" + maxHeadersSize + ","
                        + "validateRequestHeaders=" + validateRequestHeaders + ","
                        + "validateRequestHostHeader=" + validateRequestHostHeader + ","
                        + "validateResponseHeaders=" + validateResponseHeaders + ","
                        + "validatePrologue=" + validatePrologue + ","
                        + "validatePath=" + validatePath + ","
                        + "receiveLog=" + receiveLog + ","
                        + "sendLog=" + sendLog + ","
                        + "continueImmediately=" + continueImmediately + ","
                        + "requestedUriDiscovery=" + requestedUriDiscovery + ","
                        + "sendListeners=" + sendListeners + ","
                        + "receiveListeners=" + receiveListeners + ","
                        + "compositeSendListener=" + compositeSendListener + ","
                        + "compositeReceiveListener=" + compositeReceiveListener
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof Http1Config other)) {
                    return false;
                }
                return Objects.equals(name, other.name())
                    && maxPrologueLength == other.maxPrologueLength()
                    && maxHeadersSize == other.maxHeadersSize()
                    && validateRequestHeaders == other.validateRequestHeaders()
                    && validateRequestHostHeader == other.validateRequestHostHeader()
                    && validateResponseHeaders == other.validateResponseHeaders()
                    && validatePrologue == other.validatePrologue()
                    && validatePath == other.validatePath()
                    && receiveLog == other.receiveLog()
                    && sendLog == other.sendLog()
                    && continueImmediately == other.continueImmediately()
                    && Objects.equals(requestedUriDiscovery, other.requestedUriDiscovery())
                    && Objects.equals(sendListeners, other.sendListeners())
                    && Objects.equals(receiveListeners, other.receiveListeners())
                    && Objects.equals(compositeSendListener, other.compositeSendListener())
                    && Objects.equals(compositeReceiveListener, other.compositeReceiveListener());
            }

            @Override
            public int hashCode() {
                return Objects.hash(name, maxPrologueLength, maxHeadersSize, validateRequestHeaders, validateRequestHostHeader, validateResponseHeaders, validatePrologue, validatePath, receiveLog, sendLog, continueImmediately, requestedUriDiscovery, sendListeners, receiveListeners, compositeSendListener, compositeReceiveListener);
            }

        }

    }

    /**
     * Fluent API builder for {@link Http1Config}.
     */
    class Builder extends Http1Config.BuilderBase<Http1Config.Builder, Http1Config> implements io.helidon.common.Builder<Http1Config.Builder, Http1Config> {

        private Builder() {
        }

        @Override
        public Http1Config buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new Http1ConfigImpl(this);
        }

        @Override
        public Http1Config build() {
            return buildPrototype();
        }

    }

}
