/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webserver.http1;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;
import io.helidon.webserver.http1.spi.Http1Upgrader;

/**
 * Configuration of an {@link Http1ConnectionSelector}.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.webserver.http1.Http1ConnectionSelectorConfigBlueprint")
public interface Http1ConnectionSelectorConfig extends Http1ConnectionSelectorConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static Http1ConnectionSelectorConfig.Builder builder() {
        return new Http1ConnectionSelectorConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static Http1ConnectionSelectorConfig.Builder builder(Http1ConnectionSelectorConfig instance) {
        return Http1ConnectionSelectorConfig.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static Http1ConnectionSelectorConfig create() {
        return Http1ConnectionSelectorConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link Http1ConnectionSelector}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends Http1ConnectionSelectorConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends Http1ConnectionSelectorConfig> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private final Map<String, Http1Upgrader> upgraders = new LinkedHashMap<>();
        private Http1Config config;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance. This method disables automatic service discovery.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(Http1ConnectionSelectorConfig prototype) {
            addUpgraders(prototype.upgraders());
            config(prototype.config());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(Http1ConnectionSelectorConfig.BuilderBase<?, ?> builder) {
            addUpgraders(builder.upgraders);
            builder.config().ifPresent(this::config);
            return self();
        }

        /**
         * Add a new upgrader, replacing an existing one for the same protocol.
         *
         * @param upgrader upgrader to add
         * @return updated builder instance
         */
        public BUILDER addUpgrader(Http1Upgrader upgrader) {
            Http1ConnectionSelectorConfigBlueprint.CustomMethods.addUpgrader(this, upgrader);
            return self();
        }

        /**
         * This method replaces all values with the new ones.
         *
         * @param upgraders map of protocol name to upgrader
         * @return updated builder instance
         * @see #upgraders()
         */
        public BUILDER upgraders(Map<? extends String, ? extends Http1Upgrader> upgraders) {
            Objects.requireNonNull(upgraders);
            this.upgraders.clear();
            this.upgraders.putAll(upgraders);
            return self();
        }

        /**
         * This method keeps existing values, then puts all new values into the map.
         *
         * @param upgraders map of protocol name to upgrader
         * @return updated builder instance
         * @see #upgraders()
         */
        public BUILDER addUpgraders(Map<? extends String, ? extends Http1Upgrader> upgraders) {
            Objects.requireNonNull(upgraders);
            this.upgraders.putAll(upgraders);
            return self();
        }

        /**
         * This method adds a new value to the map, or replaces it if the key already exists.
         *
         * @param key key to add or replace
         * @param upgrader new value for the key
         * @return updated builder instance
         * @see #upgraders()
         */
        public BUILDER putUpgrader(String key, Http1Upgrader upgrader) {
            Objects.requireNonNull(key);
            Objects.requireNonNull(upgrader);
            this.upgraders.put(key, upgrader);
            return self();
        }

        /**
         * HTTP/1 protocol configuration to use for this connection selector.
         *
         * @param config HTTP/1 protocol configuration
         * @return updated builder instance
         * @see #config()
         */
        public BUILDER config(Http1Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            return self();
        }

        /**
         * HTTP/1 protocol configuration to use for this connection selector.
         *
         * @param consumer consumer of builder for
         *                 HTTP/1 protocol configuration
         * @return updated builder instance
         * @see #config()
         */
        public BUILDER config(Consumer<Http1Config.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = Http1Config.builder();
            consumer.accept(builder);
            this.config(builder.build());
            return self();
        }

        /**
         * HTTP/1 protocol configuration to use for this connection selector.
         *
         * @param supplier supplier of
         *                 HTTP/1 protocol configuration
         * @return updated builder instance
         * @see #config()
         */
        public BUILDER config(Supplier<? extends Http1Config> supplier) {
            Objects.requireNonNull(supplier);
            this.config(supplier.get());
            return self();
        }

        /**
         * Upgraders to support upgrading from HTTP/1.1 to a different protocol (such as {@code websocket}).
         *
         * @return the upgraders
         */
        public Map<String, Http1Upgrader> upgraders() {
            return upgraders;
        }

        /**
         * HTTP/1 protocol configuration to use for this connection selector.
         *
         * @return the config
         */
        public Optional<Http1Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "Http1ConnectionSelectorConfigBuilder{"
                    + "upgraders=" + upgraders + ","
                    + "config=" + config
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (config == null) {
                collector.fatal(getClass(), "Property \"config\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class Http1ConnectionSelectorConfigImpl implements Http1ConnectionSelectorConfig, Supplier<Http1ConnectionSelector> {

            private final Http1Config config;
            private final Map<String, Http1Upgrader> upgraders;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected Http1ConnectionSelectorConfigImpl(Http1ConnectionSelectorConfig.BuilderBase<?, ?> builder) {
                this.upgraders = Collections.unmodifiableMap(new LinkedHashMap<>(builder.upgraders()));
                this.config = builder.config().get();
            }

            @Override
            public Http1ConnectionSelector build() {
                return Http1ConnectionSelector.create(this);
            }

            @Override
            public Http1ConnectionSelector get() {
                return build();
            }

            @Override
            public Map<String, Http1Upgrader> upgraders() {
                return upgraders;
            }

            @Override
            public Http1Config config() {
                return config;
            }

            @Override
            public String toString() {
                return "Http1ConnectionSelectorConfig{"
                        + "upgraders=" + upgraders + ","
                        + "config=" + config
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof Http1ConnectionSelectorConfig other)) {
                    return false;
                }
                return Objects.equals(upgraders, other.upgraders())
                    && Objects.equals(config, other.config());
            }

            @Override
            public int hashCode() {
                return Objects.hash(upgraders, config);
            }

        }

    }

    /**
     * Fluent API builder for {@link Http1ConnectionSelector}.
     */
    class Builder extends Http1ConnectionSelectorConfig.BuilderBase<Http1ConnectionSelectorConfig.Builder, Http1ConnectionSelectorConfig> implements io.helidon.common.Builder<Http1ConnectionSelectorConfig.Builder, Http1ConnectionSelector> {

        private Builder() {
        }

        @Override
        public Http1ConnectionSelectorConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new Http1ConnectionSelectorConfigImpl(this);
        }

        @Override
        public Http1ConnectionSelector build() {
            return Http1ConnectionSelector.create(this.buildPrototype());
        }

    }

}
