/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.orc;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import io.airlift.compress.Compressor;
import io.airlift.compress.lz4.Lz4Compressor;
import io.airlift.compress.snappy.SnappyCompressor;
import io.airlift.compress.zstd.ZstdCompressor;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceOutput;
import io.airlift.slice.Slices;
import io.prestosql.orc.ChunkedSliceOutput;
import io.prestosql.orc.DeflateCompressor;
import io.prestosql.orc.checkpoint.InputStreamCheckpoint;
import io.prestosql.orc.metadata.CompressionKind;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Objects;
import javax.annotation.Nullable;
import org.openjdk.jol.info.ClassLayout;
import sun.misc.Unsafe;

public class OrcOutputBuffer
extends SliceOutput {
    private static final int INSTANCE_SIZE = ClassLayout.parseClass(OrcOutputBuffer.class).instanceSize();
    private static final int INITIAL_BUFFER_SIZE = 256;
    private static final int DIRECT_FLUSH_SIZE = 32768;
    private static final int MINIMUM_OUTPUT_BUFFER_CHUNK_SIZE = 4096;
    private static final int MAXIMUM_OUTPUT_BUFFER_CHUNK_SIZE = 0x100000;
    private final int maxBufferSize;
    private final ChunkedSliceOutput compressedOutputStream;
    @Nullable
    private final Compressor compressor;
    private byte[] compressionBuffer = new byte[0];
    private Slice slice;
    private byte[] buffer;
    private long bufferOffset;
    private int bufferPosition;

    public OrcOutputBuffer(CompressionKind compression, int maxBufferSize) {
        Objects.requireNonNull(compression, "compression is null");
        Preconditions.checkArgument((maxBufferSize > 0 ? 1 : 0) != 0, (Object)"maximum buffer size should be greater than 0");
        this.maxBufferSize = maxBufferSize;
        this.buffer = new byte[256];
        this.slice = Slices.wrappedBuffer((byte[])this.buffer);
        this.compressedOutputStream = new ChunkedSliceOutput(4096, 0x100000);
        if (compression == CompressionKind.NONE) {
            this.compressor = null;
        } else if (compression == CompressionKind.SNAPPY) {
            this.compressor = new SnappyCompressor();
        } else if (compression == CompressionKind.ZLIB) {
            this.compressor = new DeflateCompressor();
        } else if (compression == CompressionKind.LZ4) {
            this.compressor = new Lz4Compressor();
        } else if (compression == CompressionKind.ZSTD) {
            this.compressor = new ZstdCompressor();
        } else {
            throw new IllegalArgumentException("Unsupported compression " + (Object)((Object)compression));
        }
    }

    public long getOutputDataSize() {
        Preconditions.checkState((this.bufferPosition == 0 ? 1 : 0) != 0, (Object)"Buffer must be flushed before getOutputDataSize can be called");
        return this.compressedOutputStream.size();
    }

    public long estimateOutputDataSize() {
        return this.compressedOutputStream.size() + this.bufferPosition;
    }

    public int writeDataTo(SliceOutput outputStream) {
        Preconditions.checkState((this.bufferPosition == 0 ? 1 : 0) != 0, (Object)"Buffer must be closed before writeDataTo can be called");
        for (Slice slice : this.compressedOutputStream.getSlices()) {
            outputStream.writeBytes(slice);
        }
        return this.compressedOutputStream.size();
    }

    public long getCheckpoint() {
        if (this.compressor == null) {
            return this.size();
        }
        return InputStreamCheckpoint.createInputStreamCheckpoint(this.compressedOutputStream.size(), this.bufferPosition);
    }

    public void flush() {
        this.flushBufferToOutputStream();
    }

    public void close() {
        this.flushBufferToOutputStream();
    }

    public void reset() {
        this.compressedOutputStream.reset();
        this.bufferOffset = 0L;
        this.bufferPosition = 0;
    }

    public void reset(int position) {
        throw new UnsupportedOperationException();
    }

    public int size() {
        return Math.toIntExact(this.bufferOffset + (long)this.bufferPosition);
    }

    public long getRetainedSize() {
        return (long)INSTANCE_SIZE + this.compressedOutputStream.getRetainedSize() + this.slice.getRetainedSize() + SizeOf.sizeOf((byte[])this.compressionBuffer);
    }

    public int writableBytes() {
        return Integer.MAX_VALUE;
    }

    public boolean isWritable() {
        return true;
    }

    public void writeByte(int value) {
        this.ensureWritableBytes(1);
        this.slice.setByte(this.bufferPosition, value);
        ++this.bufferPosition;
    }

    public void writeShort(int value) {
        this.ensureWritableBytes(2);
        this.slice.setShort(this.bufferPosition, value);
        this.bufferPosition += 2;
    }

    public void writeInt(int value) {
        this.ensureWritableBytes(4);
        this.slice.setInt(this.bufferPosition, value);
        this.bufferPosition += 4;
    }

    public void writeLong(long value) {
        this.ensureWritableBytes(8);
        this.slice.setLong(this.bufferPosition, value);
        this.bufferPosition += 8;
    }

    public void writeFloat(float value) {
        this.writeInt(Float.floatToIntBits(value));
    }

    public void writeDouble(double value) {
        this.writeLong(Double.doubleToLongBits(value));
    }

    public void writeBytes(Slice source) {
        this.writeBytes(source, 0, source.length());
    }

    public void writeBytes(Slice source, int sourceIndex, int length) {
        if (length >= 32768) {
            this.flushBufferToOutputStream();
            this.writeDirectlyToOutputStream((byte[])source.getBase(), sourceIndex + (int)(source.getAddress() - (long)Unsafe.ARRAY_BYTE_BASE_OFFSET), length);
            this.bufferOffset += (long)length;
        } else {
            this.ensureWritableBytes(length);
            this.slice.setBytes(this.bufferPosition, source, sourceIndex, length);
            this.bufferPosition += length;
        }
    }

    public void writeBytes(byte[] source) {
        this.writeBytes(source, 0, source.length);
    }

    public void writeBytes(byte[] source, int sourceIndex, int length) {
        if (length >= 32768) {
            this.flushBufferToOutputStream();
            this.writeDirectlyToOutputStream(source, sourceIndex, length);
            this.bufferOffset += (long)length;
        } else {
            this.ensureWritableBytes(length);
            this.slice.setBytes(this.bufferPosition, source, sourceIndex, length);
            this.bufferPosition += length;
        }
    }

    public void writeBytes(InputStream in, int inputLength) throws IOException {
        int batch;
        for (int length = inputLength; length > 0; length -= batch) {
            batch = this.ensureBatchSize(length);
            this.slice.setBytes(this.bufferPosition, in, batch);
            this.bufferPosition += batch;
        }
    }

    public void writeZero(int inputLength) {
        int length = inputLength;
        Preconditions.checkArgument((length >= 0 ? 1 : 0) != 0, (Object)"length must be 0 or greater than 0.");
        while (length > 0) {
            int batch = this.ensureBatchSize(length);
            Arrays.fill(this.buffer, this.bufferPosition, this.bufferPosition + batch, (byte)0);
            this.bufferPosition += batch;
            length -= batch;
        }
    }

    public SliceOutput appendLong(long value) {
        this.writeLong(value);
        return this;
    }

    public SliceOutput appendDouble(double value) {
        this.writeDouble(value);
        return this;
    }

    public SliceOutput appendInt(int value) {
        this.writeInt(value);
        return this;
    }

    public SliceOutput appendShort(int value) {
        this.writeShort(value);
        return this;
    }

    public SliceOutput appendByte(int value) {
        this.writeByte(value);
        return this;
    }

    public SliceOutput appendBytes(byte[] source, int sourceIndex, int length) {
        this.writeBytes(source, sourceIndex, length);
        return this;
    }

    public SliceOutput appendBytes(byte[] source) {
        this.writeBytes(source);
        return this;
    }

    public SliceOutput appendBytes(Slice slice) {
        this.writeBytes(slice);
        return this;
    }

    public Slice slice() {
        throw new UnsupportedOperationException();
    }

    public Slice getUnderlyingSlice() {
        throw new UnsupportedOperationException();
    }

    public String toString(Charset charset) {
        throw new UnsupportedOperationException();
    }

    public String toString() {
        StringBuilder builder = new StringBuilder("OutputStreamSliceOutputAdapter{");
        builder.append("outputStream=").append((Object)this.compressedOutputStream);
        builder.append("bufferSize=").append(this.slice.length());
        builder.append('}');
        return builder.toString();
    }

    private void ensureWritableBytes(int minWritableBytes) {
        int neededBufferSize = this.bufferPosition + minWritableBytes;
        if (neededBufferSize <= this.slice.length()) {
            return;
        }
        if (this.slice.length() >= this.maxBufferSize) {
            this.flushBufferToOutputStream();
            return;
        }
        int newBufferSize = Math.min(Math.max(this.slice.length() * 2, minWritableBytes), this.maxBufferSize);
        if (neededBufferSize <= newBufferSize) {
            byte[] previousBuffer = this.buffer;
            this.buffer = new byte[newBufferSize];
            this.slice = Slices.wrappedBuffer((byte[])this.buffer);
            System.arraycopy(previousBuffer, 0, this.buffer, 0, this.bufferPosition);
        } else {
            this.flushBufferToOutputStream();
            this.buffer = new byte[newBufferSize];
            this.slice = Slices.wrappedBuffer((byte[])this.buffer);
        }
    }

    private int ensureBatchSize(int length) {
        this.ensureWritableBytes(Math.min(32768, length));
        return Math.min(length, this.slice.length() - this.bufferPosition);
    }

    private void flushBufferToOutputStream() {
        if (this.bufferPosition > 0) {
            this.writeChunkToOutputStream(this.buffer, 0, this.bufferPosition);
            this.bufferOffset += (long)this.bufferPosition;
            this.bufferPosition = 0;
        }
    }

    private void writeChunkToOutputStream(byte[] chunk, int offset, int length) {
        int compressedSize;
        if (this.compressor == null) {
            this.compressedOutputStream.write(chunk, offset, length);
            return;
        }
        Preconditions.checkArgument((length <= this.buffer.length ? 1 : 0) != 0, (Object)"Write chunk length must be less than compression buffer size");
        int minCompressionBufferSize = this.compressor.maxCompressedLength(length);
        if (this.compressionBuffer.length < minCompressionBufferSize) {
            this.compressionBuffer = new byte[minCompressionBufferSize];
        }
        if ((compressedSize = this.compressor.compress(chunk, offset, length, this.compressionBuffer, 0, this.compressionBuffer.length)) < length) {
            int chunkHeader = compressedSize << 1;
            this.compressedOutputStream.write(chunkHeader & 0xFF);
            this.compressedOutputStream.write((chunkHeader & 0xFF00) >> 8);
            this.compressedOutputStream.write((chunkHeader & 0xFF0000) >> 16);
            this.compressedOutputStream.writeBytes(this.compressionBuffer, 0, compressedSize);
        } else {
            int header = (length << 1) + 1;
            this.compressedOutputStream.write(header & 0xFF);
            this.compressedOutputStream.write((header & 0xFF00) >> 8);
            this.compressedOutputStream.write((header & 0xFF0000) >> 16);
            this.compressedOutputStream.writeBytes(chunk, offset, length);
        }
    }

    private void writeDirectlyToOutputStream(byte[] bytes, int inputBytesOffset, int inputLength) {
        int bytesOffset = inputBytesOffset;
        int length = inputLength;
        if (this.compressor == null) {
            this.compressedOutputStream.writeBytes(bytes, bytesOffset, length);
            return;
        }
        while (length > 0) {
            int chunkSize = Integer.min(length, this.buffer.length);
            this.writeChunkToOutputStream(bytes, bytesOffset, chunkSize);
            length -= chunkSize;
            bytesOffset += chunkSize;
        }
    }

    @VisibleForTesting
    int getBufferCapacity() {
        return this.slice.length();
    }
}

