/*
 * Decompiled with CFR 0.152.
 */
package io.honeycomb.libhoney.transport.batch.impl;

import io.honeycomb.libhoney.LibHoney;
import io.honeycomb.libhoney.eventdata.ResolvedEvent;
import io.honeycomb.libhoney.responses.ResponseObservable;
import io.honeycomb.libhoney.responses.impl.EventResponseFactory;
import io.honeycomb.libhoney.responses.impl.LazyServerResponse;
import io.honeycomb.libhoney.shaded.org.apache.http.HttpResponse;
import io.honeycomb.libhoney.shaded.org.apache.http.client.methods.HttpUriRequest;
import io.honeycomb.libhoney.shaded.org.apache.http.client.methods.RequestBuilder;
import io.honeycomb.libhoney.shaded.org.apache.http.client.utils.URIBuilder;
import io.honeycomb.libhoney.shaded.org.apache.http.concurrent.FutureCallback;
import io.honeycomb.libhoney.shaded.org.apache.http.entity.ByteArrayEntity;
import io.honeycomb.libhoney.shaded.org.apache.http.entity.ContentType;
import io.honeycomb.libhoney.shaded.org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import io.honeycomb.libhoney.shaded.org.apache.http.util.EntityUtils;
import io.honeycomb.libhoney.transport.batch.BatchConsumer;
import io.honeycomb.libhoney.transport.json.BatchRequestSerializer;
import io.honeycomb.libhoney.transport.json.JsonSerializer;
import io.honeycomb.libhoney.utils.ObjectUtils;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HoneycombBatchConsumer
implements BatchConsumer<ResolvedEvent> {
    private static final Logger LOG = LoggerFactory.getLogger(HoneycombBatchConsumer.class);
    private static final String BATCH_ENDPOINT_FORMAT = "/1/batch/%s";
    private static final String WRITE_KEY_HEADER = "X-Honeycomb-Team";
    private static final String USER_AGENT = "libhoney-java/" + (LibHoney.class.getPackage().getImplementationVersion() == null ? "0.0.0" : LibHoney.class.getPackage().getImplementationVersion());
    private final CloseableHttpAsyncClient internalClient;
    private final ResponseObservable observable;
    private final JsonSerializer<List<BatchRequestElement>> batchSerializer;
    private final Semaphore maximumPendingRequestSemaphore;
    private final int maximumPendingRequests;
    private final long maximumHttpRequestShutdownWait;
    private final String userAgentString;

    public HoneycombBatchConsumer(CloseableHttpAsyncClient internalClient, ResponseObservable observable, BatchRequestSerializer batchRequestSerializer, int maximumPendingRequests, int maximumHTTPRequestShutdownWait) {
        this(internalClient, observable, batchRequestSerializer, maximumPendingRequests, maximumHTTPRequestShutdownWait, null);
    }

    public HoneycombBatchConsumer(CloseableHttpAsyncClient internalClient, ResponseObservable observable, JsonSerializer<List<BatchRequestElement>> batchRequestSerializer, int maximumPendingRequests, long maximumHTTPRequestShutdownWait, String additionalUserAgent) {
        this.internalClient = internalClient;
        this.observable = observable;
        this.batchSerializer = batchRequestSerializer;
        this.maximumPendingRequests = maximumPendingRequests;
        this.maximumPendingRequestSemaphore = this.maximumPendingRequests == -1 ? null : new Semaphore(maximumPendingRequests);
        this.maximumHttpRequestShutdownWait = maximumHTTPRequestShutdownWait;
        this.userAgentString = ObjectUtils.isNullOrEmpty(additionalUserAgent) ? USER_AGENT : USER_AGENT + " " + additionalUserAgent;
    }

    @Override
    public void consume(List<ResolvedEvent> batch) throws InterruptedException {
        HttpUriRequest httpPost;
        try {
            List<BatchRequestElement> toSerialize = this.transformToBatchRequestFormat(batch);
            byte[] toSend = this.batchSerializer.serialize(toSerialize);
            httpPost = this.toPostRequest(toSend, batch.get(0));
            if (LOG.isDebugEnabled()) {
                LOG.debug("Sending HTTP request to HoneyComb. URI: {}. Body: {}. Headers: {}.", new Object[]{httpPost.getURI(), new String(toSend, StandardCharsets.UTF_8), Arrays.asList(httpPost.getAllHeaders())});
            }
        }
        catch (Exception ex) {
            this.requestBuildFailure(batch, ex);
            LOG.error("Failed to construct HTTP request for submission to HTTP client. Error has been reported to ResponseObservers.", (Throwable)ex);
            return;
        }
        if (this.maximumPendingRequestSemaphore != null) {
            this.maximumPendingRequestSemaphore.acquire();
        }
        try {
            this.internalClient.execute(httpPost, new ResponseHandlingFutureCallback(batch));
        }
        catch (Exception ex) {
            this.releaseSemaphore();
            this.consumeFailed(batch, "Unexpected failure while submitting request to HTTP client", ex);
            LOG.error("HTTP client rejected batch request. Error has been reported to ResponseObservers.", (Throwable)ex);
        }
    }

    private HttpUriRequest toPostRequest(byte[] toSend, ResolvedEvent event) throws URISyntaxException {
        String path = String.format(BATCH_ENDPOINT_FORMAT, event.getDataset());
        URI finalUri = new URIBuilder(event.getApiHost()).setPath(path).build();
        return RequestBuilder.post(finalUri).addHeader(WRITE_KEY_HEADER, event.getWriteKey()).addHeader("User-Agent", this.userAgentString).setEntity(new ByteArrayEntity(toSend, ContentType.APPLICATION_JSON)).build();
    }

    private List<BatchRequestElement> transformToBatchRequestFormat(List<ResolvedEvent> batch) {
        ArrayList<BatchRequestElement> elements = new ArrayList<BatchRequestElement>(batch.size());
        SimpleDateFormat localDateFormat = ObjectUtils.getRFC3339DateTimeFormatter();
        for (ResolvedEvent event : batch) {
            String dateTimeString = localDateFormat.format(new Date(event.getTimestamp()));
            elements.add(new BatchRequestElement(dateTimeString, event.getSampleRate(), event.getFields()));
        }
        return elements;
    }

    private void requestBuildFailure(List<ResolvedEvent> batch, Exception exception) {
        for (ResolvedEvent resolvedEvent : batch) {
            this.observable.publish(EventResponseFactory.requestBuildFailure(resolvedEvent, exception));
        }
    }

    private void consumeFailed(List<ResolvedEvent> batch, String message, Exception exception) {
        for (ResolvedEvent resolvedEvent : batch) {
            this.observable.publish(EventResponseFactory.httpClientError(resolvedEvent, message, exception));
        }
    }

    private void releaseSemaphore() {
        if (this.maximumPendingRequestSemaphore != null) {
            this.maximumPendingRequestSemaphore.release();
        }
    }

    @Override
    public void close() throws IOException {
        try {
            if (this.maximumPendingRequestSemaphore != null) {
                LOG.debug("Waiting for pending HTTP requests to complete.");
                this.maximumPendingRequestSemaphore.tryAcquire(this.maximumPendingRequests, this.maximumHttpRequestShutdownWait, TimeUnit.MILLISECONDS);
            } else {
                LOG.debug("Waiting for pending HTTP requests to complete.");
                Thread.sleep(this.maximumHttpRequestShutdownWait);
            }
        }
        catch (InterruptedException ex) {
            LOG.error("Interrupted during wait for HTTP requests to complete", (Throwable)ex);
            Thread.currentThread().interrupt();
        }
        LOG.debug("Closing HTTP client");
        this.internalClient.close();
        LOG.debug("Closed HTTP client");
    }

    private class ResponseHandlingFutureCallback
    implements FutureCallback<HttpResponse> {
        private final List<ResolvedEvent> batch;

        ResponseHandlingFutureCallback(List<ResolvedEvent> batch) {
            this.batch = batch;
            this.markStartOfHttpRequest(batch);
        }

        private void markStartOfHttpRequest(List<ResolvedEvent> batch) {
            for (ResolvedEvent resolvedEvent : batch) {
                resolvedEvent.markStartOfHttpRequest();
            }
        }

        private void markEndOfHttpRequest() {
            for (ResolvedEvent event : this.batch) {
                event.markEndOfHttpRequest();
            }
        }

        @Override
        public void completed(HttpResponse httpResponse) {
            HoneycombBatchConsumer.this.releaseSemaphore();
            this.consumeSuccessful(httpResponse);
        }

        @Override
        public void failed(Exception exception) {
            HoneycombBatchConsumer.this.releaseSemaphore();
            HoneycombBatchConsumer.this.consumeFailed(this.batch, "HTTP client completed request with an exception", exception);
            LOG.error("Unexpected error. Batch request failed. An error has been published to the ResponseObservers for each event in the errored batch.");
        }

        @Override
        public void cancelled() {
            HoneycombBatchConsumer.this.releaseSemaphore();
            HoneycombBatchConsumer.this.consumeFailed(this.batch, "HTTP client request was unexpectedly cancelled", null);
            LOG.error("Unexpected error. Batch request cancelled. An error has been published to the ResponseObservers for each event in the errored batch.");
        }

        private void consumeSuccessful(HttpResponse httpResponse) {
            this.markEndOfHttpRequest();
            if (httpResponse.getStatusLine().getStatusCode() == 401 && !HoneycombBatchConsumer.this.observable.hasObservers()) {
                LOG.error("Server responded with a 401 HTTP error code to a batch request. This is likely caused by using an incorrect 'Team Write Key'. Check https://ui.honeycomb.io/account to verify your team write key. An error has been published to the ResponseObservers for each event in the errored batch.");
            }
            if (HoneycombBatchConsumer.this.observable.hasObservers()) {
                try {
                    List<LazyServerResponse> toPublish = LazyServerResponse.createEventsWithServerResponse(this.batch, EntityUtils.toByteArray(httpResponse.getEntity()), httpResponse.getStatusLine().getStatusCode());
                    for (LazyServerResponse response : toPublish) {
                        response.publishTo(HoneycombBatchConsumer.this.observable);
                    }
                }
                catch (IOException e) {
                    for (ResolvedEvent resolvedEvent : this.batch) {
                        HoneycombBatchConsumer.this.observable.publish(EventResponseFactory.httpClientError(resolvedEvent, "Reading from HTTP response threw an exception", e));
                    }
                    LOG.error("Unable to read server HTTP response. An error has been published to the ResponseObservers.", (Throwable)e);
                }
            } else {
                EntityUtils.consumeQuietly(httpResponse.getEntity());
                LOG.trace("No observers registered so not publishing to responses");
            }
        }
    }

    public static class BatchRequestElement {
        private final String time;
        private final int samplerate;
        private final Map<String, Object> data;

        public BatchRequestElement(String time, int samplerate, Map<String, Object> data) {
            this.time = time;
            this.samplerate = samplerate;
            this.data = data;
        }

        public String getTime() {
            return this.time;
        }

        public Map<String, Object> getData() {
            return this.data;
        }

        public int getSamplerate() {
            return this.samplerate;
        }
    }
}

