package io.hypertrack.lib.transmitter.service;

import android.content.ContentValues;
import android.database.Cursor;
import android.database.DatabaseUtils;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteStatement;
import android.text.TextUtils;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

import io.hypertrack.lib.common.util.HTLog;

/**
 * Created by piyush on 09/08/16.
 */
/** package */ class DeviceInfoTable {

    private static final String TAG = DeviceInfoTable.class.getSimpleName();
    private static final int DEVICE_INFO_REQUEST_QUERY_LIMIT = 500;

    private static final String TABLE_NAME = "device_info";
    private static final String COLUMN_ID = "_id";
    private static final String COLUMN_DRIVER_ID = "driver_id";
    private static final String COLUMN_DEVICE_INFO = "device_info";

    private static final String DATABASE_CREATE = "CREATE TABLE IF NOT EXISTS "
            + TABLE_NAME
            + " ("
            + COLUMN_ID + " INTEGER PRIMARY KEY AUTOINCREMENT, "
            + COLUMN_DRIVER_ID + " TEXT, "
            + COLUMN_DEVICE_INFO + " TEXT"
            + ");";

    private static final String BULK_INSERT_DEVICE_INFO_FOR_DRIVER_ID = "INSERT INTO "
            + TABLE_NAME
            + " (" + COLUMN_DRIVER_ID + ", " + COLUMN_DEVICE_INFO + ")"
            + " VALUES (?,?);";

    public static void onCreate(SQLiteDatabase db) {
        if (db == null) {
            return;
        }

        try {
            db.execSQL(DATABASE_CREATE);
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "DeviceInfoTable: Exception occurred while onCreate: " + e);
        }
    }

    public static void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        if (db == null) {
            return;
        }

        try {
            db.execSQL("DROP TABLE IF EXISTS " + TABLE_NAME);
            onCreate(db);
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "DeviceInfoTable: Exception occurred while onUpgrade: " + e);
        }
    }

    public static long getCount(SQLiteDatabase db) {
        try {
            if (db == null) {
                return 0;
            }

            return DatabaseUtils.queryNumEntries(db, TABLE_NAME);
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "DeviceInfoTable: Exception occurred while getCount: " + e);
            return 0L;
        }
    }

    public static void addDeviceInfo(SQLiteDatabase db, String driverID, JSONObject deviceInfo) {
        if (db == null || TextUtils.isEmpty(driverID) || TextUtils.isEmpty(deviceInfo.toString())) {
            return;
        }
        ContentValues contentValues = new ContentValues();

        String deviceInfoJson = deviceInfo.toString();

        contentValues.put(COLUMN_DEVICE_INFO, deviceInfoJson);
        contentValues.put(COLUMN_DRIVER_ID, driverID);

        try {
            db.insert(TABLE_NAME, null, contentValues);
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "DeviceInfoTable: Exception occurred while addDeviceInfo: " + e.getMessage());
        }
    }

    public static void addDeviceInfo(SQLiteDatabase db, String driverID, List<JSONObject> deviceInfoList) {
        if (db == null || TextUtils.isEmpty(driverID) || deviceInfoList == null || deviceInfoList.isEmpty()) {
            return;
        }

        SQLiteStatement statement = db.compileStatement(BULK_INSERT_DEVICE_INFO_FOR_DRIVER_ID);
        try {
            db.beginTransaction();
            for (JSONObject deviceInfo : deviceInfoList) {

                String deviceInfoJson = deviceInfo.toString();

                statement.clearBindings();
                statement.bindString(1, driverID);
                statement.bindString(2, deviceInfoJson);
                statement.execute();
            }
            db.setTransactionSuccessful();
            db.endTransaction();
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "DeviceInfoTable: Exception occurred while addDeviceInfo: " + e.getMessage());
        }
    }

    public static void deleteDeviceInfo(SQLiteDatabase db, List<DeviceInfo> deviceInfoList) {
        if (db == null)
            return;

        StringBuilder builder = new StringBuilder();
        for (DeviceInfo deviceInfo : deviceInfoList) {
            if (deviceInfo != null && deviceInfo.getId() > 0) {
                builder.append(deviceInfo.getId())
                        .append(",");
            }
        }

        if (builder.length() == 0) {
            return;
        }

        try {
            String ids = builder.toString();
            ids = ids.substring(0, ids.length() - 1);

            String whereClause = COLUMN_ID +
                    " IN (" +
                    ids +
                    ")";

            db.delete(TABLE_NAME, whereClause, null);
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "DeviceInfoTable: Exception occurred while deleteDeviceInfo: " + e.getMessage());
        }
    }

    public static void deleteAllDeviceInfo(SQLiteDatabase db) {
        if (db == null) {
            return;
        }

        try {
            db.delete(TABLE_NAME, null, null);
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "DeviceInfoTable: Exception occurred while deleteAllDeviceInfo: " + e.getMessage());
        }
    }

    public static void deleteDeviceInfoOtherThanDriverID(SQLiteDatabase db, String driverID) {
        if (db == null || TextUtils.isEmpty(driverID)) {
            return;
        }

        try {
            db.delete(TABLE_NAME, COLUMN_DRIVER_ID + "<>?", new String[]{driverID});
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "DeviceInfoTable: Exception occurred while deleteDeviceInfoOtherThanDriverID: " + e.getMessage());
        }
    }

    public static List<DeviceInfo> getDeviceInfoWithDriverID(SQLiteDatabase db, String driverID) {
        if (db == null || driverID == null) {
            return null;
        }

        Cursor cursor = db.query(TABLE_NAME, new String[]{COLUMN_ID, COLUMN_DEVICE_INFO},
                COLUMN_DRIVER_ID + "=?", new String[]{driverID}, null, null, null,
                String.valueOf(DEVICE_INFO_REQUEST_QUERY_LIMIT));

        if (cursor == null || cursor.isClosed()) {
            return null;
        }

        ArrayList<DeviceInfo> deviceInfoList = null;

        try {
            if (cursor.moveToFirst()) {
                deviceInfoList = new ArrayList<>();

                do {
                    if (cursor.isClosed()) {
                        break;
                    }

                    String deviceInfoString = cursor.getString(1);
                    if (!TextUtils.isEmpty(deviceInfoString)) {
                        DeviceInfo deviceInfo = new DeviceInfo(new JSONObject(deviceInfoString));

                        // Get RowId for DeviceInfo
                        Integer rowId = Integer.valueOf(cursor.getString(0));
                        deviceInfo.setId(rowId != null ? rowId : 0);

                        deviceInfoList.add(deviceInfo);
                    }
                } while (cursor.moveToNext());
            }
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "DeviceInfoTable: Exception occurred while getDeviceInfoWithDriverID: " + e.getMessage());
        } finally {
            cursor.close();
        }

        return deviceInfoList;
    }
}
