package io.hypertrack.lib.transmitter.service;

import android.Manifest;
import android.app.IntentService;
import android.app.PendingIntent;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.location.Location;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.IBinder;
import android.provider.Settings;
import android.support.annotation.NonNull;
import android.support.v4.app.ActivityCompat;
import android.support.v4.content.LocalBroadcastManager;
import android.text.TextUtils;
import android.util.Log;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.location.ActivityRecognition;
import com.google.android.gms.location.LocationListener;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationServices;

import java.util.ArrayList;
import java.util.List;

import io.hypertrack.lib.common.controls.SDKControls;
import io.hypertrack.lib.common.model.HTConstants;
import io.hypertrack.lib.common.util.HTLog;
import io.hypertrack.lib.common.util.Utils;
import io.hypertrack.lib.transmitter.model.TransmitterConstants;

/**
 * Created by piyush on 06/08/16.
 */
public class HTLocationService extends IntentService implements GoogleApiClient.ConnectionCallbacks,
        GoogleApiClient.OnConnectionFailedListener, LocationListener {

    private final static String TAG = HTLocationService.class.getSimpleName();
    public static final String STOP_LOCATION_SERVICE_INTENT = "io.hypertrack.lib:StopLocationService";
    private static final int FOREGROUND_ID = 101010;

    public static HTLocationService htLocationService;

    private GoogleApiClient mGoogleApiClient;
    private LocationRequest mLocationRequest;

    private String driverID;
    private SDKControls sdkControls;

    private PendingIntent mActivityPendingIntent;
    private String mActivityName;
    private Integer mActivityConfidence;

    private GPSLocationList mLocations;
    private android.location.LocationManager locationManager;
    private GPSStatusListener gpsStatusListener;

    private boolean powerSaverModeRegistered = false;

    // Private Constructor to prevent instantiation of LocationService from other modules
    public HTLocationService() {
        super("HTLocationService");
    }

    // On Service Destroyed
    @Override
    public void onDestroy() {
        try {
            htLocationService = null;

            this.stopServices();
            this.stopListenForGPSStatus();
            this.registerPowerSaverModeReceiver(false);
            super.onDestroy();
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while HTLocationService.onDestroy: " + e);
        }
    }

    // Stop Services & Updates
    private void stopServices() {
        // Stop Foreground Service Notification
        this.stopForeground(true);

        // Stop Location & Activity Updates
        this.stopLocationAndActivityUpdates();

        // Disconnect GoogleAPIClient
        if (mGoogleApiClient != null && mGoogleApiClient.isConnected()) {
            mGoogleApiClient.disconnect();
        }
    }

    // Stop Location & Activity Updates
    private void stopLocationAndActivityUpdates() {
        this.stopLocationPolling();
        this.stopActivityRecognition();
        this.stopListenForGPSStatus();
    }

    // Self Stopping the Service
    public void stopSelfService() {
        HTLog.i(TAG, "Self stopping LocationService");
        this.sendStopServiceBroadcast();
        this.stopSelf();
    }

    // Send Broadcast on Service Stopped
    private void sendStopServiceBroadcast() {
        HTLog.i(TAG, "Broadcasting LocationService stop signal");
        Intent intent = new Intent(STOP_LOCATION_SERVICE_INTENT);
        LocalBroadcastManager.getInstance(this).sendBroadcast(intent);
    }

    @Override
    public void onCreate() {
        super.onCreate();
    }

    @Override
    protected void onHandleIntent(Intent intent) {
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {

        try {
            HTLog.i(TAG, "onStartCommand for LocationService called");

            // Check if driverID for Location Service has been setup
            if (!this.setupDriverID()) {
                this.stopSelfService();
                return START_STICKY;

            } else {
                if (intent != null && intent.hasExtra(TransmitterConstants.HT_SDK_CONTROLS_KEY)) {
                    sdkControls = (SDKControls) intent.getSerializableExtra(TransmitterConstants.HT_SDK_CONTROLS_KEY);
                }

                // Initialize Location DB & GPSStatusListener
                setupLocationManager();

                // Configure
                this.configureLocationServiceForID();

                //Create Location Request object
                createLocationRequest();

                //Construct and connect GoogleApiClient object
                connectGoogleApiClient();

                // Start Service in Foreground Mode
                this.startForeground();

                htLocationService = this;
            }

            this.startListenForGPSStatus();
            this.registerPowerSaverModeReceiver(true);
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while HTLocationService.onCreate: " + e);
        }

        HTLog.i(TAG, "LocationService Duration: " + sdkControls.getMinimumDuration()
                + ", Displacement: " + sdkControls.getMinimumDisplacement());

        return START_REDELIVER_INTENT;
    }

    // Setup driverID for HTLocationService
    private boolean setupDriverID() {
        SharedPreferences preferences = getSharedPreferences(HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        String driverID = preferences.getString(TransmitterConstants.HT_SHARED_PREFERENCE_DRIVER_ID_KEY, null);

        if (TextUtils.isEmpty(driverID)) {
            HTLog.e(TAG, "driverID for LocationService is empty or null");

            return false;
        }

        this.driverID = driverID;
        return true;
    }

    private void configureLocationServiceForID() {
        // Clear any pending data for other driverID here
        try {
            mLocations.clearLocationsOtherThanDriverID(driverID);
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while configureLocationServiceForID(): " + e.getMessage());
        }
    }

    // Activity Recognition Receiver
    private BroadcastReceiver mActivityRecognitionReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            mActivityName = intent.getStringExtra(ActivityRecognitionIntentService.ACTIVITY_NAME_EXTRA_FIELD);
            mActivityConfidence = intent.getIntExtra(ActivityRecognitionIntentService.ACTIVITY_CONFIDENCE_EXTRA_FIELD, -1);
        }
    };

    // Location Changed Listener
    @Override
    public void onLocationChanged(Location location) {
        try {
            if (location == null || location.getLatitude() == 0.0 || location.getLongitude() == 0.0) {
                HTLog.e(TAG, "Invalid Location received in onLocationChanged: " +
                        (location != null ? location.toString() : "null"));
                return;
            } else {
                HTLog.i(TAG, "Location Changed: " + location.toString());
            }

            // Initialize Location DB
            setupLocationManager();

            // Add GPSLocation to DB
            addGPSLocation(location);

            // Broadcast Driver's Current Location
            broadcastCurrentLocation(location);
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while onLocationChanged: " + e);
        }
    }

    private void setupLocationManager() {
        if (this.mLocations == null) {
            this.mLocations = new GPSLocationList(LocationsDatabaseHelper.getInstance(this));
        }

        startListenForGPSStatus();
    }

    private void addGPSLocation(Location location) {
        // Initialize GPSStatusListener in case it is not initialized yet
        this.startListenForGPSStatus();

        String provider = null;
        if (gpsStatusListener != null)
            provider = gpsStatusListener.getProvider();

        // Update GPSLocation with Location Provider, Activity Data & DriverID
        GPSLocation gpsLocation = new GPSLocation(location, provider);
        gpsLocation.setDriverID(this.driverID);
        this.updateGPSLocationActivity(gpsLocation);

        // Add Location to Locations List
        mLocations.addLocation(gpsLocation);
    }

    private void updateGPSLocationActivity(GPSLocation location) {
        if (mActivityName != null && mActivityConfidence != -1) {
            List<String> activities = new ArrayList<>();
            activities.add(mActivityName);

            location.setActivityDetails(activities, mActivityConfidence);
        }
    }

    private void broadcastCurrentLocation(Location location) {
        try {
            Bundle bundle = new Bundle();
            bundle.putString(TransmitterConstants.HT_DRIVER_ID_KEY, driverID);
            bundle.putParcelable(TransmitterConstants.HT_DRIVER_CURRENT_LOCATION_KEY, location);

            Intent intent = new Intent(TransmitterConstants.HT_DRIVER_CURRENT_LOCATION_INTENT);
            intent.putExtras(bundle);
            LocalBroadcastManager.getInstance(getApplicationContext()).sendBroadcast(intent);
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while broadcastCurrentLocation: " + e);
        }
    }

    private void registerPowerSaverModeReceiver(boolean register) {
        try {
            if (register && !powerSaverModeRegistered) {
                powerSaverModeRegistered = true;
                this.registerReceiver(mPowerSaverModeChangedReceiver,
                        new IntentFilter("android.os.action.POWER_SAVE_MODE_CHANGED"));
            } else if (powerSaverModeRegistered) {
                powerSaverModeRegistered = false;
                this.unregisterReceiver(mPowerSaverModeChangedReceiver);
            }
        } catch (IllegalArgumentException e) {
            HTLog.e(TAG, "Exception occurred while registerPowerSaverModeReceiver(" + register + "): " + e);
        }
    }

    BroadcastReceiver mPowerSaverModeChangedReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            // Check for PowerSaverMode
            checkForPowerSaverMode(context);
        }
    };

    private void checkForPowerSaverMode(Context context) {
        try {
            // Update HTJobScheduler on PowerSaver Mode Change
            boolean isPowerSaveMode = Utils.checkIfPowerSaverModeEnabled(context);
            HTTransmitterService.getInstance(context).getJobScheduler().onPowerSaverModeChanged(isPowerSaveMode);
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while checkForPowerSaverMode: " + e);
        }
    }

    private void startForeground() {
        try {
            startForeground(FOREGROUND_ID, HTServiceNotificationUtils.getForegroundNotification(getApplicationContext()));
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while startForeground: " + e);
        }
    }

    public void updateForegroundNotification() {
        try {
            startForeground(FOREGROUND_ID, HTServiceNotificationUtils.getForegroundNotification(getApplicationContext()));
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while updateForegroundNotification: " + e);
        }
    }

    @Override
    public void onConnected(Bundle bundle) {
        Log.i(TAG, "Connection to GoogleApiClient SUCCESSFUL");
        //Start Location Polling to get Location updates regularly
        startLocationPolling();
        startActivityRecognition();
    }

    /**
     * Method to initiate Location Polling
     */
    private void startLocationPolling() {
        try {
            if (mGoogleApiClient != null && mLocationRequest != null) {

                if (mGoogleApiClient.isConnected()) {

                    if (isLocationEnabled(this)) {

                        if (ActivityCompat.checkSelfPermission(this, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED
                                && ActivityCompat.checkSelfPermission(this, Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
                            HTLog.e(TAG, "Location Permission unavailable, startLocationPolling failed");
                            return;
                        }

                        LocationServices.FusedLocationApi.requestLocationUpdates(mGoogleApiClient, mLocationRequest, this);
                        HTLog.i(TAG, "FusedLocation Updates Initiated!");

                    } else {
                        HTLog.e(TAG, "Could not initiate startLocationPolling: locationEnabled: " + isLocationEnabled(this));
                    }
                    return;

                } else {
                    HTLog.w(TAG, "GoogleAPIClient not connected yet. Retrying to connect");
                }
            }

            HTLog.w(TAG, "GoogleAPIClient or Location Request null. Re-initializing them.");

            //Create Location Request object
            this.createLocationRequest();

            //Construct and connect GoogleApiClient object
            this.connectGoogleApiClient();
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while startLocationPolling: " + e);
        }
    }

    /**
     * Method to cease further Location Polling
     */
    private void stopLocationPolling() {
        if (mGoogleApiClient != null && mGoogleApiClient.isConnected()) {
            LocationServices.FusedLocationApi.removeLocationUpdates(mGoogleApiClient, HTLocationService.this);
        }
    }

    private void startListenForGPSStatus() {
        if (gpsStatusListener != null)
            return;

        try {
            locationManager = (android.location.LocationManager) this.getSystemService(this.LOCATION_SERVICE);
            gpsStatusListener = new GPSStatusListener(locationManager);

            if (ActivityCompat.checkSelfPermission(this, Manifest.permission.ACCESS_FINE_LOCATION)
                    != PackageManager.PERMISSION_GRANTED) {
                HTLog.w(TAG, "Could not start Listen For GPS Status: ACCESS_FINE_LOCATION permission unavailable");
                return;
            }

            if (!locationManager.addGpsStatusListener(gpsStatusListener)) {
                HTLog.e(TAG, "GpsStatusListener could NOT be added successfully");
            }
        } catch (Exception e) {
            HTLog.w(TAG, "Exception occurred while start Listen For GPS Status: " + e.getMessage());
        }
    }

    private void stopListenForGPSStatus() {
        if (locationManager != null) {
            locationManager.removeGpsStatusListener(gpsStatusListener);
            locationManager = null;
            gpsStatusListener = null;
        }
    }

    private void startActivityRecognition() {
        try {
            if (mActivityPendingIntent != null)
                return;

            // Get Minimum Duration to start ActivityRecognition
            long updateDuration = sdkControls.getMinimumDuration() * 1000;

            Intent activityIntent = new Intent(this, ActivityRecognitionIntentService.class);
            mActivityPendingIntent = PendingIntent.getService(this, 0, activityIntent, PendingIntent.FLAG_UPDATE_CURRENT);
            ActivityRecognition.ActivityRecognitionApi.requestActivityUpdates(mGoogleApiClient, updateDuration, mActivityPendingIntent);

            this.registerForActivityRecgonitionIntents();
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while startActivityRecognition: " + e);
        }
    }

    private void stopActivityRecognition() {
        this.unregisterForActivityRecognitionIntents();
        this.removeActivityUpdates();
    }

    private void removeActivityUpdates() {
        if (mGoogleApiClient != null && mActivityPendingIntent != null) {
            ActivityRecognition.ActivityRecognitionApi.removeActivityUpdates(mGoogleApiClient, mActivityPendingIntent);
            mActivityPendingIntent = null;
        }
    }

    private void unregisterForActivityRecognitionIntents() {
        if (mActivityRecognitionReceiver != null) {
            LocalBroadcastManager.getInstance(this).unregisterReceiver(mActivityRecognitionReceiver);
            mActivityRecognitionReceiver = null;
        }
    }

    private void registerForActivityRecgonitionIntents() {
        IntentFilter filter = new IntentFilter();
        filter.addAction(ActivityRecognitionIntentService.ACTIVITY_CHANGED_INTENT_ACTION);

        LocalBroadcastManager.getInstance(this).registerReceiver(mActivityRecognitionReceiver, filter);
    }

    /**
     * Method to instantiate Google API Client object and initiate its connection
     */
    private boolean connectGoogleApiClient() {
        if (mGoogleApiClient == null) {
            mGoogleApiClient = new GoogleApiClient.Builder(this)
                    .addConnectionCallbacks(this)
                    .addOnConnectionFailedListener(this)
                    .addApi(LocationServices.API)
                    .addApi(ActivityRecognition.API)
                    .build();
        }

        if (!mGoogleApiClient.isConnected()) {
            mGoogleApiClient.connect();
            HTLog.i(TAG, "Initiating GoogleAPIClient connection");
            return false;
        }

        return true;
    }

    /**
     * Method to instantiate Location Request object with Location Interval and priority params specified
     */
    public void createLocationRequest() {
        if (mLocationRequest == null) {
            mLocationRequest = new LocationRequest();
        }

        mLocationRequest.setPriority(LocationRequest.PRIORITY_HIGH_ACCURACY);
        mLocationRequest.setInterval(sdkControls.getMinimumDuration() * 1000);
        mLocationRequest.setFastestInterval(sdkControls.getMinimumDuration() * 1000);
        mLocationRequest.setSmallestDisplacement(sdkControls.getMinimumDisplacement());
    }

    private boolean isLocationEnabled(Context context) {
        int locationMode = 0;
        String locationProviders;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
            try {
                locationMode = Settings.Secure.getInt(context.getContentResolver(), Settings.Secure.LOCATION_MODE);

            } catch (Settings.SettingNotFoundException e) {
                e.printStackTrace();
                HTLog.w(TAG, "Exception occurred while detecting isLocationEnabled: " + e.getMessage());
            }

            return locationMode != Settings.Secure.LOCATION_MODE_OFF;

        } else {
            locationProviders = Settings.Secure.getString(context.getContentResolver(), Settings.Secure.LOCATION_PROVIDERS_ALLOWED);
            return !TextUtils.isEmpty(locationProviders);
        }
    }

    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    @Override
    public void onConnectionFailed(@NonNull ConnectionResult connectionResult) {
        HTLog.i(TAG, "GoogleAPIClient Connection Failed: " + connectionResult.toString());

        // Retry to connect to GoogleAPIClient
        connectGoogleApiClient();
    }

    @Override
    public void onConnectionSuspended(int i) {
        // If your connection to the sensor gets lost at some point,
        // you'll be able to determine the reason and react to it here.
        if (i == GoogleApiClient.ConnectionCallbacks.CAUSE_NETWORK_LOST) {
            HTLog.i(TAG, "GoogleAPIClient Connection Suspended: Network Lost.");
        } else if (i == GoogleApiClient.ConnectionCallbacks.CAUSE_SERVICE_DISCONNECTED) {
            HTLog.i(TAG, "GoogleAPIClient Connection Suspended: Service Disconnected");
        } else {
            HTLog.i(TAG, "GoogleAPIClient Connection Suspended");
        }

        // Retry to connect to GoogleAPIClient
        new Handler().postDelayed(new Runnable() {
            @Override
            public void run() {
                connectGoogleApiClient();
            }
        }, 1000);
    }
}