package io.hypertrack.lib.transmitter.service;

import android.content.Context;
import android.support.annotation.NonNull;

import io.hypertrack.lib.common.scheduler.HTJobScheduler;
import io.hypertrack.lib.transmitter.model.HTShiftParams;
import io.hypertrack.lib.transmitter.model.callback.HTStartDriverStatusCallback;
import io.hypertrack.lib.transmitter.model.HTTripParams;
import io.hypertrack.lib.transmitter.model.ServiceNotificationParams;
import io.hypertrack.lib.transmitter.model.callback.HTCompleteTaskStatusCallback;
import io.hypertrack.lib.transmitter.model.callback.HTEndAllTripsCallback;
import io.hypertrack.lib.transmitter.model.callback.HTShiftStatusCallback;
import io.hypertrack.lib.transmitter.model.callback.HTTaskStatusCallback;
import io.hypertrack.lib.transmitter.model.callback.HTTripStatusCallback;

/**
 * This class which provides you an interface for APIs to HyperTrack TransmitterSDK.
 * It can be used to start/complete Tasks, Trips or Shifts. For further info refer to the
 * HyperTrack TransmitterSDK documentation.
 *
 * @see <a href="http://docs.hypertrack.io/docs/getting-started-android-driver">http://docs.hypertrack.io/docs/getting-started-android-driver</a>
 * <p>
 * You can obtain an instance of this class by calling
 * {@link io.hypertrack.lib.transmitter.service.HTTransmitterService#getInstance(Context)}.
 * <p>
 * The primary API you'll use is {@link #startTrip(HTTripParams, HTTripStatusCallback)},
 * {@link #completeTask(String, HTCompleteTaskStatusCallback)} and {@link #endTrip(String, HTTripStatusCallback)}.
 * These APIs can be used to start a trip, mark tasks complete and end a trip.
 */
public abstract class HTTransmitterService {

    private static HTTransmitterService sInstance;

    protected TransmissionManager transmissionManager;
    protected HTJobScheduler jobScheduler;

    /**
     * Gets an instance of HTTransmitterService class.
     * Use this reference to start a trip, end a trip or to mark a task as complete.
     *
     * @param context Pass Context as a parameter
     * @return HTTransmitterService
     */
    public static HTTransmitterService getInstance(Context context) {

        if (sInstance == null) {
            sInstance = HTTransmitterServiceImpl.getInstance(context);
        }

        return sInstance;
    }

    /**
     * Method to initialize HTTransmitter (HyperTrack's Transmitter SDK)
     * <p>
     * <u><b>IMPORTANT:</b></u>
     * Call this method from your Launcher Activity file with ApplicationContext as parameter.
     *
     * @param context Pass ApplicationContext as the parameter
     */
    public static void initHTTransmitter(Context context) {
        HTTransmitterService.getInstance(context).initializeHTTransmitter();
    }

    /**
     * Method to establish DriverSDK connection for a DriverID. Call this method when the DriverSDK connection
     * needs to established, to be able to implement backend-start calls. (Preferably in the onResume()
     * method of your app's Launcher Activity)
     * <p>
     * For more reliability in backend-start calls, call this method just before backend-start need to
     * happen in your app's workflow.
     * <p>
     * <u><b>IMPORTANT:</b></u>
     * Call this method in your app's workflow after which DriverSDK can be started via backend-start
     * with ApplicationContext as parameter.
     *
     * @param context  Pass ApplicationContext as the parameter
     * @param driverID HyperTrack Driver Id for the current driver
     * @return Returns FALSE in the case another Driver is already active, TRUE otherwise.
     */
    public static boolean connectDriver(Context context, @NonNull String driverID) {
        return HTTransmitterService.getInstance(context).connectDriverToServer(context, driverID);
    }

    /**
     * Sets the Foreground Service Notification Intent to the notificationIntentClass provided
     *
     * @param context                 Pass context as parameter
     * @param notificationIntentClass Class which will be invoked on Foreground Service Notification Click
     * @return true in case of success
     */
    public abstract boolean setServiceNotificationIntentClass(Context context, Class notificationIntentClass);

    /**
     * Clears the Class Name for which the Intent was set on Foreground Service Notification
     *
     * @param context Pass context as parameter
     */
    public abstract void clearServiceNotificationIntentClass(Context context);

    /**
     * Sets the Foreground Service Notification Params as specified.
     * <p>
     * <u><b>IMPORTANT:</b></u>
     * These Params will be stored in Persistent Storage. So, in case of any change in
     * the settings, they should be either reset explicitly or clearServiceNotificationParams()
     * method should be called first.
     *
     * @param notificationParams NotificationParams object to be set.
     */
    public abstract boolean setServiceNotificationParams(ServiceNotificationParams notificationParams);

    /**
     * Clears the Foreground Service Notification Params, if any exists.
     */
    public abstract void clearServiceNotificationParams();

    /**
     * Check if the driver is currently active
     *
     * @return
     */
    public abstract boolean isDriverLive();

    /**
     * Check if given driver is connected to server to receive commands
     * @return
     */
    public abstract boolean isDriverConnected(String driverID);

    /**
     * Get currently active driverID, if any exists
     *
     * @return Returns string containing active DriverID, null otherwise.
     */
    public abstract String getActiveDriverID();

    /**
     * Get driverID, if any exists
     *
     * @return Returns string containing DriverID, null otherwise.
     */
    public abstract String getDriverID();

    /**
     * Starts the DriverSDK's LocationService which will initiate transmitting Driver's location.
     * <p>
     * <u><b>IMPORTANT:</b></u>
     * Call this API to start LocationService for a Trip/Shift already started on the backend,
     * otherwise the LocationService would stop when it syncs with the backend.
     *
     * @param driverID Pass driverID as first parameter
     * @param callback Pass HTStartDriverStatusCallback as second parameter.
     */
    public abstract void startLocationService(String driverID, final HTStartDriverStatusCallback callback);

    /**
     * Starts a trip.
     * <p>
     * Create a object of HTTripParams using HTTripParamsBuilder.
     * <p>
     * <u><b>IMPORTANT:</b></u>
     * In case TripID for a trip created on backend is provided in HTTripParams, startTrip API
     * starts this Trip on the server.
     *
     * @param htTripParams Pass HTTripParams as first parameter.
     * @param callback     Pass HTTripStatusCallback as second parameter.
     */
    public abstract void startTrip(HTTripParams htTripParams, HTTripStatusCallback callback);

    /**
     * Fetches trip data for a given tripID.
     *
     * @param tripID   Pass tripID for which tripData has to be fetched as first parameter
     * @param callback Pass HTTripStatusCallback as second parameter.
     */
    public abstract void refreshTripForTripID(String tripID, HTTripStatusCallback callback);

    /**
     * Ends a trip.
     *
     * @param tripID   Pass tripID to be ended as first parameter
     * @param callback Pass HTTripStatusCallback as second parameter.
     */
    public abstract void endTrip(final String tripID, final HTTripStatusCallback callback);

    /**
     * Ends all trips for a given driver.
     *
     * @param driverID Pass driverID for whom the trips have to be ended as first parameter.
     * @param callback Pass HTEndAllTripsCallback as second parameter.
     */
    public abstract void endAllTrips(String driverID, final HTEndAllTripsCallback callback);

    /**
     * Fetches task data.
     *
     * @param taskID   Pass taskID as first parameter.
     * @param callback Pass HTTaskStatusCallback as second parameter.
     */
    public abstract void refreshTask(final String taskID, final HTTaskStatusCallback callback);

    /**
     * Marks a task as complete.
     *
     * @param taskID   Pass Task ID as first parameter.
     * @param callback Pass HTCompleteTaskStatusCallback as second parameter.
     */
    public abstract void completeTask(String taskID, final HTCompleteTaskStatusCallback callback);

    /**
     * Makes a request to cancel a task.
     *
     * @param taskID   Pass Task ID as first parameter.
     * @param callback Pass HTTaskStatusCallback as second parameter.
     */
    public abstract void cancelTask(String taskID, final HTTaskStatusCallback callback);

    /**
     * Starts the shift for a driver
     *
     * @param htShiftParams Pass HTShiftParams as first parameter.
     * @param callback      Pass HTShiftStatusCallback as second parameter.
     */
    public abstract void startShift(HTShiftParams htShiftParams, HTShiftStatusCallback callback);

    /**
     * Ends the shift for a driver
     *
     * @param callback Pass HTShiftStatusCallback as first parameter.
     */
    public abstract void endShift(HTShiftStatusCallback callback);

    protected abstract void initializeHTTransmitter();

    protected abstract boolean connectDriverToServer(Context context, String driverID);

    protected TransmissionManager getTransmissionManager() {
        return transmissionManager;
    }

    protected HTJobScheduler getJobScheduler() {
        return jobScheduler;
    }

    /**
     * Method to check if Location Service is running, start it in case not
     */
    protected abstract void restartLocationServiceIfNotActive();

    /**
     * Method to start Location Service, in case not active
     */
    protected abstract void startLocationService();
}