package io.hypertrack.lib.transmitter.service;

import android.content.Context;
import android.content.SharedPreferences;
import android.location.Location;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.android.volley.VolleyError;
import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.location.LocationServices;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

import java.util.Date;

import io.hypertrack.lib.common.HyperTrack;
import io.hypertrack.lib.common.logs.PostDeviceLogsManager;
import io.hypertrack.lib.common.model.HTConstants;
import io.hypertrack.lib.common.model.HTLocation;
import io.hypertrack.lib.common.model.HTTask;
import io.hypertrack.lib.common.network.HTHttpClient;
import io.hypertrack.lib.common.network.HTNetworkRequest;
import io.hypertrack.lib.common.network.NetworkManager;
import io.hypertrack.lib.common.util.HTLog;
import io.hypertrack.lib.transmitter.BuildConfig;
import io.hypertrack.lib.transmitter.model.HTShift;
import io.hypertrack.lib.transmitter.model.HTShiftParams;
import io.hypertrack.lib.transmitter.model.callback.HTStartDriverStatusCallback;
import io.hypertrack.lib.transmitter.model.HTTrip;
import io.hypertrack.lib.transmitter.model.HTTripParams;
import io.hypertrack.lib.transmitter.model.ServiceNotificationParams;
import io.hypertrack.lib.transmitter.model.TransmitterConstants;
import io.hypertrack.lib.transmitter.model.callback.ErrorCallback;
import io.hypertrack.lib.transmitter.model.callback.HTCompleteTaskStatusCallback;
import io.hypertrack.lib.transmitter.model.callback.HTEndAllTripsCallback;
import io.hypertrack.lib.transmitter.model.callback.HTShiftStatusCallback;
import io.hypertrack.lib.transmitter.model.callback.HTTaskStatusCallback;
import io.hypertrack.lib.transmitter.model.callback.HTTripStatusCallback;

/*default*/ class HTTransmitterServiceImpl extends HTTransmitterService {

    private static class ExceptionMessages {
        private static final String StartTripCallbackRequiredMessage = "Required Parameter: 'callback' is required to start a trip and handle errors.";
        private static final String HTTripParamsRequiredMessage = "Required Parameter: 'htTripParams' is required to start a trip.";
        private static final String StartTripDriverIDRequiredMessage = "Required Parameter: 'driverID' is required to start a trip";

        private static final String AnotherDriverActiveMessage = "Cannot start tasks for another driver when a driver is active. End pending tasks/shift for the current driver before starting a new one.";
        private static final String RefreshTaskCallbackRequiredMessage = "Required Parameter: 'callback' is required to refresh a task and handle errors.";
        private static final String RefreshTaskNoTaskMessage = "Cannot refresh. No active task.";
        private static final String CancelTaskCallbackRequiredMessage = "Required Parameter: 'callback' is required to cancel a task and handle errors.";
        private static final String TaskIDRequiredForCancelMessage = "Required Parameter: 'taskID' is required to cancel a task";

        private static final String ContextRequiredMessage = "Context is required for this method. Please use HTTransmitterServiceImpl singleton constructor to call this method.";
        private static final String GoogleAPIClientConnectionFailedMessage = "GoogleAPIClient connection failed. Reason : ";
        private static final String GoogleAPIClientConnectionSuspendedMessage = "GoogleAPIClient connection suspended.";
        private static final String FusedLocationSecurityMessage = "Failed to get location from FusedLocationAPI. SecurityException.";
        private static final String RefreshTripCallbackRequiredMessage = "Required Parameter: 'callback' is required to refresh a trip and handle errors.";
        private static final String TripIDRequiredMessage = "Required Parameter: 'tripID' is required to refresh trip data";
        private static final String CompleteTripCallbackRequiredMessage = "Required Parameter: 'callback' is required to complete a trip and handle errors.";
        private static final String EndTripTripIDNotAvailableMessage = "Required Parameter: 'tripID' is required to end a trip";
        private static final String EndTripDriverIDRequiredMessage = "Required Parameter: 'driverID' is required to end all trips for a driver";

        private static final String StartShiftCallbackRequiredMessage = "Required Parameter: 'callback' is required to start a shift and handle errors.";
        private static final String HTShiftParamsRequiredMessage = "Required Parameter: 'htTripParams' is required to start a shift.";
        private static final String ShiftDriverIDRequiredMessage = "Required Parameter: 'driverID' is required to start a shift";
        private static final String EndShiftCallbackRequiredMessage = "Required Parameter: 'callback' is required to end a shift and handle errors.";
        private static final String EndShiftNoShiftMessage = "Cannot end shift. No active shift.";

        private static final String CompleteTaskCallbackRequiredMessage = "Required Parameter: 'callback' is required to complete a task and handle errors.";
        private static final String CompleteTaskEmptyTaskIDMessage = "Cannot complete task. Empty taskID.";
    }

    private abstract class GoogleAPIClientCallback extends ErrorCallback {
        public abstract void OnSuccess();
    }

    private final static String TAG = HTTransmitterServiceImpl.class.getSimpleName();
    private final static String VERSION_NAME = "TransmitterSDK/" + BuildConfig.VERSION_NAME;

    private static HTTransmitterServiceImpl sInstance;
    private GoogleApiClient mGoogleApiClient;
    private Context mContext;
    private ServiceHTTPClient mHTTPClient;
    private NetworkManager networkManager;

    public static HTTransmitterServiceImpl getInstance(Context context) {
        if (sInstance == null) {
            sInstance = new HTTransmitterServiceImpl(context.getApplicationContext());
        }

        return sInstance;
    }

    private HTTransmitterServiceImpl(Context context) {
        mContext = context;
        mHTTPClient = new ServiceHTTPClient(context, TAG);
        jobScheduler = new TransmitterJobScheduler(mContext);

        String driverID = TransmissionManager.getDriverID(mContext);
        networkManager = NetworkManager.getInstance(mContext, driverID);

        transmissionManager = new TransmissionManager(context, jobScheduler, networkManager);
    }

    @Override
    protected void initializeHTTransmitter() {
        // Initialize HyperTrack
        HyperTrack.initHyperTrack(mContext);

        // Check if LocationService is live and restart the service, if applicable
        HTTransmitterService transmitterService = HTTransmitterService.getInstance(mContext);
        if (transmitterService.isDriverLive()) {
            transmitterService.restartLocationServiceIfNotActive();

        } else {
            // Post DeviceLogs To Server
            if (HTHttpClient.isInternetConnected(mContext)) {
                if (PostDeviceLogsManager.getInstance(mContext).hasPendingDeviceLogs()) {
                    PostDeviceLogsManager.getInstance(mContext).postDeviceLogs(VERSION_NAME,
                            HTNetworkRequest.HTNetworkClient.HT_NETWORK_CLIENT_HTTP);
                }
            }

            // Check if PostDataGcmTask was active for a driverID
            String postDataDriverID = PostDataToServer.getDriverID(mContext);

            if (!TextUtils.isEmpty(postDataDriverID)) {
                // Check for any Pending data to be pushed to Server
                if (getTransmissionManager().hasPendingData(postDataDriverID)) {
                    // TransmissionManager has Pending Data, Initiate PostData PeriodicTask
                    getTransmissionManager().initPostDataPeriodicTask();
                }
            }
        }
    }

    @Override
    protected boolean connectDriverToServer(Context context, String driverID) {
        if (TextUtils.isEmpty(driverID))
            return false;

        if (!TextUtils.isEmpty(getActiveDriverID()) && !getActiveDriverID().equalsIgnoreCase(driverID)) {
            return false;
        }

        if (isDriverConnected(driverID)) {
            setDriverID(driverID);
            HTLog.i(TAG, "DriverID: " + driverID + " connected");
            return true;
        }

        HTLog.i(TAG, "connectDriver called for DriverID: " + driverID);
        setDriverID(driverID);

        // Initialize DriverSDK connection for the given driverID
        HTTransmitterService transmitterService = HTTransmitterService.getInstance(context);
        transmitterService.getTransmissionManager().subscribeToSDKControlsUpdate();
        return true;
    }

    @Override
    public boolean setServiceNotificationIntentClass(Context context, Class notificationIntentClass) {
        if (notificationIntentClass == null || TextUtils.isEmpty(notificationIntentClass.getName())) {
            return false;
        }

        SharedPreferences sharedpreferences = context.getSharedPreferences(
                HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.putString(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_INTENT_CLASS_NAME,
                notificationIntentClass.getName());
        editor.apply();

        return true;
    }

    @Override
    public void clearServiceNotificationIntentClass(Context context) {
        SharedPreferences sharedpreferences = context.getSharedPreferences(
                HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.remove(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_INTENT_CLASS_NAME);
        editor.apply();
    }

    @Override
    public boolean setServiceNotificationParams(ServiceNotificationParams notificationParams) {
        if (notificationParams == null) {
            return false;
        }

        // Set Service Notification Params to SharedPreferences
        setNotificationBuilder(notificationParams);

        // Update Service Notification in case Service is already running
        if (HTLocationService.htLocationService != null) {
            HTLocationService.htLocationService.updateForegroundNotification();
            return true;
        }

        return false;
    }

    @Override
    public void clearServiceNotificationParams() {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(
                HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.remove(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_LARGE_ICON_RES_ID);
        editor.remove(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_SMALL_ICON_RES_ID);
        editor.remove(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_SMALL_ICON_BG_COLOR);
        editor.remove(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_TITLE);
        editor.remove(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_TEXT);
        editor.remove(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_INTENT_CLASS_NAME);
        editor.remove(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_INTENT_EXTRAS);
        editor.remove(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_REMOTE_VIEWS);
        editor.remove(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_ACTION_LIST);
        editor.apply();
    }

    private void setNotificationBuilder(ServiceNotificationParams params) {
        try {
            Gson gson = new GsonBuilder().create();
            String remoteViews = gson.toJson(params.getContentView());

            SharedPreferences sharedpreferences = mContext.getSharedPreferences(
                    HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
            SharedPreferences.Editor editor = sharedpreferences.edit();
            editor.putInt(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_LARGE_ICON_RES_ID, params.getLargeIconResId());
            editor.putInt(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_SMALL_ICON_RES_ID, params.getSmallIconResId());
            editor.putInt(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_SMALL_ICON_BG_COLOR, params.getSmallIconBGColor());
            editor.putString(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_TITLE, params.getContentTitle());
            editor.putString(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_TEXT, params.getContentText());
            editor.putString(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_INTENT_CLASS_NAME,
                    params.getContentIntentActivityClassName());
            editor.putStringSet(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_INTENT_EXTRAS,
                    HTServiceNotificationUtils.getNotificationIntentExtras(params.getContentIntentExtras()));
            editor.putString(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_REMOTE_VIEWS, remoteViews);
            editor.putStringSet(TransmitterConstants.HT_SHARED_PREFERENCE_NOTIFICATION_ACTION_LIST,
                    HTServiceNotificationUtils.getNotificationActionListJSON(gson, params.getActionsList()));
            editor.apply();
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while setNotificationBuilder: " + e.getMessage());
        }
    }

    private void connectGoogleClient(final GoogleAPIClientCallback callback) {
        if (mGoogleApiClient != null && mGoogleApiClient.isConnected()) {
            if (callback != null) {
                callback.OnSuccess();
            }

            return;
        }

        if (mGoogleApiClient != null) {
            mGoogleApiClient.disconnect();
            mGoogleApiClient = null;
        }

        mGoogleApiClient = new GoogleApiClient.Builder(mContext)
                .addApi(LocationServices.API)
                .addConnectionCallbacks(new GoogleApiClient.ConnectionCallbacks() {
                    @Override
                    public void onConnected(@Nullable Bundle bundle) {
                        if (callback != null) {
                            callback.OnSuccess();
                        }
                    }

                    @Override
                    public void onConnectionSuspended(int i) {
                        String message = ExceptionMessages.GoogleAPIClientConnectionSuspendedMessage + "Status : " + i;
                        if (callback != null) {
                            callback.onError(new RuntimeException(message));
                        }
                    }
                })
                .addOnConnectionFailedListener(new GoogleApiClient.OnConnectionFailedListener() {
                    @Override
                    public void onConnectionFailed(@NonNull ConnectionResult connectionResult) {
                        String message = ExceptionMessages.GoogleAPIClientConnectionFailedMessage + connectionResult.toString();
                        if (callback != null) {
                            callback.onError(new RuntimeException(message));
                        }
                    }
                })
                .build();

        mGoogleApiClient.connect();
    }

    private Location getLastKnownLocation() throws SecurityException {
        try {
            Location location = LocationServices.FusedLocationApi.getLastLocation(mGoogleApiClient);
            return ValidationUtility.getValidatedLocation(location);

        } catch (SecurityException e) {
            throw e;
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while getLastKnownLocation: " + e);
        }

        return null;
    }

    // Method to get currently set DriverID
    @Override
    public String getDriverID() {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        return sharedpreferences.getString(TransmitterConstants.HT_SHARED_PREFERENCE_DRIVER_ID_KEY, null);
    }

    // Method to set DriverID
    private void setDriverID(String driverID) {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.putString(TransmitterConstants.HT_SHARED_PREFERENCE_DRIVER_ID_KEY, driverID);
        editor.apply();
    }

    protected void startLocationService() {
        try {
            if (HTLocationService.htLocationService == null) {

                HTLog.i(TAG, "Starting HTLocationService");
                // Initiate DriverActivity
                getTransmissionManager().startDriverActivity(true);
            }
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while startLocationService: " + e);
        }
    }

    protected void restartLocationServiceIfNotActive() {
        if (HTLocationService.htLocationService == null) {
            HTLog.w(TAG, "HTLocationService not active and Driver is Live");
            // Re-initiate DriverActivity
            getTransmissionManager().startDriverActivity(true);
        }
    }

    @Override
    public void startLocationService(String driverID, @NonNull final HTStartDriverStatusCallback callback) {
        if (mContext == null) {
            HTLog.e(TAG, ExceptionMessages.ContextRequiredMessage);
            callback.onError(new IllegalStateException(ExceptionMessages.ContextRequiredMessage));
            return;
        }

        Exception exception = ValidationUtility.getValidationException(mContext);
        if (exception != null) {
            HTLog.e(TAG, exception.toString());
            callback.onError(exception);
            return;
        }

        // Check if a driverID was provided in the params
        if (!TextUtils.isEmpty(driverID)) {

            // Check if LocationService was already running with a driverId different
            // to the one passed with these htTaskParams
            if (isDriverLive() && !TextUtils.isEmpty(this.getDriverID()) && !this.getDriverID().equalsIgnoreCase(driverID)) {
                HTLog.e(TAG, ExceptionMessages.AnotherDriverActiveMessage);
                callback.onError(new IllegalStateException(ExceptionMessages.AnotherDriverActiveMessage));
                return;
            }
        } else {
            // Set driverID if it exists in SharedPreferences
            driverID = this.getDriverID();
        }

        if (TextUtils.isEmpty(driverID)) {
            HTLog.e(TAG, ExceptionMessages.StartTripDriverIDRequiredMessage);
            callback.onError(new IllegalArgumentException(ExceptionMessages.StartTripDriverIDRequiredMessage));
            return;
        }

        HTLog.i(TAG, "Starting LocationService for a Driver: " + driverID);
        connectDriver(mContext, driverID);

        // Collect DeviceInfo in onStartDriverForTrip
        CollectDeviceInfo.collect(mContext);

        // Start Location Service to start collecting Locations
        startLocationService();
    }

    @Override
    public void startTrip(final HTTripParams htTripParams, final HTTripStatusCallback callback) throws IllegalArgumentException {
        if (callback == null) {
            HTLog.e(TAG, ExceptionMessages.StartTripCallbackRequiredMessage);
            throw new IllegalArgumentException(ExceptionMessages.StartTripCallbackRequiredMessage);
        }

        if (htTripParams == null) {
            HTLog.e(TAG, ExceptionMessages.HTTripParamsRequiredMessage);
            throw new IllegalArgumentException(ExceptionMessages.HTTripParamsRequiredMessage);
        }

        if (mContext == null) {
            HTLog.e(TAG, ExceptionMessages.ContextRequiredMessage);
            callback.onError(new IllegalStateException(ExceptionMessages.ContextRequiredMessage));
            return;
        }

        Exception exception = ValidationUtility.getValidationException(mContext);
        if (exception != null) {
            HTLog.e(TAG, exception.toString());
            callback.onError(exception);
            return;
        }

        String activeDriverID = this.getDriverID();

        // Check if a driverID was provided in the params
        if (!TextUtils.isEmpty(htTripParams.getDriverID())) {

            // Check if LocationService was already active with a driverId different
            // to the one passed with these htTripParams
            if (isDriverLive() && !TextUtils.isEmpty(activeDriverID) && !activeDriverID.equalsIgnoreCase(htTripParams.getDriverID())) {
                HTLog.e(TAG, ExceptionMessages.AnotherDriverActiveMessage);
                callback.onError(new IllegalStateException(ExceptionMessages.AnotherDriverActiveMessage));
                return;
            }
        } else {

            // Set driverID if it exists in SharedPreferences
            htTripParams.setDriverID(this.getDriverID());
        }

        HTLog.i(TAG, "Trip start initiated. Trip Params: " + htTripParams.toString());
        if (!TextUtils.isEmpty(htTripParams.getDriverID())) {
            // Set DriverID for LocationService
            connectDriver(mContext, htTripParams.getDriverID());
        }

        this.connectGoogleClient(new GoogleAPIClientCallback() {
            @Override
            public void OnSuccess() {
                onTripStartGoogleAPIClientConnect(htTripParams, callback);
            }

            @Override
            public void onError(Exception exception) {
                HTLog.e(TAG, exception.toString());
                callback.onError(exception);
            }
        });
    }

    private void onTripStartGoogleAPIClientConnect(final HTTripParams tripParams, final HTTripStatusCallback callback) {
        try {
            // Collect DeviceInfo in onTripStart attempted
            CollectDeviceInfo.collect(mContext);

            Location location = getLastKnownLocation();
            if (location != null) {
                tripParams.setStartLocation(new HTLocation(location));
            }

            getTransmissionManager().startTrip(tripParams, new ClientTripStatusCallback() {
                @Override
                public void onSuccess(boolean isOffline, HTTrip trip) {
                    onTripStart(isOffline, tripParams.getDriverID(), trip);
                    callback.onSuccess(isOffline, trip);
                }

                @Override
                public void onError(VolleyError error, Exception exception) {
                    HTLog.e(TAG, exception.toString());
                    callback.onError(exception);
                }
            });

        } catch (SecurityException exception) {
            HTLog.e(TAG, ExceptionMessages.FusedLocationSecurityMessage);
            callback.onError(new RuntimeException(ExceptionMessages.FusedLocationSecurityMessage));
        }
    }

    private void onTripStart(boolean isOffline, String driverID, HTTrip trip) {

        if (trip != null && trip.getDriverID() != null) {
            driverID = trip.getDriverID();
        }

        connectDriver(mContext, driverID);
        HTLog.i(TAG, "Trip started " + (isOffline ? "offline" : "online") + " for Driver ID: " + driverID);

        // Collect DeviceInfo in onStartDriverForTask attempted
        CollectDeviceInfo.collect(mContext);

        // Start Location Service to start collecting Locations
        startLocationService();
    }

    @Override
    public void refreshTripForTripID(String tripID, final HTTripStatusCallback callback) {
        if (TextUtils.isEmpty(tripID)) {
            HTLog.e(TAG, ExceptionMessages.TripIDRequiredMessage);
            callback.onError(new IllegalStateException(ExceptionMessages.TripIDRequiredMessage));
            return;
        }

        if (callback == null) {
            HTLog.e(TAG, ExceptionMessages.RefreshTripCallbackRequiredMessage);
            throw new IllegalArgumentException(ExceptionMessages.RefreshTripCallbackRequiredMessage);
        }

        this.mHTTPClient.refreshTrip(tripID, new ClientTripStatusCallback() {
            @Override
            public void onSuccess(boolean isOffline, HTTrip trip) {
                callback.onSuccess(isOffline, trip);
            }

            @Override
            public void onError(VolleyError error, Exception exception) {
                callback.onError(exception);
            }
        });
    }

    @Override
    public void endTrip(final String tripID, final HTTripStatusCallback callback) {
        if (callback == null) {
            HTLog.e(TAG, ExceptionMessages.CompleteTripCallbackRequiredMessage);
            throw new IllegalArgumentException(ExceptionMessages.CompleteTripCallbackRequiredMessage);
        }

        if (mContext == null) {
            HTLog.e(TAG, ExceptionMessages.ContextRequiredMessage);
            callback.onError(new IllegalStateException(ExceptionMessages.ContextRequiredMessage));
            return;
        }

        if (TextUtils.isEmpty(tripID)) {
            HTLog.e(TAG, ExceptionMessages.EndTripTripIDNotAvailableMessage);
            callback.onError(new IllegalArgumentException(ExceptionMessages.EndTripTripIDNotAvailableMessage));
            return;
        }

        Exception exception = ValidationUtility.getValidationException(mContext);
        if (exception != null) {
            HTLog.e(TAG, exception.toString());
            callback.onError(exception);
            return;
        }

        this.connectGoogleClient(new GoogleAPIClientCallback() {
            @Override
            public void OnSuccess() {
                onEndTripGoogleAPIClientConnect(tripID, callback);
            }

            @Override
            public void onError(Exception exception) {
                HTLog.e(TAG, exception.toString());
                callback.onError(exception);
            }
        });
    }

    private void onEndTripGoogleAPIClientConnect(final String tripID, final HTTripStatusCallback callback) {
        try {
            // Collect DeviceInfo in on completeTask attempted
            CollectDeviceInfo.collect(mContext);

            Location location = getLastKnownLocation();

            Date completionTime = new Date();
            final ClientEndTripCallback endTripCallback = new ClientEndTripCallback() {
                @Override
                public void onError(VolleyError error, Exception exception) {
                    if (callback != null) {
                        callback.onError(exception);
                    }
                }

                @Override
                public void onSuccess(boolean isOffline, HTTrip trip, boolean isDriverLive) {
                    if (callback != null) {
                        callback.onSuccess(isOffline, trip);
                    }
                }
            };

            HTLog.i(TAG, "EndTrip Initiated for tripID: " + tripID);
            getTransmissionManager().endTrip(tripID, completionTime, location, endTripCallback);

        } catch (SecurityException exception) {
            HTLog.e(TAG, ExceptionMessages.FusedLocationSecurityMessage);
            callback.onError(new RuntimeException(ExceptionMessages.FusedLocationSecurityMessage));
        }
    }

    @Override
    public void endAllTrips(String driverID, final HTEndAllTripsCallback callback) {
        if (callback == null) {
            HTLog.e(TAG, ExceptionMessages.CompleteTripCallbackRequiredMessage);
            throw new IllegalArgumentException(ExceptionMessages.CompleteTripCallbackRequiredMessage);
        }

        if (mContext == null) {
            HTLog.e(TAG, ExceptionMessages.ContextRequiredMessage);
            callback.onError(new IllegalStateException(ExceptionMessages.ContextRequiredMessage));
            return;
        }

        Exception exception = ValidationUtility.getValidationException(mContext);
        if (exception != null) {
            HTLog.e(TAG, exception.toString());
            callback.onError(exception);
            return;
        }

        // Check if a driverID was provided in the params
        if (TextUtils.isEmpty(driverID)) {
            // Set driverID if it exists in SharedPreferences
            driverID = this.getDriverID();
        }

        if (TextUtils.isEmpty(driverID)) {
            HTLog.e(TAG, ExceptionMessages.EndTripDriverIDRequiredMessage);
            callback.onError(new IllegalArgumentException(ExceptionMessages.EndTripDriverIDRequiredMessage));
            return;
        }

        final String finalDriverID = driverID;
        this.connectGoogleClient(new GoogleAPIClientCallback() {
            @Override
            public void OnSuccess() {
                onEndAllTripsGoogleAPIClientConnect(finalDriverID, callback);
            }

            @Override
            public void onError(Exception exception) {
                HTLog.e(TAG, exception.toString());
                callback.onError(exception);
            }
        });
    }

    private void onEndAllTripsGoogleAPIClientConnect(final String driverID, final HTEndAllTripsCallback callback) {
        try {
            // Collect DeviceInfo in on completeTask attempted
            CollectDeviceInfo.collect(mContext);

            Location location = getLastKnownLocation();

            HTLog.i(TAG, "endAllTrips Initiated for driverID: " + driverID);

            Date completionTime = new Date();
            this.mHTTPClient.endAllTrips(driverID, completionTime, location, new ClientEndAllTripsCallback() {
                @Override
                public void onSuccess() {
                    if (callback != null) {
                        callback.onSuccess();
                    }
                }

                @Override
                public void onError(VolleyError error, Exception exception) {
                    if (callback != null) {
                        callback.onError(exception);
                    }
                }
            });

        } catch (SecurityException exception) {
            HTLog.e(TAG, ExceptionMessages.FusedLocationSecurityMessage);
            callback.onError(new RuntimeException(ExceptionMessages.FusedLocationSecurityMessage));
        }
    }

    @Override
    public void refreshTask(final String taskID, final HTTaskStatusCallback callback) throws IllegalArgumentException {
        if (callback == null) {
            HTLog.e(TAG, ExceptionMessages.RefreshTaskCallbackRequiredMessage);
            throw new IllegalArgumentException(ExceptionMessages.RefreshTaskCallbackRequiredMessage);
        }

        if (taskID == null) {
            HTLog.e(TAG, ExceptionMessages.RefreshTaskNoTaskMessage);
            callback.onError(new IllegalStateException(ExceptionMessages.RefreshTaskNoTaskMessage));
            return;
        }

        this.mHTTPClient.refreshTask(taskID, new ClientTaskStatusCallback() {
            @Override
            public void onSuccess(boolean isOffline, HTTask task) {
                callback.onSuccess(isOffline, task);
            }

            @Override
            public void onError(VolleyError error, Exception exception) {
                callback.onError(exception);
            }
        });
    }

    @Override
    public void completeTask(final String taskID, final HTCompleteTaskStatusCallback callback) throws IllegalArgumentException {
        if (callback == null) {
            HTLog.e(TAG, ExceptionMessages.CompleteTaskCallbackRequiredMessage);
            throw new IllegalArgumentException(ExceptionMessages.CompleteTaskCallbackRequiredMessage);
        }

        if (TextUtils.isEmpty(taskID)) {
            HTLog.e(TAG, ExceptionMessages.CompleteTaskEmptyTaskIDMessage);
            callback.onError(new IllegalArgumentException(ExceptionMessages.CompleteTaskEmptyTaskIDMessage));
            return;
        }

        if (mContext == null) {
            HTLog.e(TAG, ExceptionMessages.ContextRequiredMessage);
            callback.onError(new IllegalStateException(ExceptionMessages.ContextRequiredMessage));
            return;
        }

        Exception exception = ValidationUtility.getValidationException(mContext);
        if (exception != null) {
            HTLog.e(TAG, exception.toString());
            callback.onError(exception);
            return;
        }

        this.connectGoogleClient(new GoogleAPIClientCallback() {
            @Override
            public void OnSuccess() {
                onCompleteTaskGoogleAPIClientConnect(taskID, callback);
            }

            @Override
            public void onError(Exception exception) {
                HTLog.e(TAG, exception.toString());
                callback.onError(exception);
            }
        });
    }

    private void onCompleteTaskGoogleAPIClientConnect(String taskID, final HTCompleteTaskStatusCallback callback) {
        try {
            // Collect DeviceInfo in on completeTask attempted
            CollectDeviceInfo.collect(mContext);

            Location location = getLastKnownLocation();

            Date completionTime = new Date();
            final ClientCompleteTaskCallback completeTaskCallback = new ClientCompleteTaskCallback() {
                @Override
                public void onError(VolleyError error, Exception exception) {
                    if (callback != null) {
                        callback.onError(exception);
                    }
                }

                @Override
                public void onSuccess(String taskID, boolean isDriverLive) {
                    if (callback != null) {
                        callback.onSuccess(taskID);
                    }
                }
            };

            HTLog.i(TAG, "CompleteTask Initiated for taskID: " + taskID);
            getTransmissionManager().completeTask(taskID, completionTime, location, completeTaskCallback);

        } catch (SecurityException exception) {
            HTLog.e(TAG, ExceptionMessages.FusedLocationSecurityMessage);
            callback.onError(new RuntimeException(ExceptionMessages.FusedLocationSecurityMessage));
        }
    }

    @Override
    public void cancelTask(final String taskID, final HTTaskStatusCallback callback) {
        if (callback == null) {
            HTLog.e(TAG, ExceptionMessages.CancelTaskCallbackRequiredMessage);
            throw new IllegalArgumentException(ExceptionMessages.CancelTaskCallbackRequiredMessage);
        }

        if (taskID == null) {
            HTLog.e(TAG, ExceptionMessages.TaskIDRequiredForCancelMessage);
            callback.onError(new IllegalStateException(ExceptionMessages.TaskIDRequiredForCancelMessage));
            return;
        }

        this.mHTTPClient.cancelTask(taskID, new ClientTaskStatusCallback() {
            @Override
            public void onSuccess(boolean isOffline, HTTask canceledTask) {
                callback.onSuccess(isOffline, canceledTask);
            }

            @Override
            public void onError(VolleyError error, Exception exception) {
                callback.onError(exception);
            }
        });
    }

    // Methods to start shift
    @Override
    public void startShift(final HTShiftParams htShiftParams, final HTShiftStatusCallback callback) throws IllegalArgumentException {
        if (callback == null) {
            HTLog.e(TAG, ExceptionMessages.StartShiftCallbackRequiredMessage);
            throw new IllegalArgumentException(ExceptionMessages.StartShiftCallbackRequiredMessage);
        }

        if (htShiftParams == null) {
            HTLog.e(TAG, ExceptionMessages.HTShiftParamsRequiredMessage);
            throw new IllegalArgumentException(ExceptionMessages.HTShiftParamsRequiredMessage);
        }

        if (mContext == null) {
            HTLog.e(TAG, ExceptionMessages.ContextRequiredMessage);
            callback.onError(new IllegalStateException(ExceptionMessages.ContextRequiredMessage));
            return;
        }

        Exception exception = ValidationUtility.getValidationException(mContext);
        if (exception != null) {
            HTLog.e(TAG, exception.toString());
            callback.onError(exception);
            return;
        }

        String activeDriverID = this.getDriverID();

        // Check if a driverID was provided in the params
        if (!TextUtils.isEmpty(htShiftParams.getDriverID())) {

            // Check if LocationService was already running with a driverId different
            // to the one passed with these htTaskParams
            if (isDriverLive() && !TextUtils.isEmpty(activeDriverID) && !activeDriverID.equalsIgnoreCase(htShiftParams.getDriverID())) {
                HTLog.e(TAG, ExceptionMessages.AnotherDriverActiveMessage);
                callback.onError(new IllegalStateException(ExceptionMessages.AnotherDriverActiveMessage));
                return;
            }
        } else {
            // Set driverID if it exists in SharedPreferences
            htShiftParams.setDriverID(this.getDriverID());
        }

        if (htShiftParams.getDriverID() == null) {
            HTLog.e(TAG, ExceptionMessages.ShiftDriverIDRequiredMessage);
            callback.onError(new IllegalArgumentException(ExceptionMessages.ShiftDriverIDRequiredMessage));
            return;
        }

        // Set DriverID
        HTLog.i(TAG, "Shift start initiated : Shift Params" + htShiftParams.toString());
        connectDriver(mContext, htShiftParams.getDriverID());

        this.connectGoogleClient(new GoogleAPIClientCallback() {
            @Override
            public void OnSuccess() {
                onShiftStartGoogleAPIClientConnect(htShiftParams, callback);
            }

            @Override
            public void onError(Exception exception) {
                HTLog.e(TAG, exception.toString());
                callback.onError(exception);
            }
        });
    }

    private void onShiftStartGoogleAPIClientConnect(HTShiftParams shiftParams, final HTShiftStatusCallback callback) {
        try {
            // Collect DeviceInfo in onShiftStart attempted
            CollectDeviceInfo.collect(mContext);

            Location location = getLastKnownLocation();
            if (location != null)
                shiftParams.setStartLocation(new HTLocation(location));

            this.mHTTPClient.startShift(shiftParams, new ClientShiftStatusCallback() {
                @Override
                public void onSuccess(HTShift shift) {
                    onShiftStart(shift);
                    callback.onSuccess(shift);
                }

                @Override
                public void onError(VolleyError error, Exception exception) {
                    HTLog.e(TAG, exception.toString());
                    callback.onError(exception);
                }
            });
        } catch (SecurityException exception) {
            HTLog.e(TAG, ExceptionMessages.FusedLocationSecurityMessage);
            callback.onError(new RuntimeException(ExceptionMessages.FusedLocationSecurityMessage));
        }
    }

    private void onShiftStart(HTShift shiftDetails) {
        if (shiftDetails != null) {

            if (!TextUtils.isEmpty(shiftDetails.getDriverID())) {
                connectDriver(mContext, shiftDetails.getDriverID());

            } else {
                HTLog.e(TAG, "Error occurred while onShiftStart: HTDriver or HTDriver.getId() is NULL");
            }

            // Add Started Shift to ActiveShiftID
            TransmissionManager.setActiveShiftID(mContext, shiftDetails.getId());

            // Start Location Service to start collecting Locations
            startLocationService();

            return;
        }

        HTLog.e(TAG, "Error occurred while onShiftStart: HTShift object is NULL");
    }

    @Override
    public void endShift(final HTShiftStatusCallback callback) {
        if (callback == null) {
            HTLog.e(TAG, ExceptionMessages.EndShiftCallbackRequiredMessage);
            throw new IllegalArgumentException(ExceptionMessages.EndShiftCallbackRequiredMessage);
        }

        final String shiftID = TransmissionManager.getActiveShiftID(mContext);
        if (TextUtils.isEmpty(shiftID)) {
            HTLog.e(TAG, ExceptionMessages.EndShiftNoShiftMessage);
            callback.onError(new IllegalStateException(ExceptionMessages.EndShiftNoShiftMessage));
            return;
        }

        if (mContext == null) {
            HTLog.e(TAG, ExceptionMessages.ContextRequiredMessage);
            callback.onError(new IllegalStateException(ExceptionMessages.ContextRequiredMessage));
            return;
        }

        Exception exception = ValidationUtility.getValidationException(mContext);
        if (exception != null) {
            HTLog.e(TAG, exception.toString());
            callback.onError(exception);
            return;
        }

        this.connectGoogleClient(new GoogleAPIClientCallback() {
            @Override
            public void OnSuccess() {
                onEndShiftGoogleAPIClient(shiftID, callback);
            }

            @Override
            public void onError(Exception exception) {
                HTLog.e(TAG, exception.toString());
                callback.onError(exception);
            }
        });
    }

    private void onEndShiftGoogleAPIClient(final String shiftID, HTShiftStatusCallback callback) {
        try {
            // Collect DeviceInfo in on endShift attempted
            CollectDeviceInfo.collect(mContext);

            Location location = getLastKnownLocation();
            onEndShiftLocationFetch(shiftID, location, callback);

        } catch (SecurityException exception) {
            HTLog.e(TAG, ExceptionMessages.FusedLocationSecurityMessage);
            callback.onError(new RuntimeException(ExceptionMessages.FusedLocationSecurityMessage));
        }
    }

    private void onEndShiftLocationFetch(final String shiftID, Location location, final HTShiftStatusCallback callback) {
        this.mHTTPClient.endShift(shiftID, location, new Date(), new ClientShiftStatusCallback() {
            @Override
            public void onSuccess(HTShift shift) {
                HTLog.i(TAG, "Shift ended successfully with HTShift: " + (shift != null ? shift : ("id =" + shiftID)));
                onShiftEnd(shift);

                if (callback != null) {
                    callback.onSuccess(shift);
                }
            }

            @Override
            public void onError(VolleyError error, Exception exception) {
                HTLog.e(TAG, exception.toString());

                if (callback != null) {
                    callback.onError(exception);
                }
            }
        });
    }

    private void onShiftEnd(HTShift htShift) {
        // Clear ended shift from SharedPreferences
        TransmissionManager.clearActiveShiftID(mContext);

        // If isDriverActive is FALSE, initiate stopDriverActivity
        if (htShift != null && !htShift.isDriverActive()) {
            getTransmissionManager().stopDriverActivity(mContext);
        }
    }

    @Override
    public boolean isDriverLive() {
        return TransmissionManager.getIsDriverLive(mContext);
    }

    @Override
    public boolean isDriverConnected(String driverID) {
        return !TextUtils.isEmpty(driverID) && driverID.equalsIgnoreCase(getDriverID())
                && networkManager.getHtMqttClient().isTopicSubscribed(TransmissionManager.getSDKControlsTopic(mContext));
    }

    @Override
    public String getActiveDriverID() {
        if (isDriverLive() && !TextUtils.isEmpty(getDriverID())) {
            return getDriverID();
        }

        return null;
    }
}