package io.hypertrack.lib.transmitter.service;

import android.Manifest;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.location.LocationManager;
import android.os.Build;
import android.provider.Settings;
import android.text.TextUtils;

import org.json.JSONException;
import org.json.JSONObject;

import io.hypertrack.lib.common.model.HTConstants;
import io.hypertrack.lib.common.util.HTLog;

/**
 * Created by piyush on 06/08/16.
 */
/** package */ class LocationState {

    private static final String TAG = "LocationState";

    private static final String HT_SHARED_PREFERENCE_LOCATION_ENABLED = "io.hypertrack.lib:SharedPreferenceLocationEnabled";
    private static final String HT_SHARED_PREFERENCE_LOCATION_PROVIDER = "io.hypertrack.lib:SharedPreferenceLocationProvider";
    private static final String HT_SHARED_PREFERENCE_LOCATION_ACCURACY = "io.hypertrack.lib:SharedPreferenceLocationAccuracy";
    private static final String HT_SHARED_PREFERENCE_LOCATION_PERMISSION = "io.hypertrack.lib:SharedPreferenceLocationPermission";

    private static final String KEY_LOCATION_ENABLED = "enabled";
    private static final String KEY_LOCATION_PROVIDER = "provider";
    private static final String KEY_LOCATION_ACCURACY = "accuracy";
    private static final String KEY_LOCATION_PERMISSION = "permission";

    private static final String LOCATION_PROVIDER_GPS = "gps";
    private static final String LOCATION_PROVIDER_NETWORK = "network";
    private static final String LOCATION_PROVIDER_PASSIVE = "passive";
    private static final String LOCATION_PROVIDER_DISABLED = "disabled";
    private static final String LOCATION_PROVIDER_INVALID = "invalid";

    private static final String LOCATION_ACCURACY_HIGH = "high";
    private static final String LOCATION_ACCURACY_MEDIUM = "medium";
    private static final String LOCATION_ACCURACY_LOW = "low";

    private Context mContext;

    public LocationState(Context mContext) {
        this.mContext = mContext;
    }

    public JSONObject getLocationInfo() {
        JSONObject jsonObject = new JSONObject();

        try {
            // Fetch Current LocationState Params
            boolean isLocationEnabled = isLocationEnabled(mContext);
            String locationProvider = getLocationProvider(mContext);
            String locationAccuracy = getLocationAccuracy(mContext);
            Boolean locationPermissionAvailable = locationPermissionAvailable(mContext);

            // Fetch Cached LocationState Params
            boolean savedIsLocationEnabled = getSavedLocationEnabled();
            String savedLocationProvider = getSavedLocationProvider();
            String savedLocationAccuracy = getSavedLocationAccuracy();
            Boolean savedLocationPermissionAvailable = getSavedLocationPermission();

            // Add only updated data to locationState JSON
            if (isLocationEnabled != savedIsLocationEnabled) {
                setLocationEnabled(isLocationEnabled);
                jsonObject.put(KEY_LOCATION_ENABLED, isLocationEnabled);
            }

            if (TextUtils.isEmpty(savedLocationProvider) || !savedLocationProvider.equalsIgnoreCase(locationProvider)) {
                setLocationProvider(locationProvider);
                jsonObject.put(KEY_LOCATION_PROVIDER, locationProvider);
            }

            if (locationAccuracy != null && (TextUtils.isEmpty(savedLocationAccuracy) || !savedLocationAccuracy.equalsIgnoreCase(locationAccuracy))) {
                setLocationAccuracy(locationAccuracy);
                jsonObject.put(KEY_LOCATION_ACCURACY, locationAccuracy);
            }

            if (locationPermissionAvailable != null && (locationPermissionAvailable != savedLocationPermissionAvailable)) {
                setLocationPermission(locationPermissionAvailable);
                jsonObject.put(KEY_LOCATION_PERMISSION, locationPermissionAvailable);
            }

        } catch (JSONException e) {
            e.printStackTrace();
            HTLog.w(TAG, "Exception occurred while getLocationInfo: " + e.getMessage());
        }

        return jsonObject.length() > 0 ? jsonObject : null;
    }

    private boolean isLocationEnabled(Context context) {
        int locationMode = 0;
        String locationProviders;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
            try {
                locationMode = Settings.Secure.getInt(context.getContentResolver(), Settings.Secure.LOCATION_MODE);

            } catch (Settings.SettingNotFoundException e) {
                e.printStackTrace();
                HTLog.w(TAG, "Exception occurred while detecting isLocationEnabled: " + e.getMessage());
            }

            return locationMode != Settings.Secure.LOCATION_MODE_OFF;

        } else {
            locationProviders = Settings.Secure.getString(context.getContentResolver(), Settings.Secure.LOCATION_PROVIDERS_ALLOWED);
            return !TextUtils.isEmpty(locationProviders);
        }
    }

    private String getLocationProvider(Context context) {
        LocationManager locationManager = (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);

        try {
            if (locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER)) {
                return LOCATION_PROVIDER_GPS;

            } else if (locationManager.isProviderEnabled(LocationManager.PASSIVE_PROVIDER)) {
                return LOCATION_PROVIDER_NETWORK;

            } else if (locationManager.isProviderEnabled(LocationManager.NETWORK_PROVIDER)) {
                return LOCATION_PROVIDER_PASSIVE;

            } else {
                return LOCATION_PROVIDER_DISABLED;
            }
        } catch (Exception e) {
            HTLog.w(TAG, "Exception occurred while detecting LocationProvider: " + e.getMessage());
            return LOCATION_PROVIDER_INVALID;
        }
    }

    private String getLocationAccuracy(Context context) {
        int locationMode;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
            try {
                locationMode = Settings.Secure.getInt(context.getContentResolver(), Settings.Secure.LOCATION_MODE);

            } catch (Settings.SettingNotFoundException e) {
                e.printStackTrace();
                HTLog.w(TAG, "Exception occurred while detecting isLocationEnabled: " + e.getMessage());
                return null;
            }

            switch (locationMode) {
                case Settings.Secure.LOCATION_MODE_HIGH_ACCURACY:
                    return LOCATION_ACCURACY_HIGH;

                case Settings.Secure.LOCATION_MODE_BATTERY_SAVING:
                    return LOCATION_ACCURACY_MEDIUM;

                case Settings.Secure.LOCATION_MODE_SENSORS_ONLY:
                    return LOCATION_ACCURACY_LOW;

                case Settings.Secure.LOCATION_MODE_OFF:
                default:
                    return null;
            }
        }

        return null;
    }

    private Boolean locationPermissionAvailable(Context context) {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
            return null;
        }

        return (context.checkSelfPermission(Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED);
    }

    // Methods for Caching & Fetching LocationEnabled flag
    private boolean getSavedLocationEnabled() {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        return sharedpreferences.getBoolean(HT_SHARED_PREFERENCE_LOCATION_ENABLED, false);
    }

    private void setLocationEnabled(boolean locationEnabled) {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.putBoolean(HT_SHARED_PREFERENCE_LOCATION_ENABLED, locationEnabled);
        editor.apply();
    }

    // Methods for Caching & Fetching LocationProvider
    private String getSavedLocationProvider() {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        return sharedpreferences.getString(HT_SHARED_PREFERENCE_LOCATION_PROVIDER, null);
    }

    private void setLocationProvider(String locationProvider) {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.putString(HT_SHARED_PREFERENCE_LOCATION_PROVIDER, locationProvider);
        editor.apply();
    }

    // Methods for Caching & Fetching LocationAccuracy
    private String getSavedLocationAccuracy() {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        return sharedpreferences.getString(HT_SHARED_PREFERENCE_LOCATION_ACCURACY, null);
    }

    private void setLocationAccuracy(String locationAccuracy) {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.putString(HT_SHARED_PREFERENCE_LOCATION_ACCURACY, locationAccuracy);
        editor.apply();
    }

    // Methods for Caching & Fetching LocationPermission flag
    private boolean getSavedLocationPermission() {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        return sharedpreferences.getBoolean(HT_SHARED_PREFERENCE_LOCATION_PERMISSION, false);
    }

    private void setLocationPermission(boolean locationPermission) {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.putBoolean(HT_SHARED_PREFERENCE_LOCATION_PERMISSION, locationPermission);
        editor.apply();
    }

    // Methods to clear cached LocationState Data
    public static void clearSavedLocationStateData(Context context) {
        LocationState.clearSavedLocationEnabled(context);
        LocationState.clearSavedLocationProvider(context);
        LocationState.clearSavedLocationAccuracy(context);
        LocationState.clearSavedLocationPermission(context);
    }

    private static void clearSavedLocationEnabled(Context context) {
        SharedPreferences sharedpreferences = context.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.remove(HT_SHARED_PREFERENCE_LOCATION_ENABLED);
        editor.apply();
    }

    private static void clearSavedLocationProvider(Context context) {
        SharedPreferences sharedpreferences = context.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.remove(HT_SHARED_PREFERENCE_LOCATION_PROVIDER);
        editor.apply();
    }

    private static void clearSavedLocationAccuracy(Context context) {
        SharedPreferences sharedpreferences = context.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.remove(HT_SHARED_PREFERENCE_LOCATION_ACCURACY);
        editor.apply();
    }

    private static void clearSavedLocationPermission(Context context) {
        SharedPreferences sharedpreferences = context.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.remove(HT_SHARED_PREFERENCE_LOCATION_PERMISSION);
        editor.apply();
    }
}
