package io.hypertrack.lib.transmitter.service;

import android.content.ContentValues;
import android.database.Cursor;
import android.database.DatabaseUtils;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteStatement;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;

import io.hypertrack.lib.common.util.HTLog;

/**
 * Created by ulhas on 18/05/16.
 */
/** package */ class LocationTable {

    private static final String TAG = LocationTable.class.getSimpleName();
    private static final int LOCATION_REQUEST_QUERY_LIMIT = 500;

    private static final String TABLE_LOCATION = "location";
    private static final String COLUMN_ID = "_id";
    private static final String COLUMN_TRIP_ID = "trip_id";
    private static final String COLUMN_DRIVER_ID = "driver_id";
    private static final String COLUMN_GPS_LOCATION = "gps_location";

    private static final String DATABASE_CREATE = "CREATE TABLE IF NOT EXISTS "
            + TABLE_LOCATION
            + " ("
            + COLUMN_ID + " INTEGER PRIMARY KEY AUTOINCREMENT, "
            + COLUMN_TRIP_ID + " TEXT, "
            + COLUMN_DRIVER_ID + " TEXT, "
            + COLUMN_GPS_LOCATION + " TEXT NOT NULL"
            + ");";

    private static final String BULK_INSERT_LOCATION_WITH_TRIP_ID_OR_DRIVER_ID = "INSERT INTO "
            + TABLE_LOCATION
            + " (" + COLUMN_TRIP_ID + ", " + COLUMN_DRIVER_ID + ", " + COLUMN_GPS_LOCATION + ")"
            + " VALUES (?,?,?);";

    private static final Gson gson = new Gson();
    private static final Type type = new TypeToken<GPSLocation>() {}.getType();

    public static void onCreate(SQLiteDatabase db) {
        if (db == null) {
            return;
        }

        try {
            db.execSQL(DATABASE_CREATE);
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "LocationTable: Exception occurred while onCreate: " + e);
        }
    }

    public static void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        if (db == null) {
            return;
        }

        try {
            db.execSQL("DROP TABLE IF EXISTS " + TABLE_LOCATION);
            onCreate(db);
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "LocationTable: Exception occurred while onUpgrade: " + e);
        }
    }

    public static long getCount(SQLiteDatabase db) {
        try {
            if (db == null) {
                return 0;
            }

            return DatabaseUtils.queryNumEntries(db, TABLE_LOCATION);
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "LocationTable: Exception occurred while getCount: " + e);
            return 0L;
        }
    }

    public static void addLocation(SQLiteDatabase db, GPSLocation location) {
        if (db == null || location == null) {
            return;
        }
        ContentValues contentValues = new ContentValues();

        String gpsLocationJson = gson.toJson(location);

        contentValues.put(COLUMN_GPS_LOCATION, gpsLocationJson);
        contentValues.put(COLUMN_TRIP_ID, location.getTripID());
        contentValues.put(COLUMN_DRIVER_ID, location.getDriverID());

        try {
            db.insert(TABLE_LOCATION, null, contentValues);
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "LocationTable: Exception occurred while addLocation: " + e.getMessage());
        }
    }

    public static void addLocations(SQLiteDatabase db, List<GPSLocation> locations) {
        if (db == null || locations == null || locations.isEmpty()) {
            return;
        }

        SQLiteStatement statement = db.compileStatement(BULK_INSERT_LOCATION_WITH_TRIP_ID_OR_DRIVER_ID);

        try {
            db.beginTransaction();
            for (GPSLocation location : locations) {
                String gpsLocationJson = gson.toJson(location);

                statement.clearBindings();
                statement.bindString(1, (location.getTripID() != null ? location.getTripID() : ""));
                statement.bindString(2, (location.getDriverID() != null ? location.getDriverID() : ""));
                statement.bindString(3, gpsLocationJson);
                statement.execute();
            }
            db.setTransactionSuccessful();
            db.endTransaction();
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "LocationTable: Exception occurred while addLocations: " + e.getMessage());
        }
    }

    public static void deleteLocations(SQLiteDatabase db, List<GPSLocation> locationsList) {
        if (db == null)
            return;

        StringBuilder builder = new StringBuilder();
        for (GPSLocation location : locationsList) {
            if (location != null && location.getId() > 0) {
                builder.append(location.getId())
                        .append(",");
            }
        }

        if (builder.length() == 0) {
            return;
        }

        try {
            String ids = builder.toString();
            ids = ids.substring(0, ids.length() - 1);

            String whereClause = COLUMN_ID +
                    " IN (" +
                    ids +
                    ")";

            db.delete(TABLE_LOCATION, whereClause, null);
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "LocationTable: Exception occurred while deleteLocations: " + e.getMessage());
        }
    }

    public static void deleteAllLocations(SQLiteDatabase db) {
        if (db == null) {
            return;
        }

        try {
            db.delete(TABLE_LOCATION, null, null);
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "LocationTable: Exception occurred while deleteAllLocations: " + e.getMessage());
        }
    }

    public static void deleteLocationsOtherThanTripID(SQLiteDatabase db, String tripID) {
        if (db == null || tripID == null) {
            return;
        }

        try {
            db.delete(TABLE_LOCATION, COLUMN_TRIP_ID + "<>?", new String[]{tripID});
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "LocationTable: Exception occurred while deleteLocationsOtherThanTripID: " + e.getMessage());
        }
    }

    public static void deleteLocationsOtherThanDriverID(SQLiteDatabase db, String driverID) {
        if (db == null || driverID == null) {
            return;
        }

        try {
            db.delete(TABLE_LOCATION, COLUMN_DRIVER_ID + "<>?", new String[]{driverID});
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "LocationTable: Exception occurred while deleteLocationsOtherThanDriverID: " + e.getMessage());
        }
    }

    public static List<GPSLocation> getLocationsWithTripID(SQLiteDatabase db, String tripID) {
        if (db == null || tripID == null) {
            return null;
        }

        Cursor cursor = db.query(TABLE_LOCATION, new String[]{COLUMN_GPS_LOCATION},
                COLUMN_TRIP_ID + "=?", new String[]{tripID}, null, null, null,
                String.valueOf(LOCATION_REQUEST_QUERY_LIMIT));

        if (cursor == null || cursor.isClosed()) {
            return null;
        }

        ArrayList<GPSLocation> locations = null;

        try {
            if (cursor.moveToFirst()) {
                locations = new ArrayList<>();

                do {
                    if (cursor.isClosed()) {
                        break;
                    }

                    GPSLocation location = gson.fromJson(cursor.getString(0), type);
                    locations.add(location);
                } while (cursor.moveToNext());
            }
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "LocationTable: Exception occurred while getLocationsWithTripID: " + e.getMessage());
        } finally {
            cursor.close();
        }

        return locations;
    }

    public static List<GPSLocation> getLocationsWithDriverID(SQLiteDatabase db, String driverID) {
        if (db == null || driverID == null) {
            return null;
        }

        Cursor cursor = db.query(TABLE_LOCATION, new String[]{COLUMN_ID, COLUMN_GPS_LOCATION},
                COLUMN_DRIVER_ID + "=?", new String[]{driverID}, null, null, null,
                String.valueOf(LOCATION_REQUEST_QUERY_LIMIT));

        if (cursor == null || cursor.isClosed()) {
            return null;
        }

        ArrayList<GPSLocation> locations = null;

        try {
            if (cursor.moveToFirst()) {
                locations = new ArrayList<>();

                do {
                    if (cursor.isClosed()) {
                        break;
                    }

                    GPSLocation location = gson.fromJson(cursor.getString(1), type);
                    // Get RowId for Location
                    Integer rowId = Integer.valueOf(cursor.getString(0));
                    location.setId(rowId != null ? rowId : -1);

                    locations.add(location);
                } while (cursor.moveToNext());
            }
        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "LocationTable: Exception occurred while getLocationsWithDriverID: " + e.getMessage());
        } finally {
            cursor.close();
        }

        return locations;
    }
}
