package io.hypertrack.lib.transmitter.service;

import android.content.Context;
import android.content.SharedPreferences;
import android.location.Location;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.reflect.TypeToken;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;

import io.hypertrack.lib.common.model.HTConstants;

/**
 * Created by piyush on 26/09/16.
 */
/** package */ class PendingCompleteTripList {

    private static final String PENDING_COMPLETE_TRIPS_PREFERENCE_KEY = "io.hypertrack.lib:SharedPreferencePendingCompleteTripList";

    private Context mContext;

    public PendingCompleteTripList(Context context) {
        this.mContext = context;
    }

    public boolean hasPendingCompleteTrips() {
        ArrayList<PendingCompleteTrip> pendingCompleteTrips = this.getPendingCompleteTrips();

        if (pendingCompleteTrips == null) {
            return false;
        }

        return !pendingCompleteTrips.isEmpty();
    }

    public void addPendingCompleteTrip(PendingCompleteTrip pendingCompleteTrip) {
        if (pendingCompleteTrip == null) {
            return;
        }

        ArrayList<PendingCompleteTrip> pendingCompleteTrips = this.getPendingCompleteTrips();

        if (pendingCompleteTrips == null) {
            pendingCompleteTrips = new ArrayList<>();
        }

        if (pendingCompleteTrips.contains(pendingCompleteTrip)) {
            return;
        }

        pendingCompleteTrips.add(pendingCompleteTrip);
        this.savePendingCompleteTrips(pendingCompleteTrips);
    }

    public void removePendingCompleteTrip(PendingCompleteTrip pendingCompleteTrip) {
        if (pendingCompleteTrip == null) {
            return;
        }

        ArrayList<PendingCompleteTrip> pendingCompleteTrips = this.getPendingCompleteTrips();

        if (pendingCompleteTrips == null) {
            return;
        }

        pendingCompleteTrips.remove(pendingCompleteTrip);
        this.savePendingCompleteTrips(pendingCompleteTrips);
    }

    private void savePendingCompleteTrips(List<PendingCompleteTrip> pendingCompleteTrips) {
        if (pendingCompleteTrips == null) {
            return;
        }

        if (pendingCompleteTrips.isEmpty()) {
            clearPendingCompleteTrips();
            return;
        }

        SharedPreferences preferences = this.getSharedPreferences();
        SharedPreferences.Editor editor = preferences.edit();

        Gson gson = new Gson();
        String endTripsJson = gson.toJson(pendingCompleteTrips);

        editor.putString(PENDING_COMPLETE_TRIPS_PREFERENCE_KEY, endTripsJson);
        editor.apply();
    }

    public ArrayList<PendingCompleteTrip> getPendingCompleteTrips() {
        SharedPreferences preferences = this.getSharedPreferences();
        String completeTripsJson = preferences.getString(PENDING_COMPLETE_TRIPS_PREFERENCE_KEY, null);

        if (completeTripsJson == null) {
            return null;
        }

        GsonBuilder gsonBuilder = new GsonBuilder();
        gsonBuilder.registerTypeAdapter(Location.class, new LocationDeserializer());
        gsonBuilder.registerTypeAdapter(Location.class, new LocationSerializer());
        Gson gson = gsonBuilder.create();
        Type type = new TypeToken<ArrayList<PendingCompleteTrip>>() {}.getType();

        return gson.fromJson(completeTripsJson, type);
    }

    public void clearPendingCompleteTrips() {
        SharedPreferences.Editor editor = this.getSharedPreferences().edit();

        editor.remove(PENDING_COMPLETE_TRIPS_PREFERENCE_KEY);
        editor.apply();
    }

    public static boolean haveAllTripsCompleted(List<PendingCompleteTrip> pendingCompleteTripList) {
        boolean hasCompleted = true;

        for (PendingCompleteTrip pendingCompleteTrip : pendingCompleteTripList) {
            if (pendingCompleteTrip.isCompleted() == null) {
                hasCompleted = false;
                break;
            }
        }

        return hasCompleted;
    }

    public static boolean haveAllTripsCompletedSuccessfully(List<PendingCompleteTrip> pendingCompleteTripList) {
        boolean hasCompletedSuccessfully = true;

        for (PendingCompleteTrip pendingCompleteTrip : pendingCompleteTripList) {
            if (pendingCompleteTrip.isCompleted() == null || !pendingCompleteTrip.isCompleted()) {
                hasCompletedSuccessfully = false;
                break;
            }
        }

        return hasCompletedSuccessfully;
    }

    public void clearPendingCompleteTrip(String tripID) {
        if (tripID == null) {
            return;
        }

        ArrayList<PendingCompleteTrip> pendingCompleteTrips = this.getPendingCompleteTrips();

        if (pendingCompleteTrips == null || pendingCompleteTrips.isEmpty()) {
            return;
        }

        PendingCompleteTrip pendingCompleteTrip = new PendingCompleteTrip(tripID);
        if (!pendingCompleteTrips.contains(pendingCompleteTrip)) {
            return;
        }

        pendingCompleteTrips.remove(pendingCompleteTrip);
        this.savePendingCompleteTrips(pendingCompleteTrips);
    }

    public void clearPendingTripsOtherThanID(String tripID) {
        if (tripID == null || tripID.isEmpty()) {
            return;
        }

        List<PendingCompleteTrip> pendingCompleteTripList = this.getPendingCompleteTrips();
        if (pendingCompleteTripList == null || pendingCompleteTripList.isEmpty()) {
            return;
        }

        ArrayList<PendingCompleteTrip> filteredPendingTrips = new ArrayList<>();
        for (PendingCompleteTrip pendingCompleteTrip : pendingCompleteTripList) {
            if (pendingCompleteTrip.getTripID().equalsIgnoreCase(tripID)) {
                filteredPendingTrips.add(pendingCompleteTrip);
            }
        }

        this.savePendingCompleteTrips(filteredPendingTrips);
    }

    private SharedPreferences getSharedPreferences() {
        return this.mContext.getSharedPreferences(HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
    }

    class LocationSerializer implements JsonSerializer<Location> {

        public JsonElement serialize(Location t, Type type,
                                     JsonSerializationContext jsc) {
            JsonObject jsonObject = new JsonObject();
            jsonObject.addProperty("mProvider", t.getProvider());
            jsonObject.addProperty("mAccuracy", t.getAccuracy());
            jsonObject.addProperty("mLatitude", t.getLatitude());
            jsonObject.addProperty("mLongitude", t.getLongitude());
            jsonObject.addProperty("mTime", t.getTime());
            jsonObject.addProperty("mSpeed", t.getSpeed());
            jsonObject.addProperty("mBearing", t.getBearing());
            return jsonObject;
        }
    }

    class LocationDeserializer implements JsonDeserializer<Location> {

        public Location deserialize(JsonElement je, Type type,
                                    JsonDeserializationContext jdc) {
            JsonObject jsonObject = je.getAsJsonObject();
            Location location = new Location(jsonObject.getAsJsonPrimitive("mProvider").getAsString());
            location.setAccuracy(jsonObject.getAsJsonPrimitive("mAccuracy").getAsFloat());
            location.setLatitude(jsonObject.getAsJsonPrimitive("mLatitude").getAsDouble());
            location.setLatitude(jsonObject.getAsJsonPrimitive("mLongitude").getAsDouble());
            location.setLatitude(jsonObject.getAsJsonPrimitive("mTime").getAsLong());
            location.setLatitude(jsonObject.getAsJsonPrimitive("mSpeed").getAsFloat());
            location.setLatitude(jsonObject.getAsJsonPrimitive("mBearing").getAsFloat());
            return location;
        }
    }
}
