package io.hypertrack.lib.transmitter.service;

import android.content.Context;
import android.content.SharedPreferences;
import android.text.TextUtils;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;

import io.hypertrack.lib.common.model.HTConstants;

/**
 * Created by piyush on 26/09/16.
 */
/** package */ class PendingStartTripList {
    private static final String PENDING_START_TRIPS_PREFERENCE_KEY = "io.hypertrack.lib:SharedPreferencePendingStartTripList";

    private Context mContext;

    public PendingStartTripList(Context context) {
        this.mContext = context;
    }

    public boolean hasPendingStartTrips() {
        ArrayList<PendingStartTrip> pendingStartTrips = this.getPendingStartTrips();

        if (pendingStartTrips == null) {
            return false;
        }

        return !pendingStartTrips.isEmpty();
    }

    public void addPendingStartTrip(PendingStartTrip pendingStartTrip) {
        if (pendingStartTrip == null) {
            return;
        }

        ArrayList<PendingStartTrip> pendingStartTrips = this.getPendingStartTrips();

        if (pendingStartTrips == null) {
            pendingStartTrips = new ArrayList<>();
        }

        if (pendingStartTrips.contains(pendingStartTrip)) {
            return;
        }

        pendingStartTrips.add(pendingStartTrip);
        this.savePendingStartTrips(pendingStartTrips);
    }

    public void removePendingStartTrip(PendingStartTrip startTripToBeRemoved) {
        if (startTripToBeRemoved == null) {
            return;
        }

        ArrayList<PendingStartTrip> pendingStartTrips = this.getPendingStartTrips();

        if (pendingStartTrips == null) {
            return;
        }

        pendingStartTrips.remove(startTripToBeRemoved);
        this.savePendingStartTrips(pendingStartTrips);
    }

    private void savePendingStartTrips(List<PendingStartTrip> pendingStartTrips) {
        if (pendingStartTrips == null) {
            return;
        }

        if (pendingStartTrips.isEmpty()) {
            clearPendingStartTrips();
            return;
        }

        SharedPreferences preferences = this.getSharedPreferences();
        SharedPreferences.Editor editor = preferences.edit();

        Gson gson = new Gson();
        String tripsJson = gson.toJson(pendingStartTrips);

        editor.putString(PENDING_START_TRIPS_PREFERENCE_KEY, tripsJson);
        editor.apply();
    }

    public ArrayList<PendingStartTrip> getPendingStartTrips() {
        SharedPreferences preferences = this.getSharedPreferences();
        String tripsJson = preferences.getString(PENDING_START_TRIPS_PREFERENCE_KEY, null);

        if (tripsJson == null) {
            return null;
        }

        Gson gson = new Gson();
        Type type = new TypeToken<ArrayList<PendingStartTrip>>(){}.getType();

        return gson.fromJson(tripsJson, type);
    }

    private SharedPreferences getSharedPreferences() {
        return this.mContext.getSharedPreferences(HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
    }

    private void clearPendingStartTrips() {
        SharedPreferences.Editor editor = this.getSharedPreferences().edit();

        editor.remove(PENDING_START_TRIPS_PREFERENCE_KEY);
        editor.apply();
    }

    public static boolean haveAllTripsStarted(List<PendingStartTrip> pendingStartTripList) {
        boolean hasStarted = true;

        for (PendingStartTrip pendingStartTrip : pendingStartTripList) {
            if (pendingStartTrip.isStarted() == null) {
                hasStarted = false;
                break;
            }
        }

        return hasStarted;
    }

    public static boolean haveAllTripsStartedSuccessfully(List<PendingStartTrip> pendingStartTripList) {
        boolean hasStartedSuccessfully = true;

        for (PendingStartTrip pendingStartTrip : pendingStartTripList) {
            if (pendingStartTrip.isStarted() == null || !pendingStartTrip.isStarted()) {
                hasStartedSuccessfully = false;
                break;
            }
        }

        return hasStartedSuccessfully;
    }

    public void clearPendingStartTrip(String driverID) {
        if (driverID == null) {
            return;
        }

        ArrayList<PendingStartTrip> pendingStartTripList = this.getPendingStartTrips();

        if (pendingStartTripList == null || pendingStartTripList.isEmpty()) {
            return;
        }

        PendingStartTrip pendingStartTrip = new PendingStartTrip(driverID);
        if (!pendingStartTripList.contains(pendingStartTrip)) {
            return;
        }

        pendingStartTripList.remove(pendingStartTrip);
        this.savePendingStartTrips(pendingStartTripList);
    }

    public void clearPendingStartTripsOtherThanID(String driverID) {
        if (TextUtils.isEmpty(driverID)) {
            return;
        }

        List<PendingStartTrip> pendingStartTripList = this.getPendingStartTrips();
        if (pendingStartTripList == null || pendingStartTripList.isEmpty()) {
            return;
        }

        ArrayList<PendingStartTrip> filteredPendingStartTrips = new ArrayList<>();
        for (PendingStartTrip pendingStartTrip : pendingStartTripList) {
            if (pendingStartTrip.getDriverID().equalsIgnoreCase(driverID)) {
                filteredPendingStartTrips.add(pendingStartTrip);
            }
        }

        this.savePendingStartTrips(filteredPendingStartTrips);
    }
}
