package io.hypertrack.lib.transmitter.service;

import android.content.Context;
import android.content.SharedPreferences;
import android.location.Location;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.reflect.TypeToken;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;

import io.hypertrack.lib.common.model.HTConstants;

/**
 * Created by ulhas on 10/05/16.
 */

/**
 * package
 */
class PendingTaskList {

    private static final String PENDING_TASKS_PREFERENCE_KEY = "io.hypertrack.lib:SharedPreferencePendingTaskList";

    private Context mContext;

    public PendingTaskList(Context context) {
        this.mContext = context;
    }

    public boolean hasPendingTasks() {
        ArrayList<PendingTask> pendingTasks = this.getTasks();

        if (pendingTasks == null) {
            return false;
        }

        return !pendingTasks.isEmpty();
    }

    public void addPendingTask(PendingTask pendingTask) {
        if (pendingTask == null) {
            return;
        }

        ArrayList<PendingTask> pendingTasks = this.getTasks();

        if (pendingTasks == null) {
            pendingTasks = new ArrayList<>();
        }

        if (pendingTasks.contains(pendingTask)) {
            return;
        }

        pendingTasks.add(pendingTask);
        this.savePendingTasks(pendingTasks);
    }

    public void removePendingTask(PendingTask pendingTask) {
        if (pendingTask == null) {
            return;
        }

        ArrayList<PendingTask> pendingTasks = this.getTasks();

        if (pendingTasks == null) {
            return;
        }

        pendingTasks.remove(pendingTask);
        this.savePendingTasks(pendingTasks);
    }

    private void savePendingTasks(List<PendingTask> pendingTasks) {
        if (pendingTasks == null) {
            return;
        }

        if (pendingTasks.isEmpty()) {
            clearPendingTasks();
            return;
        }

        SharedPreferences preferences = this.getSharedPreferences();
        SharedPreferences.Editor editor = preferences.edit();

        Gson gson = new Gson();
        String tasksJson = gson.toJson(pendingTasks);

        editor.putString(PENDING_TASKS_PREFERENCE_KEY, tasksJson);
        editor.apply();
    }

    public ArrayList<PendingTask> getTasks() {
        SharedPreferences preferences = this.getSharedPreferences();
        String tasksJson = preferences.getString(PENDING_TASKS_PREFERENCE_KEY, null);

        if (tasksJson == null) {
            return null;
        }

        GsonBuilder gsonBuilder = new GsonBuilder();
        gsonBuilder.registerTypeAdapter(Location.class, new LocationDeserializer());
        gsonBuilder.registerTypeAdapter(Location.class, new LocationSerializer());
        Gson gson = gsonBuilder.create();
        Type type = new TypeToken<ArrayList<PendingTask>>() {
        }.getType();

        return gson.fromJson(tasksJson, type);
    }

    private SharedPreferences getSharedPreferences() {
        return this.mContext.getSharedPreferences(HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
    }

    public void clearPendingTasks() {
        SharedPreferences.Editor editor = this.getSharedPreferences().edit();

        editor.remove(PENDING_TASKS_PREFERENCE_KEY);
        editor.apply();
    }

    public static boolean haveAllTasksCompleted(List<PendingTask> taskList) {
        boolean hasCompleted = true;

        for (PendingTask task : taskList) {
            if (task.isCompleted() == null) {
                hasCompleted = false;
                break;
            }
        }

        return hasCompleted;
    }

    public static boolean haveAllTasksCompletedSuccessfully(List<PendingTask> taskList) {
        boolean hasCompletedSuccessfully = true;

        for (PendingTask task : taskList) {
            if (task.isCompleted() == null || !task.isCompleted()) {
                hasCompletedSuccessfully = false;
                break;
            }
        }

        return hasCompletedSuccessfully;
    }

    public void clearPendingTask(String taskID) {
        if (taskID == null) {
            return;
        }

        ArrayList<PendingTask> tasks = this.getTasks();

        if (tasks == null || tasks.isEmpty()) {
            return;
        }

        PendingTask task = new PendingTask(taskID);
        if (!tasks.contains(task)) {
            return;
        }

        tasks.remove(task);
        this.savePendingTasks(tasks);
    }

    public void clearPendingTasksOtherThanID(String tripID) {
        if (tripID == null || tripID.isEmpty()) {
            return;
        }

        List<PendingTask> pendingTasks = this.getTasks();
        if (pendingTasks == null || pendingTasks.isEmpty()) {
            return;
        }

        ArrayList<PendingTask> filteredPendingTasks = new ArrayList<>();
        for (PendingTask pendingTask : pendingTasks) {
            if (pendingTask.getTripID().equalsIgnoreCase(tripID)) {
                filteredPendingTasks.add(pendingTask);
            }
        }

        this.savePendingTasks(filteredPendingTasks);
    }

    class LocationSerializer implements JsonSerializer<Location> {

        public JsonElement serialize(Location t, Type type,
                                     JsonSerializationContext jsc) {
            JsonObject jsonObject = new JsonObject();
            jsonObject.addProperty("mProvider", t.getProvider());
            jsonObject.addProperty("mAccuracy", t.getAccuracy());
            jsonObject.addProperty("mLatitude", t.getLatitude());
            jsonObject.addProperty("mLongitude", t.getLongitude());
            jsonObject.addProperty("mTime", t.getTime());
            jsonObject.addProperty("mSpeed", t.getSpeed());
            jsonObject.addProperty("mBearing", t.getBearing());
            return jsonObject;
        }
    }

    class LocationDeserializer implements JsonDeserializer<Location> {

        public Location deserialize(JsonElement je, Type type,
                                    JsonDeserializationContext jdc) {
            JsonObject jsonObject = je.getAsJsonObject();
            Location location = new Location(jsonObject.getAsJsonPrimitive("mProvider").getAsString());
            location.setAccuracy(jsonObject.getAsJsonPrimitive("mAccuracy").getAsFloat());
            location.setLatitude(jsonObject.getAsJsonPrimitive("mLatitude").getAsDouble());
            location.setLatitude(jsonObject.getAsJsonPrimitive("mLongitude").getAsDouble());
            location.setLatitude(jsonObject.getAsJsonPrimitive("mTime").getAsLong());
            location.setLatitude(jsonObject.getAsJsonPrimitive("mSpeed").getAsFloat());
            location.setLatitude(jsonObject.getAsJsonPrimitive("mBearing").getAsFloat());
            return location;
        }
    }
}
