package io.hypertrack.lib.transmitter.service;

import android.content.Context;

import com.android.volley.VolleyError;

import org.json.JSONArray;

import java.util.List;

import io.hypertrack.lib.common.controls.SDKControls;
import io.hypertrack.lib.common.network.HTNetworkRequest.HTNetworkClient;
import io.hypertrack.lib.common.network.HTNetworkResponse;
import io.hypertrack.lib.common.network.HTPostNetworkRequest;
import io.hypertrack.lib.common.network.NetworkManager;
import io.hypertrack.lib.common.util.HTLog;
import io.hypertrack.lib.transmitter.BuildConfig;

/**
 * Created by piyush on 09/08/16.
 */
/** package */ class PostDeviceInfoManager {
    private final String TAG = PostDeviceInfoManager.class.getSimpleName();
    private static final String HEALTH_LOG_BASE_TOPIC = "DeviceHealth/";
    private final String VERSION_NAME = "TransmitterSDK/" + BuildConfig.VERSION_NAME;

    private Context mContext;
    private ServiceHTTPClient mHttpClient;
    private NetworkManager networkManager;

    public PostDeviceInfoManager(Context mContext, NetworkManager networkManager) {
        this.mContext = mContext;
        this.mHttpClient = new ServiceHTTPClient(mContext, TAG);
        this.networkManager = networkManager;
    }

    public boolean hasPendingDeviceInfo(String driverID) {
        final DeviceInfoList mDeviceInfoList = new DeviceInfoList(driverID, DeviceInfoDatabaseHelper.getInstance(mContext));

        final List<DeviceInfo> deviceInfoList = mDeviceInfoList.getDeviceInfoListWithDriverID(driverID);
        if (deviceInfoList == null || deviceInfoList.isEmpty()) {
            return false;
        }

        return true;
    }

    public void postDeviceInfo(final String driverID, final PostDeviceInfoCallback callback) {
        final DeviceInfoList mDeviceInfoList = new DeviceInfoList(driverID, DeviceInfoDatabaseHelper.getInstance(mContext));

        final List<DeviceInfo> deviceInfoList = mDeviceInfoList.getDeviceInfoListWithDriverID(driverID);
        if (deviceInfoList == null || deviceInfoList.isEmpty()) {
            if (callback != null)
                callback.onPostDeviceInfoSuccess(null);
            return;
        }

        final DeviceInfoRequestList requestList = new DeviceInfoRequestList(deviceInfoList);
        final List<DeviceInfoRequest> requests = requestList.getRequests();

        if (requests == null || requests.isEmpty()) {
            if (callback != null)
                callback.onPostDeviceInfoSuccess(null);
            return;
        }

        mHttpClient.cancelRequests(TAG);

        for (final DeviceInfoRequest request : requests) {
            final List<DeviceInfo> requestDeviceInfo = request.getDeviceInfo();

            JSONArray jsonArray = this.getJSONArray(requestDeviceInfo);
            if (jsonArray == null) {
                request.setCompleted(false);

                if (callback != null && requestList.haveAllLogsBeenPosted()) {
                    callback.onError(new RuntimeException("JsonArray for one of the requestDeviceInfo was null"));
                }
                return;
            }

            this.postDeviceInfo(driverID, jsonArray, new PostDeviceInfoCallback() {
                @Override
                public void onPostDeviceInfoSuccess(SDKControls controls) {
                    HTLog.i(TAG, "Pushed " + requestDeviceInfo.size() + " deviceInfo logs");

                    // Clear Pushed DeviceInfo from DeviceInfoDB
                    mDeviceInfoList.clearDeviceInfo(requestDeviceInfo);
                    request.setCompleted(true);
                    requestDeviceInfo.clear();

                   if (callback != null && requestList.haveAllLogsBeenPosted()) {

                        if (requestList.haveAllLogsBeenPostedSuccessfully()) {
                            callback.onPostDeviceInfoSuccess(controls);
                        } else {
                            callback.onError(new RuntimeException("All deviceInfo weren't posted successfully!"));
                        }
                    }
                }

                @Override
                public void onError(Exception error) {
                    request.setCompleted(false);
                    HTLog.e(TAG, "OnPostingDeviceInfo Failure. Count: " + requestDeviceInfo.size()
                            + " Exception: " + (error != null ? error : "null"));

                    if (callback != null && requestList.haveAllLogsBeenPosted()) {
                        if (error == null) {
                            error = new RuntimeException("All deviceInfo weren't posted successfully!");
                        }
                        callback.onError(error);
                    }
                }
            });
        }
    }

    private void postDeviceInfo(String driverID, JSONArray jsonArray, final PostDeviceInfoCallback callback) {

        // Get Topic for postLocations executeMqttPOST call
        String healthLogsTopic = io.hypertrack.lib.common.BuildConfig.MQTT_BASE_TOPIC + HEALTH_LOG_BASE_TOPIC + driverID;

        HTPostNetworkRequest<SDKControls> postNetworkRequest = new HTPostNetworkRequest<>(TAG, mContext, VERSION_NAME,
                healthLogsTopic, HTNetworkClient.HT_NETWORK_CLIENT_MQTT, jsonArray, SDKControls.class,
                new HTNetworkResponse.Listener<SDKControls>() {
                    @Override
                    public void onResponse(SDKControls sdkControls) {
                        if (callback != null) {
                            callback.onPostDeviceInfoSuccess(sdkControls);
                        }
                    }
                },
                new HTNetworkResponse.ErrorListener() {
                    @Override
                    public void onErrorResponse(VolleyError error, Exception exception) {
                        if (callback != null) {
                            callback.onError(exception);
                        }
                    }
                });

        networkManager.execute(mContext, postNetworkRequest);
    }

    private JSONArray getJSONArray(List<DeviceInfo> deviceInfoList) {
        JSONArray jsonArray = null;
        if (deviceInfoList != null) {
            jsonArray = new JSONArray();

            for (DeviceInfo deviceInfo : deviceInfoList) {
                jsonArray.put(deviceInfo.getDeviceInfoJson());
            }
        }

        return jsonArray;
    }
}
