package io.hypertrack.lib.transmitter.service;

import android.content.Context;

import com.android.volley.VolleyError;
import com.google.gson.Gson;

import org.json.JSONArray;
import org.json.JSONException;

import java.util.List;

import io.hypertrack.lib.common.network.HTGson;
import io.hypertrack.lib.common.network.HTNetworkRequest.HTNetworkClient;
import io.hypertrack.lib.common.network.HTNetworkResponse;
import io.hypertrack.lib.common.network.HTPostNetworkRequest;
import io.hypertrack.lib.common.network.NetworkManager;
import io.hypertrack.lib.common.util.HTLog;
import io.hypertrack.lib.transmitter.BuildConfig;

/**
 * Created by piyush on 07/08/16.
 */
/** package */ class PostLocationManager {

    private final String TAG = PostLocationManager.class.getSimpleName();
    private final String GPS_BULK_URL = "gps/bulk/";
    private final String GPS_LOG_BASE_TOPIC = "GPSLog/";
    private final String VERSION_NAME = "TransmitterSDK/" + BuildConfig.VERSION_NAME;

    private Context mContext;
    private ServiceHTTPClient mHttpClient;
    private NetworkManager networkManager;

    public PostLocationManager(Context mContext, NetworkManager networkManager) {
        this.mContext = mContext;
        mHttpClient = new ServiceHTTPClient(mContext, TAG);
        this.networkManager = networkManager;
    }

    public boolean hasPendingLocations(String driverID) {
        final GPSLocationList mLocationsList = new GPSLocationList(LocationsDatabaseHelper.getInstance(mContext));

        final List<GPSLocation> locations = mLocationsList.getLocationsListWithDriverID(driverID);
        if (locations == null || locations.isEmpty()) {
            return false;
        }

        return true;
    }

    public void postLocations(final String driverID, final PostLocationCallback callback) {

        final GPSLocationList mLocationsList = new GPSLocationList(LocationsDatabaseHelper.getInstance(mContext));

        final List<GPSLocation> locations = mLocationsList.getLocationsListWithDriverID(driverID);
        if (locations == null || locations.isEmpty()) {
            if (callback != null) {
                callback.onPostLocationSuccess();
            }
            return;
        }

        final GPSLogRequestList requestList = new GPSLogRequestList(locations);
        final List<GPSLogRequest> requests = requestList.getRequests();

        if (requests == null || requests.isEmpty()) {
            if (callback != null) {
                callback.onPostLocationSuccess();
            }
            return;
        }

        for (final GPSLogRequest request : requests) {
            final List<GPSLocation> requestLocations = request.getGpsLocations();

            JSONArray jsonArray = this.getJSONArray(requestLocations);
            if (jsonArray == null) {
                request.setCompleted(false);
                if (callback != null && requestList.haveAllLogsBeenPosted()) {
                    callback.onError(new RuntimeException("JsonArray for one of the requestLocations was null"));
                }
                return;
            }

            this.postLocation(driverID, jsonArray, new PostLocationCallback() {
                @Override
                public void onPostLocationSuccess() {
                    HTLog.i(TAG, "Pushed " + requestLocations.size() + " locations");

                    // Clear Pushed Locations from LocationsDB
                    mLocationsList.clearLocations(requestLocations);
                    request.setCompleted(true);
                    requestLocations.clear();

                    if (callback != null && requestList.haveAllLogsBeenPosted()) {

                        if (requestList.haveAllLogsBeenPostedSuccessfully()) {
                            callback.onPostLocationSuccess();
                        } else {
                            callback.onError(new RuntimeException("All locations weren't posted successfully!"));
                        }
                    }
                }

                @Override
                public void onError(Exception error) {
                    request.setCompleted(false);
                    HTLog.e(TAG, "OnPostingLocation Failure. Count: " + requestLocations.size()
                            + " Exception: " + (error != null ? error : "null"));

                    if (callback != null && requestList.haveAllLogsBeenPosted()) {
                        if (error == null) {
                            error = new RuntimeException("All locations weren't posted successfully!");
                        }
                        callback.onError(error);
                    }
                }
            });
        }
    }

    private void postLocation(String driverID, JSONArray jsonArray, final PostLocationCallback callback) {

        // Get Topic for postLocations executeMqttPOST call
        String gpsLogTopic = io.hypertrack.lib.common.BuildConfig.MQTT_BASE_TOPIC + GPS_LOG_BASE_TOPIC + driverID;

        HTPostNetworkRequest<JSONArray> postNetworkRequest = new HTPostNetworkRequest<>(TAG, mContext,
                VERSION_NAME, gpsLogTopic, HTNetworkClient.HT_NETWORK_CLIENT_MQTT, jsonArray, JSONArray.class,
                new HTNetworkResponse.Listener<JSONArray>() {
                    @Override
                    public void onResponse(JSONArray response) {
                        if (callback != null)
                            callback.onPostLocationSuccess();
                    }
                },
                new HTNetworkResponse.ErrorListener() {
                    @Override
                    public void onErrorResponse(VolleyError error, Exception exception) {
                        if (callback != null)
                            callback.onError(exception);
                    }
                });
        networkManager.execute(mContext, postNetworkRequest);
    }

    private JSONArray getJSONArray(List<GPSLocation> locations) {
        if (locations == null)
            return null;

        Gson gson = HTGson.gson();
        String jsonString = gson.toJson(locations);

        JSONArray jsonArray = null;
        try {
            jsonArray = new JSONArray(jsonString);
        } catch (JSONException e) {
            HTLog.e(TAG, "Exception while creating JSONObject");
        }

        return jsonArray;
    }
}