package io.hypertrack.lib.transmitter.service;

/**
 * Created by piyush on 06/08/16.
 */

import android.content.Context;
import android.content.SharedPreferences;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Build;
import android.telephony.CellInfo;
import android.telephony.CellInfoCdma;
import android.telephony.CellInfoGsm;
import android.telephony.CellInfoLte;
import android.telephony.CellInfoWcdma;
import android.telephony.CellSignalStrengthCdma;
import android.telephony.CellSignalStrengthGsm;
import android.telephony.CellSignalStrengthLte;
import android.telephony.CellSignalStrengthWcdma;
import android.telephony.TelephonyManager;
import android.text.TextUtils;

import org.json.JSONException;
import org.json.JSONObject;

import io.hypertrack.lib.common.model.HTConstants;
import io.hypertrack.lib.common.util.HTLog;

/** package*/ class RadioState {

    private static final String TAG = "RadioState";

    private static final String HT_SHARED_PREFERENCE_SIGNAL_STRENGTH = "io.hypertrack.lib:SharedPreferenceSignalStrength";
    private static final String HT_SHARED_PREFERENCE_NETWORK_TYPE = "io.hypertrack.lib:SharedPreferenceNetworkType";
    private static final String HT_SHARED_PREFERENCE_NETWORK_STATE = "io.hypertrack.lib:SharedPreferenceNetworkState";

    private static final String KEY_NETWORK_TYPE = "network";
    private static final String KEY_NETWORK_STATE = "state";
    private static final String KEY_MOBILE_NETWORK_SIGNAL_STRENGTH = "signal_strength";

    private static final String NETWORK_TYPE_NONE = "NONE";
    private static final String NETWORK_TYPE_WIFI = "WIFI";
    private static final String NETWORK_TYPE_OTHER = "OTHER";

    private static final String NETWORK_TYPE_GPRS = "GPRS";
    private static final String NETWORK_TYPE_EDGE = "EDGE";
    private static final String NETWORK_TYPE_CDMA = "CDMA";
    private static final String NETWORK_TYPE_1xRTT = "1XRTT";
    private static final String NETWORK_TYPE_IDEN = "IDEN";

    private static final String NETWORK_TYPE_UMTS = "UMTS";
    private static final String NETWORK_TYPE_EVDO_0 = "EVDO_0";
    private static final String NETWORK_TYPE_EVDO_A = "EVDO_A";
    private static final String NETWORK_TYPE_HSDPA = "HSDPA";
    private static final String NETWORK_TYPE_HSUPA = "HSUPA";
    private static final String NETWORK_TYPE_HSPA = "HSPA";
    private static final String NETWORK_TYPE_EVDO_B = "EVDO_B";
    private static final String NETWORK_TYPE_EHRPD = "EHRPD";
    private static final String NETWORK_TYPE_HSPAP = "HSPAP";

    private static final String NETWORK_TYPE_LTE = "LTE";
    private static final String NETWORK_TYPE_UNKNOWN = "UNKNOWN";
    private static final String NETWORK_TYPE_INVALID = "INVALID";

    private static final String NETWORK_STATE_DISCONNECTED = "DISCONNECTED";

    private Context mContext;

    public RadioState(Context mContext) {
        this.mContext = mContext;
    }

    public JSONObject getRadioInfo() {
        JSONObject jsonObject = new JSONObject();

        try {
            // Fetch Current RadioState Params
            Integer signalStrength = getSignalStrength(mContext);
            String networkType = getNetworkType(mContext);
            String networkState = getNetworkState(mContext);

            // Set NetworkState as Disconnected if null
            if (TextUtils.isEmpty(networkState))
                networkState = NETWORK_STATE_DISCONNECTED;

            // Fetch Cached RadioState Params
            Integer savedSignalStrength = getSavedSignalStrength();
            String savedNetworkType = getSavedNetworkType();
            String savedNetworkState = getSavedNetworkState();

            // Add only updated data to RadioState JSON
            if (TextUtils.isEmpty(savedNetworkType) || !savedNetworkType.equalsIgnoreCase(networkType)) {
                setNetworkType(networkType);
                jsonObject.put(KEY_NETWORK_TYPE, networkType);
            }

            if (TextUtils.isEmpty(savedNetworkState) || !savedNetworkState.equalsIgnoreCase(networkState)) {
                setNetworkState(networkState);
                jsonObject.put(KEY_NETWORK_STATE, networkState);
            }

            if (signalStrength != null && (savedSignalStrength == null || !savedSignalStrength.equals(signalStrength))) {
                setSignalStrength(signalStrength);
                jsonObject.put(KEY_MOBILE_NETWORK_SIGNAL_STRENGTH, signalStrength);
            }

        } catch (JSONException e) {
            e.printStackTrace();
            HTLog.w(TAG, "Exception occurred while getRadioInfo: " + e.getMessage());
        }

        return jsonObject.length() > 0 ? jsonObject : null;
    }

    private String getNetworkType(Context context) {
        ConnectivityManager connectivityManager = (ConnectivityManager)
                context.getSystemService(Context.CONNECTIVITY_SERVICE);

        NetworkInfo networkInfo = connectivityManager.getActiveNetworkInfo();

        if (networkInfo == null) {
            return NETWORK_TYPE_NONE;
        }

        switch (networkInfo.getType()) {
            case ConnectivityManager.TYPE_MOBILE:
                return getMobileNetworkType(context);

            case ConnectivityManager.TYPE_WIFI:
                return NETWORK_TYPE_WIFI;

            default:
                return NETWORK_TYPE_OTHER;
        }
    }

    private String getMobileNetworkType(Context context) {

        TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
        int networkType = telephonyManager.getNetworkType();

        switch (networkType) {
            case TelephonyManager.NETWORK_TYPE_UNKNOWN:
                return NETWORK_TYPE_UNKNOWN;

            case TelephonyManager.NETWORK_TYPE_GPRS:
                return NETWORK_TYPE_GPRS;

            case TelephonyManager.NETWORK_TYPE_EDGE:
                return NETWORK_TYPE_EDGE;

            case TelephonyManager.NETWORK_TYPE_CDMA:
                return NETWORK_TYPE_CDMA;

            case TelephonyManager.NETWORK_TYPE_1xRTT:
                return NETWORK_TYPE_1xRTT;

            case TelephonyManager.NETWORK_TYPE_IDEN:
                return NETWORK_TYPE_IDEN;

            case TelephonyManager.NETWORK_TYPE_UMTS:
                return NETWORK_TYPE_UMTS;

            case TelephonyManager.NETWORK_TYPE_EVDO_0:
                return NETWORK_TYPE_EVDO_0;

            case TelephonyManager.NETWORK_TYPE_EVDO_A:
                return NETWORK_TYPE_EVDO_A;

            case TelephonyManager.NETWORK_TYPE_HSDPA:
                return NETWORK_TYPE_HSDPA;

            case TelephonyManager.NETWORK_TYPE_HSUPA:
                return NETWORK_TYPE_HSUPA;

            case TelephonyManager.NETWORK_TYPE_HSPA:
                return NETWORK_TYPE_HSPA;

            case TelephonyManager.NETWORK_TYPE_EVDO_B:
                return NETWORK_TYPE_EVDO_B;

            case TelephonyManager.NETWORK_TYPE_EHRPD:
                return NETWORK_TYPE_EHRPD;

            case TelephonyManager.NETWORK_TYPE_HSPAP:
                return NETWORK_TYPE_HSPAP;

            case TelephonyManager.NETWORK_TYPE_LTE:
                return NETWORK_TYPE_LTE;

            default:
                return NETWORK_TYPE_INVALID;
        }
    }

    private String getNetworkState(Context context) {
        ConnectivityManager connectivityManager = (ConnectivityManager)
                context.getSystemService(Context.CONNECTIVITY_SERVICE);

        NetworkInfo networkInfo = connectivityManager.getActiveNetworkInfo();
        if (networkInfo == null || networkInfo.getState() == null || TextUtils.isEmpty(networkInfo.getState().name()))
            return null;

        return networkInfo.getState().name();
    }

    private Integer getSignalStrength(Context context) {

        // Check if Android API Version is 17 (JellyBeanMR1) or Above
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {

            try {
                final TelephonyManager tm = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
                if (tm != null && tm.getAllCellInfo() != null) {
                    for (final CellInfo info : tm.getAllCellInfo()) {

                        if (info == null)
                            continue;

                        if (info.isRegistered()) {
                            if (info instanceof CellInfoGsm) {
                                final CellSignalStrengthGsm gsm = ((CellInfoGsm) info).getCellSignalStrength();
                                return gsm != null ? gsm.getDbm() : null;

                            } else if (info instanceof CellInfoCdma) {
                                final CellSignalStrengthCdma cdma = ((CellInfoCdma) info).getCellSignalStrength();
                                return cdma != null ? cdma.getDbm() : null;

                            } else if (info instanceof CellInfoLte) {
                                final CellSignalStrengthLte lte = ((CellInfoLte) info).getCellSignalStrength();
                                return lte != null ? lte.getDbm() : null;

                            } else if (info instanceof CellInfoWcdma) {
                                final CellSignalStrengthWcdma wcdma = ((CellInfoWcdma) info).getCellSignalStrength();
                                return wcdma != null ? wcdma.getDbm() : null;
                            }
                        }
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        return null;
    }

    // Methods for Caching & Fetching SignalStrength
    private Integer getSavedSignalStrength() {
        try {
            SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
            String signalStrength = sharedpreferences.getString(HT_SHARED_PREFERENCE_SIGNAL_STRENGTH, null);

            if (TextUtils.isEmpty(signalStrength))
                return null;

            return Integer.valueOf(signalStrength);
        } catch (NumberFormatException e) {
            HTLog.e(TAG, "Exception occurred while getSavedSignalStrength: " + e);
        }

        return null;
    }

    private void setSignalStrength(int signalStrength) {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.putString(HT_SHARED_PREFERENCE_SIGNAL_STRENGTH, String.valueOf(signalStrength));
        editor.apply();
    }

    // Methods for Caching & Fetching SignalStrength
    private String getSavedNetworkType() {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        return sharedpreferences.getString(HT_SHARED_PREFERENCE_NETWORK_TYPE, null);
    }

    private void setNetworkType(String networkType) {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.putString(HT_SHARED_PREFERENCE_NETWORK_TYPE, networkType);
        editor.apply();
    }

    // Methods for Caching & Fetching NetworkState
    private String getSavedNetworkState() {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        return sharedpreferences.getString(HT_SHARED_PREFERENCE_NETWORK_STATE, null);
    }

    private void setNetworkState(String networkState) {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.putString(HT_SHARED_PREFERENCE_NETWORK_STATE, networkState);
        editor.apply();
    }

    // Methods to clear cached RadioState Data
    public static void clearSavedRadioStateData(Context context) {
        RadioState.clearSavedNetworkType(context);
        RadioState.clearSavedNetworkState(context);
        RadioState.clearSavedSignalStrength(context);
    }

    private static void clearSavedSignalStrength(Context context) {
        SharedPreferences sharedpreferences = context.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.remove(HT_SHARED_PREFERENCE_SIGNAL_STRENGTH);
        editor.apply();
    }

    private static void clearSavedNetworkType(Context context) {
        SharedPreferences sharedpreferences = context.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.remove(HT_SHARED_PREFERENCE_NETWORK_TYPE);
        editor.apply();
    }

    private static void clearSavedNetworkState(Context context) {
        SharedPreferences sharedpreferences = context.getSharedPreferences(HTConstants.HT_DEVICE_INFO_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.remove(HT_SHARED_PREFERENCE_NETWORK_STATE);
        editor.apply();
    }
}