package io.hypertrack.lib.transmitter.service;

/**
 * Created by Arjun on 31/05/16.
 */

import android.content.Context;
import android.content.SharedPreferences;
import android.util.Log;

import com.google.gson.Gson;

import io.hypertrack.lib.common.controls.ControlsManager;
import io.hypertrack.lib.common.controls.SDKControls;
import io.hypertrack.lib.common.util.HTLog;

/** package */ class SDKControlsManager extends ControlsManager<SDKControls> {
    private static final String TAG = SDKControlsManager.class.getSimpleName();
    private static final String SDK_CONTROLS_PREFERENCE_KEY = "io.hypertrack.lib:SharedPreferenceTripControls";

    /**
     * Default values for sdkcontrols.
     */
    private static final Integer LOCATION_REQUEST_MINIMUM_DURATION = 5; // seconds
    private static final Integer LOCATION_REQUEST_MINIMUM_DISPLACEMENT = 30; // meters
    private static final Integer BATCH_DURATION_INTERVAL = 60; // seconds

    private static final Integer OFFLINE_LOCATION_REQUEST_MINIMUM_DURATION = 8; // seconds
    private static final Integer OFFLINE_LOCATION_REQUEST_MINIMUM_DISPLACEMENT = 50; // meters
    private static final Integer OFFLINE_BATCH_DURATION_INTERVAL = 30; // seconds

    private Context mContext;

    public SDKControlsManager(Context context) {
        this.mContext = context;
    }

    public void setDriverActive(boolean isActive) {
        SDKControls controls = getSDKControls();

        // Mark driver as Active/Inactive
        if (controls != null) {
            controls.setActive(isActive);
        }

        // Save updated SDKControls
        SharedPreferences.Editor editor = this.getEditor();
        Gson gson = new Gson();
        String serializedControls = gson.toJson(controls);
        editor.putString(getControlsPreferenceKey(), serializedControls);
        editor.apply();
    }

    public void setSDKControls(SDKControls controls, final UpdateControlsCallback callback) {
        if (controls == null) {
            return;
        }

        SDKControls oldControls = getSavedControls(SDKControls.class);
        Log.d(TAG, "setSDKControls called with oldControls: " + oldControls + "and updatedControls: "
                + controls);

        SharedPreferences.Editor editor = this.getEditor();
        Gson gson = new Gson();
        String serializedControls = gson.toJson(controls);
        editor.putString(getControlsPreferenceKey(), serializedControls);
        editor.apply();

        // Check if Driver is active or not
        if (!controls.isActive() && callback != null) {
            // Driver is InActive on the Server, call stopDriverActivity on client
            callback.OnControlsDriverNotActive();
            return;
        }

        boolean restartPeriodicTask = false;
        boolean restartLocationService = false;

        if (oldControls != null) {
            if (!oldControls.getBatchDuration().equals(controls.getBatchDuration())
                    || !oldControls.getHealthDuration().equals(controls.getHealthDuration())) {
                restartPeriodicTask = true;
            }

            if (!oldControls.getMinimumDisplacement().equals(controls.getMinimumDisplacement()) ||
                    !oldControls.getMinimumDuration().equals(controls.getMinimumDuration())) {
                restartLocationService = true;
            }
        }

        if (callback != null) {
            callback.OnControlsUpdate(restartLocationService, restartPeriodicTask);
        }

        Log.d(TAG, "setSDKControls update, restartLocationService: " + restartLocationService
                + ", restartPeriodicTasks: " + restartPeriodicTask);
    }

    public SDKControls getSDKControls() {
        SDKControls savedControls = getSavedControls(SDKControls.class);

        if (savedControls == null) {
            SDKControls defaults = getDefaultControls();
            setSDKControls(defaults, null);
            return defaults;
        } else {
            return savedControls;
        }
    }

    public void setOfflineSDKControls(final UpdateControlsCallback callback) {
        // Override currentSDKControls with OfflineSDKControls
        SDKControls offlineSDKControls = getOfflineSDKControls(getSDKControls());
        setSDKControls(offlineSDKControls, callback);
    }

    private SDKControls getOfflineSDKControls(SDKControls currentSdkControls) {
        SDKControls offlineSDKControls = new SDKControls(currentSdkControls);

        // Update SDKControls to non-aggressive values
        try {
            // Check if minimumDisplacement for Location Collection is in aggressive mode
            if (currentSdkControls.getMinimumDisplacement() < OFFLINE_LOCATION_REQUEST_MINIMUM_DISPLACEMENT) {
                offlineSDKControls.setMinimumDisplacement(OFFLINE_LOCATION_REQUEST_MINIMUM_DISPLACEMENT);
            }

            // Check if minimumDuration for Location collection is in aggressive mode
            if (currentSdkControls.getMinimumDuration() < OFFLINE_LOCATION_REQUEST_MINIMUM_DURATION) {
                offlineSDKControls.setMinimumDuration(OFFLINE_LOCATION_REQUEST_MINIMUM_DURATION);
            }

            // Check if batchDuration to post data is in aggressive mode
            if (currentSdkControls.getBatchDuration() < OFFLINE_BATCH_DURATION_INTERVAL) {
                offlineSDKControls.setBatchDuration(OFFLINE_BATCH_DURATION_INTERVAL);
            }
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while getOfflineSDKControls: " + e);
        }

        return offlineSDKControls;
    }

    @Override
    public SDKControls getDefaultControls() {
        return new SDKControls("", BATCH_DURATION_INTERVAL, BATCH_DURATION_INTERVAL, LOCATION_REQUEST_MINIMUM_DURATION,
                LOCATION_REQUEST_MINIMUM_DISPLACEMENT);
    }

    @Override
    protected String getControlsPreferenceKey() {
        return SDK_CONTROLS_PREFERENCE_KEY;
    }

    @Override
    protected Context getContext() {
        return mContext;
    }
}
