package io.hypertrack.lib.transmitter.service;

import android.content.Context;
import android.location.Location;
import android.os.Build;
import android.text.TextUtils;

import com.android.volley.AuthFailureError;
import com.android.volley.NetworkResponse;
import com.android.volley.NoConnectionError;
import com.android.volley.ParseError;
import com.android.volley.Request;
import com.android.volley.Response;
import com.android.volley.VolleyError;
import com.android.volley.toolbox.HttpHeaderParser;
import com.android.volley.toolbox.JsonArrayRequest;
import com.google.gson.Gson;

import org.json.JSONArray;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.hypertrack.lib.common.HyperTrack;
import io.hypertrack.lib.common.controls.SDKControls;
import io.hypertrack.lib.common.exception.NoConnectionException;
import io.hypertrack.lib.common.exception.NoResponseException;
import io.hypertrack.lib.common.exception.ServerException;
import io.hypertrack.lib.common.model.BatteryState;
import io.hypertrack.lib.common.model.DeviceID;
import io.hypertrack.lib.common.model.HTTask;
import io.hypertrack.lib.common.network.HTCustomGetRequest;
import io.hypertrack.lib.common.network.HTCustomPostRequest;
import io.hypertrack.lib.common.network.HTGson;
import io.hypertrack.lib.common.network.HTHttpClient;
import io.hypertrack.lib.common.network.HTHttpStatusUtils;
import io.hypertrack.lib.common.util.DateTimeUtility;
import io.hypertrack.lib.common.util.HTLog;
import io.hypertrack.lib.transmitter.BuildConfig;
import io.hypertrack.lib.transmitter.model.HTShift;
import io.hypertrack.lib.transmitter.model.HTShiftParams;
import io.hypertrack.lib.transmitter.model.HTTrip;
import io.hypertrack.lib.transmitter.model.HTTripParams;

/**
 * Created by ulhas on 10/05/16.
 */

/** package */ class ServiceHTTPClient {

    private final String VERSION_NAME = "TransmitterSDK/" + BuildConfig.VERSION_NAME;

    private Context mContext;
    private HTHttpClient mHTTPClient;
    private String mTag;

    public ServiceHTTPClient(Context context, String tag) {
        this.mContext = context;
        this.mTag = tag;
        this.mHTTPClient = new HTHttpClient(context);
    }

    private void onError(VolleyError error, ApiErrorCallback callback) {
        if (error == null) {
            if (callback != null) {
                callback.onError(null, new NoResponseException("Something went wrong. Please try again."));
            }

            HTLog.e(this.mTag, "Something went wrong. Please try again.");
            return;
        }

        if (error instanceof NoConnectionError) {
            if (callback != null) {
                callback.onError(error, new NoConnectionException("We had some trouble connecting. Please try again in sometime."));
            }

            return;
        }

        if (error.networkResponse == null) {
            if (callback != null) {
                callback.onError(error, new NoResponseException("There was no response from the server. Please try again in sometime."));
            }

            return;
        }

        if (error.networkResponse.statusCode >= 500 && error.networkResponse.statusCode <= 599) {
            if (callback != null) {
                callback.onError(error, new ServerException(HTHttpStatusUtils.getMessage(error)));
            }

            HTLog.e(this.mTag, "Internal Server Error: We have been notified and try again later.");
            return;
        }

        HTLog.e(this.mTag, "Error occurred in onError in ServiceHTTPClient: " + error.networkResponse);

        Exception runtimeException = HTHttpStatusUtils.getException(error);
        HTLog.e(this.mTag, runtimeException.getMessage(), runtimeException);

        if (callback != null) {
            callback.onError(error, runtimeException);
        }
    }

    private Map<String, String> getRequestHeaders() {

        BatteryState batteryState = new BatteryState(this.mContext);
        final HashMap<String, String> additionalHeaders = new HashMap<>();
        additionalHeaders.putAll(batteryState.getBatteryHeader());

        Map<String, String> params = new HashMap<>();
        String token = HyperTrack.getPublishableKey(mContext);
        params.put("Authorization", "Token " + (token != null ? token : ""));
        params.put("User-Agent", "Hypertrack (Android " + Build.VERSION.RELEASE + ") TransmitterSDK/" + BuildConfig.VERSION_NAME);
        params.put("Device-Time", DateTimeUtility.getCurrentTime());
        params.put("DeviceID", DeviceID.getDeviceId(mContext));

        for (Map.Entry<String, String> header : additionalHeaders.entrySet()) {
            params.put(header.getKey(), header.getValue());
        }

        return params;
    }

    public void cancelRequests(String mTag) {
        if (mHTTPClient != null)
            mHTTPClient.cancelPendingRequests(mTag);
    }

    public void postLocations(String url, JSONArray jsonArray, final ClientPostLocationCallback callback) {
        if (url == null || jsonArray == null) {
            this.onError(null, callback);
            return;
        }

        JsonArrayRequest jsonArrayRequest = new JsonArrayRequest(Request.Method.POST,
                url, jsonArray,
                new Response.Listener<JSONArray>() {

                    @Override
                    public void onResponse(JSONArray response) {
                        onPostLocationSuccess(callback);
                    }
                },
                new Response.ErrorListener() {

                    @Override
                    public void onErrorResponse(VolleyError error) {
                        onPostLocationError(error, callback);
                    }
                }) {
            @Override
            public Map<String, String> getHeaders() throws AuthFailureError {
                return getRequestHeaders();
            }

            @Override
            protected Response<JSONArray> parseNetworkResponse(NetworkResponse response) {
                try {
                    return Response.success(new JSONArray("[]"), HttpHeaderParser.parseCacheHeaders(response));
                } catch (Exception e) {
                    return Response.error(new ParseError(e));
                }
            }
        };

        this.mHTTPClient.addToRequestQueue(jsonArrayRequest, this.mTag);
    }

    private void onPostLocationSuccess(final ClientPostLocationCallback callback) {
        HTLog.i(this.mTag, "Successfully posted location data");

        if (callback != null) {
            callback.onPostLocationSuccess();
        }
    }

    private void onPostLocationError(VolleyError error, final ClientPostLocationCallback callback) {
        HTLog.w(this.mTag, "Error posting location data");

        if (callback != null) {
            this.onError(error, callback);
        }
    }

    public void getSDKControls(String tag, String driverID, JSONArray jsonArray, final SDKControlsCallback controlsCallback) {
        if (TextUtils.isEmpty(driverID)) {
            HTLog.e(this.mTag, "'driverID' is required to fetch SdkControls");
            throw new IllegalArgumentException("Required Parameter: 'driverID' is required to fetch SdkControls");
        }

        if (controlsCallback == null) {
            HTLog.e(this.mTag, "'callback' is required to fetch SdkControls");
            throw new IllegalArgumentException("Required Parameter: 'callback' is required to fetch SdkControls");
        }

        String url = BuildConfig.BASE_URL + "drivers/" + driverID + "/health/";

        BatteryState batteryState = new BatteryState(mContext);
        DeviceID deviceID = new DeviceID();
        HashMap<String, String> additionalHeaders = new HashMap<>();
        additionalHeaders.putAll(batteryState.getBatteryHeader());
        additionalHeaders.putAll(deviceID.getDeviceHeader());

        String requestBody = null;
        try {
            requestBody = jsonArray != null ? jsonArray.toString() : null;
        } catch (OutOfMemoryError e) {
            e.printStackTrace();
        }

        HTCustomPostRequest<SDKControls> jsonObjReq = new HTCustomPostRequest<>(VERSION_NAME, url,
                additionalHeaders, requestBody, this.mContext, SDKControls.class,
                new Response.Listener<SDKControls>() {
                    @Override
                    public void onResponse(SDKControls response) {
                        controlsCallback.onSuccess(response);
                    }
                },
                new Response.ErrorListener() {
                    @Override
                    public void onErrorResponse(VolleyError error) {
                        controlsCallback.onError(error);
                    }
                });

        mHTTPClient.addToRequestQueue(jsonObjReq, tag);
    }

    public void startTrip(HTTripParams tripParams, final ClientTripStatusCallback callback) {
        String url = BuildConfig.BASE_URL + "trips/";

        // Start a trip for an already created trip_id
        if (!TextUtils.isEmpty(tripParams.getTripID())) {
            url = url + tripParams.getTripID() + "/start/";
        }

        HTLog.i(this.mTag, "Starting Trip with following details: " + tripParams.toString());

        Gson gson = HTGson.gson();
        final String requestBody = !TextUtils.isEmpty(gson.toJson(tripParams)) ? gson.toJson(tripParams) : null;

        BatteryState batteryState = new BatteryState(mContext);
        HashMap<String, String> additionalHeaders = batteryState.getBatteryHeader();

        HTCustomPostRequest<HTTrip> jsonObjReq = new HTCustomPostRequest<>(
                VERSION_NAME, url, additionalHeaders, requestBody, this.mContext, HTTrip.class,
                new Response.Listener<HTTrip>() {
                    @Override
                    public void onResponse(HTTrip response) {
                        if (callback != null) {
                            callback.onSuccess(false, response);
                        }
                    }
                },
                new Response.ErrorListener() {
                    @Override
                    public void onErrorResponse(VolleyError error) {
                        onError(error, callback);
                    }
                });

        mHTTPClient.addToRequestQueue(jsonObjReq, this.mTag);
    }

    public void refreshTrip(String tripID, final ClientTripStatusCallback callback) {
        // Cancel Pending Requests for current tripID
        cancelRequests(tripID);

        String url = BuildConfig.BASE_URL + "trips/" + tripID + "/";

        BatteryState batteryState = new BatteryState(mContext);
        HashMap<String, String> additionalHeaders = batteryState.getBatteryHeader();

        HTCustomGetRequest<HTTrip> jsonRequest = new HTCustomGetRequest<>(VERSION_NAME, url,
                additionalHeaders, null, this.mContext, HTTrip.class, new Response.Listener<HTTrip>() {
            @Override
            public void onResponse(HTTrip response) {
                if (callback != null) {
                    callback.onSuccess(false, response);
                }
            }
        },
                new Response.ErrorListener() {
                    @Override
                    public void onErrorResponse(VolleyError error) {
                        onError(error, callback);
                    }
                });

        mHTTPClient.addToRequestQueue(jsonRequest, tripID);
    }

    public void endTrip(final String tripID, final Date completionTime, final Location completionLocation,
                        final ClientEndTripCallback callback) {
        String url = BuildConfig.BASE_URL + "trips/" + tripID + "/end/";

        EndTripRequest endTripRequest = new EndTripRequest(completionLocation, completionTime);
        Gson gson = HTGson.gson();
        final String requestBody = !TextUtils.isEmpty(gson.toJson(endTripRequest)) ? gson.toJson(endTripRequest) : null;

        BatteryState batteryState = new BatteryState(mContext);
        HashMap<String, String> additionalHeaders = batteryState.getBatteryHeader();

        HTCustomPostRequest<HTTrip> jsonObjReq = new HTCustomPostRequest<>(
                VERSION_NAME, url, additionalHeaders, requestBody, this.mContext, HTTrip.class,
                new Response.Listener<HTTrip>() {

                    @Override
                    public void onResponse(HTTrip response) {
                        HTLog.i(mTag, "Trip ended successfully.");

                        boolean isDriverActive = true;
                        if (response != null && response.getIsDriverLive() != null)
                            isDriverActive = response.getIsDriverLive();

                        if (callback != null) {
                            callback.onSuccess(false, response, isDriverActive);
                        }
                    }

                },
                new Response.ErrorListener() {

                    @Override
                    public void onErrorResponse(VolleyError error) {
                        onError(error, callback);
                    }
                });

        this.mHTTPClient.addToRequestQueue(jsonObjReq, this.mTag);
    }

    public void endAllTrips(final String driverID, final Date completionTime, final Location completionLocation,
                            final ClientEndAllTripsCallback callback) {
        String url = BuildConfig.BASE_URL + "drivers/" + driverID + "/end_trip/";

        EndTripRequest endTripRequest = new EndTripRequest(completionLocation, completionTime);
        Gson gson = HTGson.gson();
        final String requestBody = !TextUtils.isEmpty(gson.toJson(endTripRequest)) ? gson.toJson(endTripRequest) : null;

        BatteryState batteryState = new BatteryState(mContext);
        HashMap<String, String> additionalHeaders = batteryState.getBatteryHeader();

        HTCustomPostRequest<Object> jsonObjReq = new HTCustomPostRequest<>(
                VERSION_NAME, url, additionalHeaders, requestBody, this.mContext, Object.class,
                new Response.Listener<Object>() {

                    @Override
                    public void onResponse(Object response) {
                        HTLog.i(mTag, "All Trips ended successfully for driver: " + driverID);
                        if (callback != null) {
                            callback.onSuccess();
                        }
                    }
                },
                new Response.ErrorListener() {

                    @Override
                    public void onErrorResponse(VolleyError error) {
                        onError(error, callback);
                    }
                });

        this.mHTTPClient.addToRequestQueue(jsonObjReq, this.mTag);
    }

    public void refreshTask(final String taskID, final ClientTaskStatusCallback callback) {
        // Cancel Pending Requests for current taskID
        cancelRequests(taskID);

        StringBuilder url = new StringBuilder(BuildConfig.BASE_URL + "tasks/expanded?id=" + taskID);

        BatteryState batteryState = new BatteryState(mContext);
        HashMap<String, String> additionalHeaders = batteryState.getBatteryHeader();

        HTCustomGetRequest<TaskListResponse> jsonRequest = new HTCustomGetRequest<>(
                VERSION_NAME, url.toString(), additionalHeaders, null, this.mContext, TaskListResponse.class,
                new Response.Listener<TaskListResponse>() {
                    @Override
                    public void onResponse(TaskListResponse response) {
                        List<HTTask> taskList = response.getTaskList();

                        if (taskList != null && taskList.get(0) != null) {
                            if (callback != null) {
                                callback.onSuccess(false, taskList.get(0));
                            }
                        } else {
                            onError(new VolleyError("Empty response received from server"),
                                    callback);
                        }
                    }
                },
                new Response.ErrorListener() {
                    @Override
                    public void onErrorResponse(VolleyError error) {
                        onError(error, callback);
                    }
                });

        mHTTPClient.addToRequestQueue(jsonRequest, taskID);
    }

    public void completeTask(final String taskID, final Date completionTime, Location location, final ClientCompleteTaskCallback callback) {
        String url = BuildConfig.BASE_URL + "tasks/" + taskID + "/completed/";

        TaskCompletionRequest taskCompletionRequest = new TaskCompletionRequest(location, completionTime);
        Gson gson = HTGson.gson();
        final String requestBody = !TextUtils.isEmpty(gson.toJson(taskCompletionRequest)) ? gson.toJson(taskCompletionRequest) : null;

        BatteryState batteryState = new BatteryState(mContext);
        HashMap<String, String> additionalHeaders = batteryState.getBatteryHeader();

        HTCustomPostRequest<HTTask> jsonObjReq = new HTCustomPostRequest<>(
                VERSION_NAME, url, additionalHeaders, requestBody, this.mContext, HTTask.class,
                new Response.Listener<HTTask>() {

                    @Override
                    public void onResponse(HTTask response) {
                        HTLog.i(mTag, "Task completed successfully.");

                        if (callback != null) {

                            boolean isDriverActive = true;
                            if (response != null && response.isDriverLive() != null)
                                isDriverActive = response.isDriverLive();

                            callback.onSuccess(taskID, isDriverActive);
                        }
                    }
                },
                new Response.ErrorListener() {

                    @Override
                    public void onErrorResponse(VolleyError error) {
                        onError(error, callback);
                    }
                });

        this.mHTTPClient.addToRequestQueue(jsonObjReq, this.mTag);
    }

    public void cancelTask(final String taskID, final ClientTaskStatusCallback callback) {
        String url = BuildConfig.BASE_URL + "tasks/" + taskID + "/canceled/";

        BatteryState batteryState = new BatteryState(mContext);
        HashMap<String, String> additionalHeaders = batteryState.getBatteryHeader();

        HTCustomGetRequest<TaskListResponse> jsonRequest = new HTCustomGetRequest<>(
                VERSION_NAME, url, additionalHeaders, null, this.mContext, TaskListResponse.class,
                new Response.Listener<TaskListResponse>() {
                    @Override
                    public void onResponse(TaskListResponse response) {
                        List<HTTask> taskList = response.getTaskList();

                        if (taskList != null && taskList.get(0) != null) {
                            if (callback != null) {
                                callback.onSuccess(false, taskList.get(0));
                            }
                        } else {
                            onError(new VolleyError("Empty response received from server"),
                                    callback);
                        }
                    }
                },
                new Response.ErrorListener() {
                    @Override
                    public void onErrorResponse(VolleyError error) {
                        onError(error, callback);
                    }
                });

        mHTTPClient.addToRequestQueue(jsonRequest, this.mTag);
    }

    public void startShift(HTShiftParams shiftParams, final ClientShiftStatusCallback callback) {
        String url = BuildConfig.BASE_URL + "shifts/";

        HTLog.i(this.mTag, "Starting Shift with following details: " + shiftParams.toString());

        Gson gson = HTGson.gson();
        final String requestBody = !TextUtils.isEmpty(gson.toJson(shiftParams)) ? gson.toJson(shiftParams) : null;

        BatteryState batteryState = new BatteryState(mContext);
        HashMap<String, String> additionalHeaders = batteryState.getBatteryHeader();

        HTCustomPostRequest<HTShift> jsonObjReq = new HTCustomPostRequest<HTShift>(
                VERSION_NAME, url, additionalHeaders, requestBody, this.mContext, HTShift.class,
                new Response.Listener<HTShift>() {

                    @Override
                    public void onResponse(HTShift response) {
                        HTLog.i(mTag, "Shift Response: " + response.toString());
                        if (callback != null) {
                            callback.onSuccess(response);
                        }
                    }
                },
                new Response.ErrorListener() {
                    @Override
                    public void onErrorResponse(VolleyError error) {
                        onError(error, callback);
                    }
                });

        mHTTPClient.addToRequestQueue(jsonObjReq, this.mTag);
    }

    public void endShift(final String shiftID, Location location, Date date, final ClientShiftStatusCallback callback) {
        EndTripRequest endTripRequest = new EndTripRequest(location, date);

        Gson gson = HTGson.gson();
        final String requestBody = !TextUtils.isEmpty(gson.toJson(endTripRequest)) ? gson.toJson(endTripRequest) : null;

        HTLog.i(this.mTag, "Ending shift: " + (requestBody != null ? requestBody : "null"));

        BatteryState batteryState = new BatteryState(mContext);
        HashMap<String, String> additionalHeaders = batteryState.getBatteryHeader();

        String url = BuildConfig.BASE_URL + "shifts/" + shiftID + "/end/";

        HTCustomPostRequest<HTShift> jsonObjReq = new HTCustomPostRequest<HTShift>(
                VERSION_NAME, url, additionalHeaders, requestBody, this.mContext, HTShift.class,
                new Response.Listener<HTShift>() {

                    @Override
                    public void onResponse(HTShift response) {
                        if (callback != null) {
                            callback.onSuccess(response);
                        }
                    }
                },
                new Response.ErrorListener() {
                    @Override
                    public void onErrorResponse(VolleyError error) {
                        onError(error, callback);
                    }
                });

        mHTTPClient.addToRequestQueue(jsonObjReq, this.mTag);
    }
}