package io.hypertrack.lib.transmitter.service;

import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.location.Location;
import android.support.annotation.NonNull;
import android.support.v4.content.LocalBroadcastManager;
import android.text.TextUtils;
import android.util.Log;

import com.android.volley.VolleyError;
import com.google.gson.Gson;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Date;

import io.hypertrack.lib.common.controls.SDKControls;
import io.hypertrack.lib.common.logs.PostDeviceLogsManager;
import io.hypertrack.lib.common.model.HTConstants;
import io.hypertrack.lib.common.network.HTGetNetworkRequest;
import io.hypertrack.lib.common.network.HTHttpClient;
import io.hypertrack.lib.common.network.HTNetworkRequest.HTNetworkClient;
import io.hypertrack.lib.common.network.HTNetworkResponse;
import io.hypertrack.lib.common.network.HTPostNetworkRequest;
import io.hypertrack.lib.common.network.MQTTLastWillTestament;
import io.hypertrack.lib.common.network.MQTTMessageArrivedCallback;
import io.hypertrack.lib.common.network.MQTTSubscriptionSuccessCallback;
import io.hypertrack.lib.common.network.NetworkManager;
import io.hypertrack.lib.common.scheduler.HTJob;
import io.hypertrack.lib.common.scheduler.HTJobScheduler;
import io.hypertrack.lib.common.util.HTLog;
import io.hypertrack.lib.transmitter.BuildConfig;
import io.hypertrack.lib.transmitter.model.HTTrip;
import io.hypertrack.lib.transmitter.model.HTTripParams;
import io.hypertrack.lib.transmitter.model.HTTripParamsBuilder;
import io.hypertrack.lib.transmitter.model.TransmitterConstants;

/**
 * Created by piyush on 08/08/16.
 */

/** package */ class TransmissionManager implements HTJobScheduler.HTJobScheduledCallback,
        MQTTMessageArrivedCallback, MQTTSubscriptionSuccessCallback {
    private static final String TAG = TransmissionManager.class.getSimpleName();
    private static final int ALARM_JOB_TIME_MULTIPLIER = 5;
    private static final int POST_DATA_FALLBACK_JOB_TIME_MULTIPLIER = 3;

    private final static String VERSION_NAME = "TransmitterSDK/" + BuildConfig.VERSION_NAME;
    private static final String LOCATION_SERVICE_ALARM_TAG = "io.hypertrack.lib:LocationServiceAlarm";
    private static final String POST_DATA_TAG = "io.hypertrack.lib:PostData";
    private static final String COLLECT_DEVICE_INFO_TAG = "io.hypertrack.lib:CollectDeviceInfo";

    private Context mContext;

    private HTJobScheduler mJobScheduler;
    private ServiceHTTPClient mHTTPClient;
    private NetworkManager mNetworkManager;

    private SDKControlsManager sdkControlsManager;
    private PostLocationManager postLocationManager;
    private PostDeviceInfoManager postDeviceInfoManager;

    private PendingTaskList mPendingTaskList;
    private PendingStartTripList mPendingStartTripList;
    private PendingCompleteTripList mPendingCompleteTripList;

    private TransmissionManager() {
    }

    public TransmissionManager(@NonNull final Context context, HTJobScheduler jobScheduler, NetworkManager networkManager) {
        mContext = context;
        mJobScheduler = jobScheduler;
        mNetworkManager = networkManager;

        initializeTransmissionManager();
    }

    private void initializeTransmissionManager() {
        if (sdkControlsManager == null)
            sdkControlsManager = new SDKControlsManager(mContext);

        if (postLocationManager == null) {
            postLocationManager = new PostLocationManager(mContext, mNetworkManager);
        }

        if (postDeviceInfoManager == null) {
            postDeviceInfoManager = new PostDeviceInfoManager(mContext, mNetworkManager);
        }

        if (mPendingTaskList == null) {
            mPendingTaskList = new PendingTaskList(mContext);
        }

        if (mPendingStartTripList == null) {
            mPendingStartTripList = new PendingStartTripList(mContext);
        }

        if (mPendingCompleteTripList == null) {
            mPendingCompleteTripList = new PendingCompleteTripList(mContext);
        }

        if (mHTTPClient == null) {
            mHTTPClient = new ServiceHTTPClient(mContext, TAG);
        }
    }

    public void startDriverActivity(boolean startLocation) {
        HTLog.i(TAG, "Driver Activity started");

        // Set isDriverLive to TRUE
        TransmissionManager.setIsDriverLive(mContext, true);

        // Initiate LocationService, if it needs to be started
        if (startLocation) {
            startLocationService();
            sdkControlsManager.setDriverActive(true);

            // Connect Driver, in case not connected yet
            HTTransmitterService.connectDriver(mContext, TransmissionManager.getDriverID(mContext));

            broadcastServiceStarted(mContext);
        }

        // Initiate Periodic Jobs, if they need to be started
        setupLocationServiceAlarmJob();
        initPeriodicTasks();
    }

    public void startLocationService() {
        initializeTransmissionManager();

        // Fetch updated SDKControls
        SDKControls sdkControls = sdkControlsManager.getSDKControls();

        // Start LocationService with SDKControls as parameter
        Intent locationServiceIntent = new Intent(mContext, HTLocationService.class);
        locationServiceIntent.putExtra(TransmitterConstants.HT_SDK_CONTROLS_KEY, sdkControls);
        mContext.startService(locationServiceIntent);
    }

    private void setupLocationServiceAlarmJob() {
        try {
            // Create a Periodic HTJob object with BatchDuration * Multiplier as the interval
            HTJob locationServiceAlarmJob = new HTJob.Builder(HTJob.JobID.LOCATION_SERVICE_ALARM,
                    HTJob.JOB_TYPE_ALARM, this, LOCATION_SERVICE_ALARM_TAG)
                    .setPeriodic(sdkControlsManager.getSDKControls().getBatchDuration() * ALARM_JOB_TIME_MULTIPLIER * 1000)
                    .build();

            // Schedule Periodic Job, if not scheduled already
            if (!mJobScheduler.contains(locationServiceAlarmJob))
                mJobScheduler.addJob(locationServiceAlarmJob);
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while setupLocationServiceAlarmJob: " + e);
        }
    }

    public void initPeriodicTasks() {
        initPostDataPeriodicTask();
        initCollectDeviceInfoPeriodicTask();
    }

    public void initPostDataPeriodicTask() {
        long repeatDuration = sdkControlsManager.getSDKControls().getBatchDuration();

        // Create a Periodic HTJob object with BatchDuration * Multiplier as the interval
        HTJob postDataPeriodicTaskJob = new HTJob.Builder(HTJob.JobID.POST_DATA_PERIODIC_TASK,
                this, POST_DATA_TAG)
                .setPeriodic(repeatDuration * 1000)
                .setRequiredNetworkType(HTJob.NETWORK_TYPE_CONNECTED)
                .build();

        // Schedule Periodic Job, if not scheduled already
        if (!mJobScheduler.contains(postDataPeriodicTaskJob))
            mJobScheduler.addJob(postDataPeriodicTaskJob);

        // Set PostDataPeriodicTask LastUpdatedTime
        PostDataToServer.setLastPostToServerTime(mContext);

        // Set DriverID (if not empty) for PostDataGcmTask
        String driverID = TransmissionManager.getDriverID(mContext);
        if (!TextUtils.isEmpty(driverID)) {
            TransmissionManager.setPostDataPeriodicTaskDriverID(mContext, driverID);
        }
    }

    public void initCollectDeviceInfoPeriodicTask() {
        long repeatDuration = sdkControlsManager.getSDKControls().getHealthDuration();

        // Create a Periodic HTJob object with BatchDuration * Multiplier as the interval
        HTJob collectDeviceHealthPeriodicTaskJob = new HTJob.Builder(HTJob.JobID.COLLECT_DEVICE_HEALTH_PERIODIC_TASK,
                HTJob.JOB_TYPE_PERIODIC_TASK, this, COLLECT_DEVICE_INFO_TAG)
                .setPeriodic(repeatDuration * 1000)
                .build();

        // Schedule Periodic Job, if not scheduled already
        if (!mJobScheduler.contains(collectDeviceHealthPeriodicTaskJob))
            mJobScheduler.addJob(collectDeviceHealthPeriodicTaskJob);
    }

    public void subscribeToSDKControlsUpdate() {
        HTGetNetworkRequest<SDKControls> subscribeRequest = new HTGetNetworkRequest<>(TAG, mContext,
                VERSION_NAME, getSDKControlsTopic(mContext), HTNetworkClient.HT_NETWORK_CLIENT_MQTT, SDKControls.class, null, null,
                this, this, TransmissionManager.getDriverID(mContext));

        mNetworkManager.execute(mContext, subscribeRequest);
    }

    private MQTTLastWillTestament getLastWillTestament() {
        String topic = io.hypertrack.lib.common.BuildConfig.MQTT_BASE_TOPIC + "DriverConnection/"
                + TransmissionManager.getDriverID(mContext);

        try {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("driver_id", TransmissionManager.getDriverID(mContext));
            jsonObject.put("is_connected", false);
            return new MQTTLastWillTestament(VERSION_NAME, topic, jsonObject, 1 , false);
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while MQTTLastWillTestament");
        }

        return null;
    }

    public static void removePostDataJob(Context context) {
        // Cancel all PostData job with TAG
        HTTransmitterService.getInstance(context).getJobScheduler().removeJob(HTJob.JobID.POST_DATA_PERIODIC_TASK);
        // Remove PostDataJob LastUpdatedTime
        PostDataToServer.clearLastPostToServerTime(context);
    }

    public static void removeCollectDeviceInfoJob(Context context) {
        // Cancel all CollectDeviceInfo job with TAG
        HTTransmitterService.getInstance(context).getJobScheduler().removeJob(HTJob.JobID.COLLECT_DEVICE_HEALTH_PERIODIC_TASK);
    }

    @Override
    public void onJobScheduled(Context context, HTJob job) {
        if (job == null)
            return;
        switch (job.getJobId()) {
            case HTJob.JobID.LOCATION_SERVICE_ALARM:
                Log.i(TAG, "LocationServiceAlarm job scheduled");
                onPostDataFallbackJobScheduled();
                break;

            case HTJob.JobID.COLLECT_DEVICE_HEALTH_PERIODIC_TASK:
                HTLog.i(TAG, "CollectDeviceHealth job scheduled");
                CollectDeviceInfo.collect(mContext);

                if (!TransmissionManager.getIsDriverLive(mContext)) {
                    // Driver is NOT Live, Remove CollectDeviceInfo PeriodicTask
                    TransmissionManager.removeCollectDeviceInfoJob(mContext);
                }
                break;

            case HTJob.JobID.POST_DATA_PERIODIC_TASK:
                Log.i(TAG, "PostData job scheduled");
                PostDataToServer.post(mContext);

                // Post DeviceLogs To Server
                if (PostDeviceLogsManager.getInstance(mContext).hasPendingDeviceLogs()) {
                    PostDeviceLogsManager.getInstance(mContext).postDeviceLogs(VERSION_NAME,
                            HTNetworkClient.HT_NETWORK_CLIENT_MQTT);
                }
                break;

            default:
                break;
        }
    }

    private void onPostDataFallbackJobScheduled() {
        // Check if LocationService is live and restart the service, if applicable
        HTTransmitterService transmitterService = HTTransmitterService.getInstance(mContext);
        if (transmitterService.isDriverLive()) {
            transmitterService.restartLocationServiceIfNotActive();
        }

        // Return if no Internet Connection available
        if (!HTHttpClient.isInternetConnected(mContext))
            return;

        Long periodicTaskLastUpdatedTime = PostDataToServer.getLastPostToServerTime(mContext);
        Long currentTime = System.currentTimeMillis();

        if (periodicTaskLastUpdatedTime != null) {
            if (((int) (currentTime - periodicTaskLastUpdatedTime) / 1000) >
                    (POST_DATA_FALLBACK_JOB_TIME_MULTIPLIER * sdkControlsManager.getSDKControls().getBatchDuration())) {

                HTLog.i(TAG, "PostData Fallback Job Initiated");

                // Post DriverData To Server
                PostDataToServer.post(mContext);

                // Post DeviceLogs To Server
                if (PostDeviceLogsManager.getInstance(mContext).hasPendingDeviceLogs()) {
                    PostDeviceLogsManager.getInstance(mContext).postDeviceLogs(VERSION_NAME,
                            HTNetworkClient.HT_NETWORK_CLIENT_MQTT);
                }
            }
        }
    }

    @Override
    public void onMessageArrived(String topic, String message) throws Exception {
        try {
            if (TextUtils.isEmpty(message)) {
                HTLog.e(TAG, "Error occurred while TransmissionManager.onMessageArrived: message is null");
                return;
            }

            // Handle SDKControls updated MQTT message
            if (topic.equalsIgnoreCase(getSDKControlsTopic(mContext))) {

                Log.d(TAG, "onMessageArrived called for topic: " + topic + ", message: " + message);
                Gson gson = new Gson();
                SDKControls updatedSDKControls = gson.fromJson(message, SDKControls.class);

                processSDKControlsUpdate(updatedSDKControls);
            }

        } catch (Exception e) {
            e.printStackTrace();
            HTLog.e(TAG, "Exception occurred while onSDKControlsUpdated: " + e);
        }
    }

    private void processSDKControlsUpdate(final SDKControls updatedSDKControls) {
        Log.d(TAG, "Calling processSDKControlsUpdate");

        sdkControlsManager.setSDKControls(updatedSDKControls, new UpdateControlsCallback() {
            @Override
            public void OnControlsUpdate(boolean restartLocation, boolean restartPeriodicTask) {
                HTLog.i(TAG, "onSDKControlsUpdated called with restartLocation: " + restartLocation
                        + ", restartPeriodicTask: " + restartPeriodicTask);

                // Re-initiate DriverActivity
                boolean startLocationService = (HTLocationService.htLocationService == null) || restartLocation;
                TransmissionManager.this.startDriverActivity(startLocationService);
            }

            @Override
            public void OnControlsDriverNotActive() {
                // Driver is Not Live on Server, Stopping Driver Activity
                stopDriverActivity(mContext);
            }
        });
    }

    @Override
    public void onMQTTSubscriptionSuccess(String topic) {
        // handle MQTTSubscription success here
        String subscriptionSuccessTopic = io.hypertrack.lib.common.BuildConfig.MQTT_BASE_TOPIC + "DriverConnection/"
                + TransmissionManager.getDriverID(mContext);

        try {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("driver_id", TransmissionManager.getDriverID(mContext));
            jsonObject.put("is_connected", true);

            HTPostNetworkRequest<JSONObject> subscriptionSuccessPublish = new HTPostNetworkRequest<>(
                    TAG, mContext, VERSION_NAME, subscriptionSuccessTopic,
                    HTNetworkClient.HT_NETWORK_CLIENT_MQTT, jsonObject, JSONObject.class, null,
                    new HTNetworkResponse.ErrorListener() {
                        @Override
                        public void onErrorResponse(VolleyError error, Exception exception) {
                            HTLog.e(TAG, "Error occurred while subscriptionSuccessPublish: " + exception);
                        }
                    });
            mNetworkManager.execute(mContext, subscriptionSuccessPublish);

            // Broadcast DriverConnection successful intent
            broadcastDriverConnectionSuccessful(TransmissionManager.getDriverID(mContext));
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while onMQTTSubscriptionSuccess");
        }
    }

    // Method to get/set isDriverLive
    public static boolean getIsDriverLive(Context context) {
        SharedPreferences sharedpreferences = context.getSharedPreferences(HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        return sharedpreferences.getBoolean(TransmitterConstants.HT_SHARED_PREFERENCES_IS_DRIVER_LIVE_KEY, false);
    }

    public static void setIsDriverLive(Context context, boolean isDriverLive) {
        SharedPreferences sharedpreferences = context.getSharedPreferences(HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.putBoolean(TransmitterConstants.HT_SHARED_PREFERENCES_IS_DRIVER_LIVE_KEY, isDriverLive);
        editor.apply();
    }

    private void setDriverID(String driverID) {
        SharedPreferences sharedpreferences = mContext.getSharedPreferences(HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.putString(TransmitterConstants.HT_SHARED_PREFERENCE_DRIVER_ID_KEY, driverID);
        editor.apply();
    }

    // Method to get current DriverID
    public static String getDriverID(Context context) {
        SharedPreferences sharedpreferences = context.getSharedPreferences(HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        return sharedpreferences.getString(TransmitterConstants.HT_SHARED_PREFERENCE_DRIVER_ID_KEY, null);
    }

    public static String getSDKControlsTopic(Context context) {
        return io.hypertrack.lib.common.BuildConfig.MQTT_BASE_TOPIC + "Push/SDKControls/"
                + TransmissionManager.getDriverID(context);
    }

    public static void setPostDataPeriodicTaskDriverID(Context context, String driverID) {
        SharedPreferences sharedpreferences = context.getSharedPreferences(HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.putString(TransmitterConstants.HT_SHARED_PREFERENCE_POST_DATA_TASK_DRIVER_ID_KEY, driverID);
        editor.apply();
    }

    public boolean hasPendingData(String driverID) {
        if (postLocationManager != null && postLocationManager.hasPendingLocations(driverID))
            return true;

        if (postDeviceInfoManager != null && postDeviceInfoManager.hasPendingDeviceInfo(driverID))
            return true;

        if (mPendingStartTripList != null && mPendingStartTripList.hasPendingStartTrips())
            return true;

        if (mPendingCompleteTripList != null && mPendingCompleteTripList.hasPendingCompleteTrips())
            return true;

        if (mPendingTaskList != null && mPendingTaskList.hasPendingTasks())
            return true;

        return false;
    }

    public void postDeviceInfoToServer(String driverID) {
        if (TextUtils.isEmpty(driverID)) {
            HTLog.e(TAG, "Required Parameter: 'driverID' is required to fetch SdkControls");
            return;
        }

        if (postDeviceInfoManager != null) {

            // Post DeviceInfo to Server and fetch SDKControls
            postDeviceInfoManager.postDeviceInfo(driverID, new PostDeviceInfoCallback() {
                @Override
                public void onPostDeviceInfoSuccess(SDKControls updatedSDKControls) {
                    if (updatedSDKControls == null)
                        return;

                    processSDKControlsUpdate(updatedSDKControls);
                }

                @Override
                public void onError(Exception exception) {
                }
            });
        }
    }

    public abstract class PostDataToServerCallback {
        public abstract void onPostDataToServerSuccess();
        public abstract void onError(Exception exception);
    }

    /**
     * Method to post Pending Task/Trip/Shift Data & Locations data to server
     */
    public void postDriverDataToServer(final String driverID) {
        // Post StartData to Server
        this.postStartDataToServer(new PostDataToServerCallback() {
            @Override
            public void onPostDataToServerSuccess() {

                //On Successful Post of StartData, Post Locations to Server
                TransmissionManager.this.postLocationsToServer(driverID, new PostLocationCallback() {
                    @Override
                    public void onPostLocationSuccess() {
                        // Post pending Complete requests to Server
                        TransmissionManager.this.postCompleteDataToServer();
                    }

                    @Override
                    public void onError(Exception exception) {
                    }
                });
            }

            @Override
            public void onError(Exception exception) {
            }
        });
    }

    private void postStartDataToServer(final PostDataToServerCallback callback) {

        // Post cached startTrip requests to Server, if any
        if (mPendingStartTripList != null && mPendingStartTripList.hasPendingStartTrips()) {
            HTLog.i(TAG, "Has pending start trips");

            // Post pending startTrip requests to Server
            this.startPendingTrips(new PendingStartTripCallback() {

                @Override
                public void onPendingTripStarted(boolean result) {

                    if (result) {
                        if (callback != null) {
                            callback.onPostDataToServerSuccess();
                        }
                    } else {
                        if (callback != null) {
                            callback.onError(new RuntimeException("Start Pending Trips Failed"));
                        }
                    }
                }
            });

            // Return success callback if no cached start requests
        } else if (callback != null) {
            callback.onPostDataToServerSuccess();
        }
    }

    private void postLocationsToServer(final String driverID, final PostLocationCallback callback) {
        // Check if entire StartData has been successfully posted to server
        if (postLocationManager != null) {

            // Post locations to Server
            postLocationManager.postLocations(driverID, new PostLocationCallback() {

                @Override
                public void onPostLocationSuccess() {
                    if (callback != null) {
                        callback.onPostLocationSuccess();
                    }
                }

                @Override
                public void onError(Exception exception) {
                    if (callback != null) {
                        callback.onError(exception);
                    }
                }
            });
        } else {
            if (callback != null) {
                callback.onError(new RuntimeException("PostLocationManager is null"));
            }
        }
    }

    private void postCompleteDataToServer() {

        // Post cached completeTask requests to Server after locations have been posted successfully
        if (mPendingTaskList != null && mPendingTaskList.hasPendingTasks()) {
            HTLog.i(TAG, "Has pending complete tasks");
            TransmissionManager.this.completePendingTasks(null);
        }

        // Post cached endTrip requests to Server after locations have been posted successfully
        if (mPendingCompleteTripList != null && mPendingCompleteTripList.hasPendingCompleteTrips()) {
            HTLog.i(TAG, "Has pending complete trips");
            TransmissionManager.this.endPendingTrip(null);
        }
    }

    /**
     * Methods for StartTrip & StartPendingTrip
     */
    private abstract class PendingStartTripCallback {
        public abstract void onPendingTripStarted(boolean result);
    }

    private void startPendingTrips(final PendingStartTripCallback callback) {
        final ArrayList<PendingStartTrip> pendingStartTripList = mPendingStartTripList.getPendingStartTrips();

        for (final PendingStartTrip pendingStartTrip : pendingStartTripList) {
            mPendingStartTripList.removePendingStartTrip(pendingStartTrip);

            final HTTripParams tripParams = new HTTripParamsBuilder().setDriverID(pendingStartTrip.getDriverID())
                    .setTripID(pendingStartTrip.getTripID())
                    .setTaskIDs(pendingStartTrip.getTaskIDs())
                    .setVehicleType(pendingStartTrip.getVehicleType())
                    .createHTTripParams();

            tripParams.setStartTime(pendingStartTrip.getStartTime());
            tripParams.setStartLocation(pendingStartTrip.getStartLocation());

            this.startTrip(tripParams, new ClientTripStatusCallback() {
                @Override
                public void onError(VolleyError error, Exception exception) {
                    HTLog.e(TAG, "Task start push to Server failed: " +
                            (exception != null ? exception.toString() : "null"));

                    // startTrip call failed because of an invalid request
                    // Mark PendingStartTrip request as completed
                    pendingStartTrip.setStarted(true);

                    // Broadcast an error of startTrip failure
                    broadcastTripStartFailed(tripParams.getDriverID(), tripParams.getTaskIDs());

                    if (callback != null) {
                        if (PendingStartTripList.haveAllTripsStarted(pendingStartTripList)) {
                            callback.onPendingTripStarted(PendingStartTripList.haveAllTripsStartedSuccessfully(pendingStartTripList));
                        }
                    }
                }

                @Override
                public void onSuccess(boolean isOffline, HTTrip trip) {
                    // Start Location Service to start collecting Locations
                    if (HTLocationService.htLocationService == null) {
                        startDriverActivity(true);
                    }

                    pendingStartTrip.setStarted(true);

                    if (callback != null) {
                        if (PendingStartTripList.haveAllTripsStarted(pendingStartTripList)) {
                            callback.onPendingTripStarted(PendingStartTripList.haveAllTripsStartedSuccessfully(pendingStartTripList));
                        }
                    }
                }
            });
        }
    }

    public void startTrip(final HTTripParams tripParams, final ClientTripStatusCallback callback) {
        // Initialize TransmissionManager, in case applicable
        initializeTransmissionManager();

        mHTTPClient.startTrip(tripParams, new ClientTripStatusCallback() {
            @Override
            public void onSuccess(boolean isOffline, HTTrip trip) {
                HTLog.i(TAG, "Trip start pushed to Server: Task Details: " + trip.toString());
                if (callback != null) {
                    callback.onSuccess(isOffline, trip);
                }

                // Broadcast startTrip call success
                broadcastTripStarted(trip.getId(), trip.getDriverID(), trip.getTaskIDs());
            }

            @Override
            public void onError(VolleyError error, Exception exception) {
                if (TransmissionManager.isInvalidRequest(error)) {
                    HTLog.e(TAG, "Invalid Request: Trip start push to Server failed: " +
                            ((error.networkResponse != null) ? error.networkResponse.toString() : "null"));
                    if (callback != null) {
                        callback.onError(error, exception);
                    }

                    return;
                }

                // Return Error if DriverID was not provided, don't cache this request
                if (TextUtils.isEmpty(tripParams.getDriverID())) {
                    HTLog.e(TAG, "Trip start call failed online. Trip should start online to auto-generate a 'driverID'");
                    if (callback != null) {
                        callback.onError(null, new IllegalArgumentException("Trip start call failed online. " +
                                "Trip should start online to auto-generate a 'driverID'"));
                    }
                    return;
                }

                HTLog.i(TAG, "Saving Start Trip: " + tripParams.getDriverID()
                        + ", error while StartTripWithClient: " + exception.toString());
                TransmissionManager.this.savePendingStartTrip(tripParams);

                // Return HTTask as null in case of an offline startTrip
                if (callback != null) {
                    callback.onSuccess(true, null);
                }
            }
        });
    }

    private void savePendingStartTrip(HTTripParams htTripParams) {
        PendingStartTrip pendingStartTrip = new PendingStartTrip(htTripParams);
        mPendingStartTripList.addPendingStartTrip(pendingStartTrip);
    }

    /**
     * Methods for CompleteTask & CompletePendingTask
     */
    private abstract class PendingCompleteTaskCallback {
        public abstract void onPendingTaskCompleted(boolean result);
    }

    private void completePendingTasks(final PendingCompleteTaskCallback callback) {
        final ArrayList<PendingTask> tasks = mPendingTaskList.getTasks();

        for (final PendingTask task : tasks) {
            mPendingTaskList.removePendingTask(task);

            // Complete Task
            this.completeTask(task.getTaskID(), task.getCompletionTime(), task.getCompletionLocation(),
                    new ClientCompleteTaskCallback() {

                        @Override
                        public void onError(VolleyError error, Exception exception) {
                            // completeTask call failed because of an invalid request
                            // Mark PendingTask request as completed and delete it from cache
                            task.setCompleted(true);
                            // Broadcast an error of completeTask failure
                            broadcastTaskCompletionFailed(task.getTaskID());

                            if (callback != null) {
                                if (PendingTaskList.haveAllTasksCompleted(tasks)) {
                                    callback.onPendingTaskCompleted(PendingTaskList.haveAllTasksCompletedSuccessfully(tasks));
                                }
                            }
                        }

                        @Override
                        public void onSuccess(String taskID, boolean isDriverLive) {
                            task.setCompleted(true);
                            broadcastTaskCompleted(taskID);

                            if (callback != null) {
                                if (PendingTaskList.haveAllTasksCompleted(tasks)) {
                                    callback.onPendingTaskCompleted(PendingTaskList.haveAllTasksCompletedSuccessfully(tasks));
                                }
                            }
                        }
                    });
        }
    }

    public void completeTask(final String taskID, final Date completionTime, final Location location,
                             final ClientCompleteTaskCallback callback) {

        // Initialize TransmissionManager, in case applicable
        initializeTransmissionManager();

        // Post DeviceLogs To Server
        if (PostDeviceLogsManager.getInstance(mContext).hasPendingDeviceLogs()) {
            PostDeviceLogsManager.getInstance(mContext).postDeviceLogs(VERSION_NAME, HTNetworkClient.HT_NETWORK_CLIENT_MQTT);
        }

        // Post DeviceInfo To Server
        this.postDeviceInfoToServer(PostDataToServer.getDriverID(mContext));

        //Post Locations to Server on CompleteTask Call
        this.postLocationsToServer(TransmissionManager.getDriverID(mContext), new PostLocationCallback() {
            @Override
            public void onPostLocationSuccess() {
                TransmissionManager.this.completeTaskWithClient(taskID, completionTime, location, callback);
            }

            @Override
            public void onError(Exception exception) {
                TransmissionManager.this.savePendingCompleteTask(taskID, location, completionTime);

                if (callback != null) {
                    callback.onSuccess(taskID, true);
                }
            }
        });
    }

    private void completeTaskWithClient(final String taskID, final Date completionTime, final Location location,
                                        final ClientCompleteTaskCallback callback) {
        mHTTPClient.completeTask(taskID, completionTime, location, new ClientCompleteTaskCallback() {
            @Override
            public void onError(VolleyError error, Exception exception) {
                if (TransmissionManager.isInvalidRequest(error)) {
                    HTLog.e(TAG, "Invalid Request: Task complete push to Server failed: " +
                            ((error.networkResponse != null) ? error.networkResponse.toString() : "null"));
                    if (callback != null) {
                        callback.onError(error, exception);
                    }

                    return;
                }

                HTLog.i(TAG, "Saving Complete Task: " + taskID + ", error while completeTaskWithClient: " + exception.toString());
                TransmissionManager.this.savePendingCompleteTask(taskID, location, completionTime);

                if (callback != null) {
                    callback.onSuccess(taskID, true);
                }
            }

            @Override
            public void onSuccess(String taskID, boolean isDriverActive) {
                HTLog.i(TAG, "Task complete pushed to Server: Task ID: " + taskID);

                // If isDriverActive is FALSE, initiate stopDriverActivity
                if (!isDriverActive) {
                    stopDriverActivity(mContext);
                }

                if (callback != null) {
                    callback.onSuccess(taskID, isDriverActive);
                }
            }
        });
    }

    private void savePendingCompleteTask(final String taskID, final Location location, final Date completionTime) {
        PendingTask task = new PendingTask(taskID, location, "", TransmissionManager.getDriverID(mContext),
                completionTime);
        mPendingTaskList.addPendingTask(task);
    }

    /**
     * Methods for CompleteTrip & CompletePendingTrip
     */
    private abstract class PendingCompleteTripCallback {
        public abstract void onPendingTripCompleted(boolean result);
    }

    private void endPendingTrip(final PendingCompleteTripCallback callback) {
        final ArrayList<PendingCompleteTrip> pendingCompleteTripList = mPendingCompleteTripList.getPendingCompleteTrips();

        for (final PendingCompleteTrip pendingCompleteTrip : pendingCompleteTripList) {
            mPendingCompleteTripList.removePendingCompleteTrip(pendingCompleteTrip);

            // Complete Trip
            this.endTrip(pendingCompleteTrip.getTripID(), pendingCompleteTrip.getCompletionTime(),
                    pendingCompleteTrip.getCompletionLocation(), new ClientEndTripCallback() {

                        @Override
                        public void onError(VolleyError error, Exception exception) {
                            // endTrip call failed because of an invalid request
                            // Mark PendingCompleteTrip request as completed and delete it from cache
                            pendingCompleteTrip.setCompleted(true);
                            // Broadcast an error of endTrip failure
                            broadcastTripCompletionFailed(pendingCompleteTrip.getTripID());

                            if (callback != null) {
                                if (PendingCompleteTripList.haveAllTripsCompleted(pendingCompleteTripList)) {
                                    callback.onPendingTripCompleted(
                                            PendingCompleteTripList.haveAllTripsCompletedSuccessfully(pendingCompleteTripList));
                                }
                            }
                        }

                        @Override
                        public void onSuccess(boolean isOffline, HTTrip trip, boolean isDriverLive) {
                            pendingCompleteTrip.setCompleted(true);

                            if (callback != null) {
                                if (PendingCompleteTripList.haveAllTripsCompleted(pendingCompleteTripList)) {
                                    callback.onPendingTripCompleted(
                                            PendingCompleteTripList.haveAllTripsCompletedSuccessfully(pendingCompleteTripList));
                                }
                            }
                        }
                    });
        }
    }

    public void endTrip(final String tripID, final Date completionTime, final Location location,
                        final ClientEndTripCallback callback) {

        // Post DeviceLogs To Server
        if (PostDeviceLogsManager.getInstance(mContext).hasPendingDeviceLogs()) {
            PostDeviceLogsManager.getInstance(mContext).postDeviceLogs(VERSION_NAME, HTNetworkClient.HT_NETWORK_CLIENT_MQTT);
        }

        // Post DeviceInfo To Server
        this.postDeviceInfoToServer(PostDataToServer.getDriverID(mContext));

        // Initialize TransmissionManager, in case applicable
        initializeTransmissionManager();

        //Post Locations to Server on CompleteTask Call
        this.postLocationsToServer(TransmissionManager.getDriverID(mContext), new PostLocationCallback() {
            @Override
            public void onPostLocationSuccess() {
                TransmissionManager.this.endTripWithClient(tripID, completionTime, location, callback);
            }

            @Override
            public void onError(Exception exception) {
                TransmissionManager.this.savePendingEndTrip(tripID, location, completionTime);

                if (callback != null) {
                    callback.onSuccess(true, null, true);
                }
            }
        });
    }

    private void endTripWithClient(final String tripID, final Date completionTime, final Location location,
                                   final ClientEndTripCallback callback) {
        mHTTPClient.endTrip(tripID, completionTime, location, new ClientEndTripCallback() {
            @Override
            public void onError(VolleyError error, Exception exception) {
                if (TransmissionManager.isInvalidRequest(error)) {
                    HTLog.e(TAG, "Invalid Request: End Trip push to Server failed: " +
                            ((error.networkResponse != null) ? error.networkResponse.toString() : "null"));
                    if (callback != null) {
                        callback.onError(error, exception);
                    }

                    return;
                }

                TransmissionManager.this.savePendingEndTrip(tripID, location, completionTime);

                if (callback != null) {
                    callback.onSuccess(true, null, true);
                }
            }

            @Override
            public void onSuccess(boolean isOffline, HTTrip trip, boolean isDriverActive) {
                HTLog.i(TAG, "End Trip pushed to Server: Trip: " + trip);

                // If isDriverActive is FALSE, initiate stopDriverActivity
                if (!isDriverActive) {
                    stopDriverActivity(mContext);
                }

                if (callback != null) {
                    callback.onSuccess(isOffline, trip, isDriverActive);
                }

                // Broadcast endTrip call success
                broadcastTripCompleted(trip.getId(), trip.getDriverID());
            }
        });
    }

    private void savePendingEndTrip(final String tripID, final Location location, final Date completionTime) {
        HTLog.i(TAG, "Saving End Trip: " + tripID);

        PendingCompleteTrip pendingCompleteTrip = new PendingCompleteTrip(tripID, TransmissionManager.getDriverID(mContext),
                location, completionTime);
        mPendingCompleteTripList.addPendingCompleteTrip(pendingCompleteTrip);
    }

    private static boolean isInvalidRequest(VolleyError error) {
        return error != null && error.networkResponse != null && error.networkResponse.statusCode >= 400
                && error.networkResponse.statusCode < 500;
    }

    public static String getActiveShiftID(Context context) {
        SharedPreferences sharedpreferences = context.getSharedPreferences(HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        return sharedpreferences.getString(TransmitterConstants.HT_SHARED_PREFERENCES_ACTIVE_SHIFT_ID_KEY, null);
    }

    public static void setActiveShiftID(Context context, String shiftID) {
        SharedPreferences sharedpreferences = context.getSharedPreferences(HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.putString(TransmitterConstants.HT_SHARED_PREFERENCES_ACTIVE_SHIFT_ID_KEY, shiftID);
        editor.apply();
    }

    public static void clearActiveShiftID(Context context) {
        SharedPreferences sharedpreferences = context.getSharedPreferences(HTConstants.HT_SHARED_PREFERENCES_KEY, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedpreferences.edit();
        editor.remove(TransmitterConstants.HT_SHARED_PREFERENCES_ACTIVE_SHIFT_ID_KEY);
        editor.apply();
    }

    public void stopDriverActivity(Context context) {
        // HACK: Check if there are any PendingStart Requests and ignore stopDriverActivity call
        PendingStartTripList mPendingStartTripList = new PendingStartTripList(context);
        if (!mPendingStartTripList.hasPendingStartTrips()) {

            HTLog.i(TAG, "isDriverLive is FALSE. Stopping Location Service & Stopping Periodic Tasks");

            if (HTLocationService.htLocationService != null) {
                HTLocationService.htLocationService.stopSelfService();
            }

            // Set isDriverLive to False because all the tasks have been completed
            sdkControlsManager.setDriverActive(false);
            TransmissionManager.setIsDriverLive(context, false);

            // Driver is NOT Live, Remove CollectDeviceInfo PeriodicTask
            TransmissionManager.removeCollectDeviceInfoJob(context);

            // Driver is NOT Live, Remove HTLocationService Alarm
            TransmissionManager.removeLocationServiceAlarmJob(context);

            // Clear cached DeviceInfo Params
            DeviceInfoUtility.clearSavedDeviceInfoParams(context);

            TransmissionManager.broadcastServiceEnded(context);
        } else {
            HTLog.i(TAG, "isDriverLive is FALSE on server but PendingStart call on SDK. Not Stopping Driver");
        }
    }

    private static void removeLocationServiceAlarmJob(Context context) {
        try {
            HTTransmitterService.getInstance(context).getJobScheduler().removeJob(HTJob.JobID.LOCATION_SERVICE_ALARM);
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while removeLocationServiceAlarmJob: " + e);
        }
    }

    /**
     * Broadcast Methods
     */
    private void broadcastDriverConnectionSuccessful(String driverID) {
        Intent intent = new Intent(TransmitterConstants.HT_DRIVER_CONNECTION_SUCCESSFUL_INTENT);
        intent.putExtra(TransmitterConstants.HT_DRIVER_ID_KEY, driverID);
        LocalBroadcastManager.getInstance(mContext).sendBroadcast(intent);
    }

    private void broadcastTaskStarted(String taskID) {
        Intent intent = new Intent(TransmitterConstants.HT_ON_TASK_STARTED_INTENT);
        intent.putExtra(TransmitterConstants.HT_ON_TASK_STARTED_INTENT_EXTRA_TASK_ID, taskID);
        LocalBroadcastManager.getInstance(mContext).sendBroadcast(intent);
    }

    private void broadcastTaskStartFailed(String taskID) {
        Intent intent = new Intent(TransmitterConstants.HT_ON_TASK_START_FAILED_INTENT);
        intent.putExtra(TransmitterConstants.HT_ON_TASK_START_FAILED_INTENT_EXTRA_TASK_ID, taskID);
        LocalBroadcastManager.getInstance(mContext).sendBroadcast(intent);
    }

    private void broadcastTaskCompleted(String taskID) {
        Intent intent = new Intent(TransmitterConstants.HT_ON_TASK_COMPLETED_INTENT);
        intent.putExtra(TransmitterConstants.HT_ON_TASK_COMPLETED_INTENT_EXTRA_TASK_ID, taskID);
        LocalBroadcastManager.getInstance(mContext).sendBroadcast(intent);
    }

    private void broadcastTaskCompletionFailed(String taskID) {
        Intent intent = new Intent(TransmitterConstants.HT_ON_TASK_COMPLETION_FAILED_INTENT);
        intent.putExtra(TransmitterConstants.HT_ON_TASK_FAILED_INTENT_EXTRA_TASK_ID, taskID);
        LocalBroadcastManager.getInstance(mContext).sendBroadcast(intent);
    }

    private void broadcastTripStarted(String tripID, String driverID, ArrayList<String> taskIDs) {
        Intent intent = new Intent(TransmitterConstants.HT_ON_TRIP_STARTED_INTENT);
        intent.putExtra(TransmitterConstants.HT_ON_TRIP_STARTED_INTENT_EXTRA_TRIP_ID, tripID);
        intent.putExtra(TransmitterConstants.HT_ON_TRIP_STARTED_INTENT_EXTRA_DRIVER_ID, driverID);
        intent.putExtra(TransmitterConstants.HT_ON_TRIP_STARTED_INTENT_EXTRA_TASK_ID_LIST, taskIDs);
        LocalBroadcastManager.getInstance(mContext).sendBroadcast(intent);
    }

    private void broadcastTripStartFailed(String driverID, ArrayList<String> taskIDs) {
        Intent intent = new Intent(TransmitterConstants.HT_ON_TRIP_START_FAILED_INTENT);
        intent.putExtra(TransmitterConstants.HT_ON_TRIP_START_FAILED_INTENT_EXTRA_DRIVER_ID, driverID);
        intent.putExtra(TransmitterConstants.HT_ON_TRIP_START_FAILED_INTENT_EXTRA_TASK_ID_LIST, taskIDs);
        LocalBroadcastManager.getInstance(mContext).sendBroadcast(intent);
    }

    private void broadcastTripCompleted(String tripID, String driverID) {
        Intent intent = new Intent(TransmitterConstants.HT_ON_TRIP_COMPLETED_INTENT);
        intent.putExtra(TransmitterConstants.HT_ON_TRIP_COMPLETED_INTENT_EXTRA_TRIP_ID, tripID);
        intent.putExtra(TransmitterConstants.HT_ON_TRIP_COMPLETED_INTENT_EXTRA_DRIVER_ID, driverID);
        LocalBroadcastManager.getInstance(mContext).sendBroadcast(intent);
    }

    private void broadcastTripCompletionFailed(String tripID) {
        Intent intent = new Intent(TransmitterConstants.HT_ON_TRIP_COMPLETION_FAILED_INTENT);
        intent.putExtra(TransmitterConstants.HT_ON_TRIP_COMPLETION_FAILED_INTENT_EXTRA_TRIP_ID, tripID);
        LocalBroadcastManager.getInstance(mContext).sendBroadcast(intent);
    }

    private static void broadcastServiceStarted(Context context) {
        Intent intent = new Intent(TransmitterConstants.HT_ON_LOCATION_SERVICE_STARTED_INTENT);
        LocalBroadcastManager.getInstance(context).sendBroadcast(intent);
    }

    private static void broadcastServiceEnded(Context context) {
        Intent intent = new Intent(TransmitterConstants.HT_ON_DRIVER_NOT_ACTIVE_INTENT);
        LocalBroadcastManager.getInstance(context).sendBroadcast(intent);
    }
}