package io.hypertrack.lib.transmitter.service;

import android.app.AlarmManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.os.Bundle;

import com.google.android.gms.gcm.GcmNetworkManager;
import com.google.android.gms.gcm.OneoffTask;
import com.google.android.gms.gcm.PeriodicTask;

import java.util.Calendar;

import io.hypertrack.lib.common.scheduler.HTJob;
import io.hypertrack.lib.common.scheduler.HTJobScheduler;
import io.hypertrack.lib.common.util.HTLog;

/**
 * Created by piyush on 08/10/16.
 */
/** package */ class TransmitterJobScheduler extends HTJobScheduler {

    private static final String TAG = TransmitterJobScheduler.class.getSimpleName();

    public TransmitterJobScheduler(Context context) {
        super(context);
    }

    @Override
    protected boolean addPeriodicTaskJob(final HTJob job) {
        try {
            if (job == null)
                return false;

            // Check for any existing jobs with this jobID but different periodicTaskTag
            if (scheduledJobs != null) {
                HTJob existingJob = scheduledJobs.get(job.getJobId());
                if (existingJob != null && !existingJob.getPeriodicTaskTag().equalsIgnoreCase(job.getPeriodicTaskTag())) {
                    // Remove existing PeriodicTask job
                    removePeriodicTaskJob(existingJob.getJobId(), existingJob.getPeriodicTaskTag());
                }
            }

            Bundle bundle = new Bundle();
            bundle.putInt(HTJobScheduler.PERIODIC_TASK_JOB_ID_KEY, job.getJobId());

            if (job.isPeriodic()) {
                PeriodicTask task = new PeriodicTask.Builder()
                        .setExtras(bundle)
                        .setService(TransmitterJSPeriodicTask.class)
                        .setPeriod(job.getIntervalMillis() / 1000)
                        .setRequiredNetwork(job.getNetworkType())
                        .setRequiresCharging(job.getRequiresCharging())
                        .setPersisted(true)
                        .setTag(job.getPeriodicTaskTag())
                        .setUpdateCurrent(true)
                        .build();
                GcmNetworkManager.getInstance(mContext).schedule(task);

                HTLog.i(TAG, "PeriodicTask job: " + job + " scheduled to run at " + job.getIntervalMillis() + "ms interval");

            } else {
                OneoffTask task = new OneoffTask.Builder()
                        .setExtras(bundle)
                        .setService(TransmitterJSPeriodicTask.class)
                        .setTag(job.getPeriodicTaskTag())
                        .setExecutionWindow(0L, job.getIntervalMillis() / 1000)
                        .setRequiredNetwork(job.getNetworkType())
                        .setRequiresCharging(job.getRequiresCharging())
                        .setPersisted(true)
                        .setUpdateCurrent(true)
                        .build();
                GcmNetworkManager.getInstance(mContext).schedule(task);

                HTLog.i(TAG, "OneoffTask job: " + job + " scheduled to run after " + job.getIntervalMillis() + "ms");
            }

            return true;
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while addPeriodicTaskJob: " + e);
            return false;
        }
    }

    @Override
    protected boolean removePeriodicTaskJob(int htJobID, String periodicTaskTag) {
        try {
            // Cancel all PostData tasks with TAG, if not required
            GcmNetworkManager.getInstance(mContext).cancelTask(periodicTaskTag,
                    TransmitterJSPeriodicTask.class);
            return false;
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while removePeriodicTaskJob: " + e);
            return false;
        }
    }

    @Override
    protected boolean addAlarmJob(final HTJob job) {
        try {
            Bundle bundle = new Bundle();
            bundle.putInt(HTJobScheduler.ALARM_JOB_ID_KEY, job.getJobId());

            Intent intent = new Intent(mContext, TransmitterAlarmReceiver.class);
            intent.putExtras(bundle);
            PendingIntent pendingIntent = PendingIntent.getBroadcast(mContext, job.getJobId(), intent,
                    PendingIntent.FLAG_UPDATE_CURRENT);

            Calendar calendar = Calendar.getInstance();

            //Set the alarm for the first time and update the same in SharedPreferences
            AlarmManager alarm = (AlarmManager) mContext.getSystemService(Context.ALARM_SERVICE);

            if (job.isPeriodic()) {
                alarm.setRepeating(AlarmManager.RTC_WAKEUP, calendar.getTimeInMillis(),
                        job.getIntervalMillis(),
                        pendingIntent);

                HTLog.i(TAG, "RepeatingAlarm job: " + job + " scheduled to run at " + job.getIntervalMillis() + "ms interval");
            } else {
                long triggerInMillis = calendar.getTimeInMillis() + job.getIntervalMillis();

                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                    alarm.setExactAndAllowWhileIdle(AlarmManager.RTC_WAKEUP, triggerInMillis, pendingIntent);

                } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                    alarm.setExact(AlarmManager.RTC_WAKEUP, triggerInMillis, pendingIntent);

                } else {
                    alarm.set(AlarmManager.RTC_WAKEUP, triggerInMillis, pendingIntent);
                }

                HTLog.i(TAG, "ExactAlarm job: " + job + " scheduled to run after " + job.getIntervalMillis() + "ms");
            }

            return true;
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while addAlarmJob: " + e);
            return false;
        }
    }

    protected boolean removeAlarmJob(final int jobID) {
        try {
            //removing existing alarm
            Intent intent = new Intent(mContext, TransmitterAlarmReceiver.class);
            PendingIntent pendingIntent = PendingIntent.getBroadcast(mContext, jobID, intent, 0);

            //check if any alarm is set or no, if yes then remove
            if (pendingIntent != null) {
                AlarmManager alarmManager = (AlarmManager) mContext.getSystemService(Context.ALARM_SERVICE);
                alarmManager.cancel(pendingIntent);
            }

            return true;
        } catch (Exception e) {
            HTLog.e(TAG, "Exception occurred while removeAlarmJob: " + e);
            return false;
        }
    }
}
