/*
 * Java Genetic Algorithm Library (jenetics-7.1.1).
 * Copyright (c) 2007-2022 Franz Wilhelmstötter
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Author:
 *    Franz Wilhelmstötter (franz.wilhelmstoetter@gmail.com)
 */
package io.jenetics.ext.util;

import static java.util.Objects.requireNonNull;

import java.util.ListIterator;
import java.util.NoSuchElementException;

/**
 * @author <a href="mailto:franz.wilhelmstoetter@gmail.com">Franz Wilhelmstötter</a>
 * @version 3.9
 * @since 3.9
 */
final class TreeChildIterator<V, T extends Tree<V, T>>
	implements ListIterator<T>
{

	private final T _tree;

	private int cursor = 0;

	TreeChildIterator(final T tree) {
		_tree = requireNonNull(tree, "Tree must not be null.");
	}

	@Override
	public boolean hasNext() {
		return cursor != _tree.childCount();
	}

	@Override
	public T next() {
		final int i = cursor;
		if (cursor >= _tree.childCount()) {
			throw new NoSuchElementException();
		}

		cursor = i + 1;
		return _tree.childAt(i);
	}

	@Override
	public int nextIndex() {
		return cursor;
	}

	@Override
	public boolean hasPrevious() {
		return cursor != 0;
	}

	@Override
	public T previous() {
		final int i = cursor - 1;
		if (i < 0) {
			throw new NoSuchElementException();
		}

		cursor = i;
		return _tree.childAt(i);
	}

	@Override
	public int previousIndex() {
		return cursor - 1;
	}

	@Override
	public void set(final T value) {
		throw new UnsupportedOperationException(
			"Iterator is immutable."
		);
	}

	@Override
	public void add(final T value) {
		throw new UnsupportedOperationException(
			"Can't change Iterator size."
		);
	}

	@Override
	public void remove() {
		throw new UnsupportedOperationException(
			"Can't change Iterator size."
		);
	}

}
