/*
 * Decompiled with CFR 0.152.
 */
package io.joynr.messaging.routing;

import com.google.inject.name.Named;
import io.joynr.exceptions.JoynrDelayMessageException;
import io.joynr.exceptions.JoynrMessageNotSentException;
import io.joynr.exceptions.JoynrSendBufferFullException;
import io.joynr.exceptions.JoynrShutdownException;
import io.joynr.messaging.FailureAction;
import io.joynr.messaging.IMessaging;
import io.joynr.messaging.IMessagingMulticastSubscriber;
import io.joynr.messaging.IMessagingSkeleton;
import io.joynr.messaging.MessagingSkeletonFactory;
import io.joynr.messaging.routing.AddressManager;
import io.joynr.messaging.routing.MessageRouter;
import io.joynr.messaging.routing.MessagingStubFactory;
import io.joynr.messaging.routing.MulticastReceiverRegistry;
import io.joynr.messaging.routing.RoutingTable;
import java.text.DateFormat;
import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.Set;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import javax.inject.Inject;
import javax.inject.Singleton;
import joynr.JoynrMessage;
import joynr.system.RoutingTypes.Address;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MessageRouterImpl
implements MessageRouter {
    private static final long TERMINATION_TIMEOUT = 5000L;
    private Logger logger = LoggerFactory.getLogger(MessageRouterImpl.class);
    private final RoutingTable routingTable;
    private static final int UUID_TAIL = 32;
    private static final DateFormat DateFormatter = new SimpleDateFormat("dd/MM HH:mm:ss:sss");
    private ScheduledExecutorService scheduler;
    private long sendMsgRetryIntervalMs;
    private MessagingStubFactory messagingStubFactory;
    private final MessagingSkeletonFactory messagingSkeletonFactory;
    private AddressManager addressManager;
    protected final MulticastReceiverRegistry multicastReceiverRegistry;

    @Inject
    @Singleton
    public MessageRouterImpl(RoutingTable routingTable, @Named(value="io.joynr.messaging.scheduledthreadpool") ScheduledExecutorService scheduler, @Named(value="joynr.messaging.sendmsgretryintervalms") long sendMsgRetryIntervalMs, MessagingStubFactory messagingStubFactory, MessagingSkeletonFactory messagingSkeletonFactory, AddressManager addressManager, MulticastReceiverRegistry multicastReceiverRegistry) {
        this.routingTable = routingTable;
        this.scheduler = scheduler;
        this.sendMsgRetryIntervalMs = sendMsgRetryIntervalMs;
        this.messagingStubFactory = messagingStubFactory;
        this.messagingSkeletonFactory = messagingSkeletonFactory;
        this.addressManager = addressManager;
        this.multicastReceiverRegistry = multicastReceiverRegistry;
    }

    @Override
    public void removeNextHop(String participantId) {
        this.routingTable.remove(participantId);
    }

    @Override
    public boolean resolveNextHop(String participantId) {
        return this.routingTable.containsKey(participantId);
    }

    @Override
    public void addMulticastReceiver(final String multicastId, String subscriberParticipantId, String providerParticipantId) {
        this.logger.debug("Adding multicast receiver {} for multicast {} on provider {}", new Object[]{subscriberParticipantId, multicastId, providerParticipantId});
        this.multicastReceiverRegistry.registerMulticastReceiver(multicastId, subscriberParticipantId);
        this.performSubscriptionOperation(multicastId, providerParticipantId, new SubscriptionOperation(){

            @Override
            public void perform(IMessagingMulticastSubscriber messagingMulticastSubscriber) {
                messagingMulticastSubscriber.registerMulticastSubscription(multicastId);
            }
        });
    }

    @Override
    public void removeMulticastReceiver(final String multicastId, String subscriberParticipantId, String providerParticipantId) {
        this.multicastReceiverRegistry.unregisterMulticastReceiver(multicastId, subscriberParticipantId);
        this.performSubscriptionOperation(multicastId, providerParticipantId, new SubscriptionOperation(){

            @Override
            public void perform(IMessagingMulticastSubscriber messagingMulticastSubscriber) {
                messagingMulticastSubscriber.unregisterMulticastSubscription(multicastId);
            }
        });
    }

    private void performSubscriptionOperation(String multicastId, String providerParticipantId, SubscriptionOperation operation) {
        Address providerAddress = this.routingTable.get(providerParticipantId);
        IMessagingSkeleton messagingSkeleton = this.messagingSkeletonFactory.getSkeleton(providerAddress);
        if (messagingSkeleton != null && messagingSkeleton instanceof IMessagingMulticastSubscriber) {
            operation.perform((IMessagingMulticastSubscriber)((Object)messagingSkeleton));
        } else {
            this.logger.debug("No messaging skeleton found for address {}, not performing multicast subscription.", (Object)providerAddress);
        }
    }

    @Override
    public void addNextHop(String participantId, Address address) {
        this.routingTable.put(participantId, address);
    }

    @Override
    public void route(JoynrMessage message) {
        this.checkExpiry(message);
        this.routeInternal(message, 0L, 0);
    }

    protected void schedule(Runnable runnable, String messageId, long delay, TimeUnit timeUnit) {
        if (this.scheduler.isShutdown()) {
            JoynrShutdownException joynrShutdownEx = new JoynrShutdownException("MessageScheduler is shutting down already. Unable to send message [messageId: " + messageId + "].");
            throw joynrShutdownEx;
        }
        this.scheduler.schedule(runnable, delay, timeUnit);
    }

    protected Set<Address> getAddresses(JoynrMessage message) {
        return this.addressManager.getAddresses(message);
    }

    private void routeInternal(final JoynrMessage message, long delayMs, final int retriesCount) {
        try {
            this.logger.debug("Scheduling {} with delay {} and retries {}", new Object[]{message, delayMs, retriesCount});
            this.schedule(new Runnable(){

                @Override
                public void run() {
                    MessageRouterImpl.this.logger.debug("Starting processing of message {}", (Object)message);
                    try {
                        MessageRouterImpl.this.checkExpiry(message);
                        Set<Address> addresses = MessageRouterImpl.this.getAddresses(message);
                        if (addresses.isEmpty()) {
                            throw new JoynrMessageNotSentException("Failed to send Request: No route for given participantId: " + message.getTo());
                        }
                        for (Address address : addresses) {
                            String messageId = message.getId().substring(32);
                            MessageRouterImpl.this.logger.info(">>>>> SEND  ID:{}:{} from: {} to: {} header: {}", new Object[]{messageId, message.getType(), message.getHeaderValue("from"), message.getHeaderValue("to"), message.getHeader().toString()});
                            MessageRouterImpl.this.logger.debug(">>>>> body  ID:{}:{}: {}", new Object[]{messageId, message.getType(), message.getPayload()});
                            IMessaging messagingStub = MessageRouterImpl.this.messagingStubFactory.create(address);
                            messagingStub.transmit(message, MessageRouterImpl.this.createFailureAction(message, retriesCount));
                        }
                    }
                    catch (Exception error) {
                        MessageRouterImpl.this.logger.error("error in scheduled message router thread: {}", (Object)error.getMessage());
                        FailureAction failureAction = MessageRouterImpl.this.createFailureAction(message, retriesCount);
                        failureAction.execute(error);
                    }
                }
            }, message.getId(), delayMs, TimeUnit.MILLISECONDS);
        }
        catch (RejectedExecutionException e) {
            this.logger.error("Execution rejected while scheduling SendSerializedMessageRequest ", (Throwable)e);
            throw new JoynrSendBufferFullException(e);
        }
    }

    private void checkExpiry(JoynrMessage message) {
        long currentTimeMillis = System.currentTimeMillis();
        long ttlExpirationDateMs = message.getExpiryDate();
        if (ttlExpirationDateMs <= currentTimeMillis) {
            String errorMessage = MessageFormat.format("ttl must be greater than 0 / ttl timestamp must be in the future: now: {0} abs_ttl: {1}", currentTimeMillis, ttlExpirationDateMs);
            this.logger.error(errorMessage);
            throw new JoynrMessageNotSentException(errorMessage);
        }
    }

    private FailureAction createFailureAction(final JoynrMessage message, final int retriesCount) {
        FailureAction failureAction = new FailureAction(){
            final String messageId;
            {
                this.messageId = message.getId();
            }

            @Override
            public void execute(Throwable error) {
                long delayMs;
                if (error instanceof JoynrShutdownException) {
                    MessageRouterImpl.this.logger.warn("{}", (Object)error.getMessage());
                    return;
                }
                if (error instanceof JoynrMessageNotSentException) {
                    MessageRouterImpl.this.logger.error(" ERROR SENDING:  aborting send of messageId: {}. Error: {}", new Object[]{this.messageId, error.getMessage()});
                    return;
                }
                MessageRouterImpl.this.logger.warn("PROBLEM SENDING, will retry. messageId: {}. Error: {} Message: {}", new Object[]{this.messageId, error.getClass().getName(), error.getMessage()});
                if (error instanceof JoynrDelayMessageException) {
                    delayMs = ((JoynrDelayMessageException)error).getDelayMs();
                } else {
                    delayMs = MessageRouterImpl.this.sendMsgRetryIntervalMs;
                    delayMs += MessageRouterImpl.this.exponentialBackoff(delayMs, retriesCount);
                }
                try {
                    MessageRouterImpl.this.logger.error("Rescheduling messageId: {} with delay " + delayMs + " ms, new TTL expiration date: {}", (Object)this.messageId, (Object)DateFormatter.format(message.getExpiryDate()));
                    MessageRouterImpl.this.routeInternal(message, delayMs, retriesCount + 1);
                    return;
                }
                catch (JoynrSendBufferFullException e) {
                    try {
                        MessageRouterImpl.this.logger.error("Rescheduling message: {} delayed {} ms because send buffer is full", (Object)delayMs, (Object)this.messageId);
                        Thread.sleep(delayMs);
                        this.execute(e);
                    }
                    catch (InterruptedException e1) {
                        Thread.currentThread().interrupt();
                        return;
                    }
                    return;
                }
            }
        };
        return failureAction;
    }

    @Override
    public void shutdown() {
        this.scheduler.shutdown();
        try {
            if (!this.scheduler.awaitTermination(5000L, TimeUnit.MILLISECONDS)) {
                this.logger.error("Message Scheduler did not shut down in time. Timedout out waiting for executor service to shutdown after {}ms.", (Object)5000L);
                this.logger.debug("Attempting to shutdown scheduler {} forcibly.", (Object)this.scheduler);
                this.scheduler.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.logger.error("Message Scheduler shutdown interrupted: {}", (Object)e.getMessage());
        }
    }

    private long exponentialBackoff(long delayMs, int retries) {
        this.logger.debug("TRIES: " + retries);
        long millis = delayMs + (long)((double)((long)(2 ^ retries) * delayMs) * Math.random());
        this.logger.debug("MILLIS: " + millis);
        return millis;
    }

    private static interface SubscriptionOperation {
        public void perform(IMessagingMulticastSubscriber var1);
    }
}

