/*
 * Decompiled with CFR 0.152.
 */
package io.joynr.messaging.routing;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.google.inject.name.Named;
import io.joynr.exceptions.JoynrIllegalStateException;
import io.joynr.exceptions.JoynrRuntimeException;
import io.joynr.messaging.routing.AddressOperation;
import io.joynr.messaging.routing.RoutingEntry;
import io.joynr.messaging.routing.RoutingTable;
import io.joynr.messaging.routing.RoutingTableAddressValidator;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import joynr.system.RoutingTypes.Address;
import joynr.system.RoutingTypes.MqttAddress;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class RoutingTableImpl
implements RoutingTable {
    private static final Logger logger = LoggerFactory.getLogger(RoutingTableImpl.class);
    private ConcurrentMap<String, RoutingEntry> hashMap = new ConcurrentHashMap<String, RoutingEntry>();
    private final long routingTableGracePeriodMs;
    private final Set<String> knownGbidsSet;
    private String gcdParticipantId;
    private final RoutingTableAddressValidator addressValidator;

    @Inject
    public RoutingTableImpl(@Named(value="joynr.messaging.routingtablegraceperiodms") long routingTableGracePeriodMs, @Named(value="joynr.internal.messaging.gbidArray") String[] gbidsArray, RoutingTableAddressValidator addressValidator) {
        this.routingTableGracePeriodMs = routingTableGracePeriodMs;
        this.knownGbidsSet = new HashSet<String>();
        this.knownGbidsSet.addAll(Arrays.asList(gbidsArray));
        this.gcdParticipantId = "";
        this.addressValidator = addressValidator;
    }

    @Override
    public void setGcdParticipantId(String gcdParticipantId) {
        if (gcdParticipantId == null) {
            throw new JoynrIllegalStateException("The provided gcdParticipantId is null.");
        }
        this.gcdParticipantId = gcdParticipantId;
    }

    @Override
    public Address get(String participantId) {
        return this.getInternal(participantId);
    }

    @Override
    public Address get(String participantId, String gbid) {
        Address address = this.getInternal(participantId);
        if (address != null && this.gcdParticipantId.equals(participantId) && address instanceof MqttAddress) {
            if (!this.knownGbidsSet.contains(gbid)) {
                logger.error("The provided gbid {} for the participantId {} is unknown", (Object)gbid, (Object)participantId);
                address = null;
            } else {
                MqttAddress mqttAddress = new MqttAddress((MqttAddress)address);
                mqttAddress.setBrokerUri(gbid);
                address = mqttAddress;
            }
        }
        logger.trace("leaving get(participantId={}, gbid={}) = {}", new Object[]{participantId, gbid, address});
        return address;
    }

    private Address getInternal(String participantId) {
        logger.trace("entering getInternal(participantId={})", (Object)participantId);
        this.dumpRoutingTableEntry();
        RoutingEntry routingEntry = (RoutingEntry)this.hashMap.get(participantId);
        if (routingEntry == null) {
            logger.trace("leaving getInternal(participantId={}) = null", (Object)participantId);
            return null;
        }
        logger.trace("leaving getInternal(participantId={}) = {}", (Object)participantId, (Object)routingEntry.getAddress());
        return routingEntry.getAddress();
    }

    private void dumpRoutingTableEntry() {
        if (logger.isTraceEnabled()) {
            StringBuilder message = new StringBuilder("Routing table entries:\n");
            for (Map.Entry eachEntry : this.hashMap.entrySet()) {
                message.append("\t> ").append((String)eachEntry.getKey()).append("\t-\t").append(((RoutingEntry)eachEntry.getValue()).address).append("\t-\t").append(((RoutingEntry)eachEntry.getValue()).isGloballyVisible).append("\t-\t").append(((RoutingEntry)eachEntry.getValue()).expiryDateMs).append("\t-\t").append(((RoutingEntry)eachEntry.getValue()).isSticky).append("\n");
            }
            logger.trace(message.toString());
        }
    }

    private void updateRoutingEntry(String participantId, RoutingEntry oldRoutingEntry, RoutingEntry newRoutingEntry) {
        logger.debug("put(participantId={}, address={}, isGloballyVisible={}, expiryDateMs={}, sticky={}). Replacing previous entry with address={}, isGloballyVisible={}, expiryDateMs={}, sticky={}", new Object[]{participantId, newRoutingEntry.address, newRoutingEntry.isGloballyVisible, newRoutingEntry.expiryDateMs, newRoutingEntry.isSticky, oldRoutingEntry.address, oldRoutingEntry.isGloballyVisible, oldRoutingEntry.expiryDateMs, oldRoutingEntry.isSticky});
        this.mergeRoutingEntryAttributes(newRoutingEntry, oldRoutingEntry.getExpiryDateMs(), oldRoutingEntry.getIsSticky());
        this.hashMap.put(participantId, newRoutingEntry);
    }

    @Override
    public void put(String participantId, Address address, boolean isGloballyVisible, long expiryDateMs) {
        boolean sticky = false;
        this.put(participantId, address, isGloballyVisible, expiryDateMs, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void put(String participantId, Address address, boolean isGloballyVisible, long expiryDateMs, boolean sticky) {
        if (!this.addressValidator.isValidForRoutingTable(address)) {
            logger.trace("participantId={} has an unsupported address within this process.", (Object)participantId);
            return;
        }
        try {
            expiryDateMs = Math.addExact(expiryDateMs, this.routingTableGracePeriodMs);
        }
        catch (ArithmeticException e) {
            expiryDateMs = Long.MAX_VALUE;
        }
        RoutingEntry newRoutingEntry = new RoutingEntry(address, isGloballyVisible, expiryDateMs, sticky);
        RoutingTableImpl routingTableImpl = this;
        synchronized (routingTableImpl) {
            boolean addressOrVisibilityOfRoutingEntryChanged;
            boolean routingEntryAlreadyPresent;
            RoutingEntry oldRoutingEntry = this.hashMap.putIfAbsent(participantId, newRoutingEntry);
            boolean bl = routingEntryAlreadyPresent = oldRoutingEntry != null;
            if (!routingEntryAlreadyPresent) {
                logger.debug("put(participantId={}, address={}, isGloballyVisible={}, expiryDateMs={}, sticky={}) successfully into routing table", new Object[]{participantId, address, isGloballyVisible, expiryDateMs, sticky});
                return;
            }
            boolean bl2 = addressOrVisibilityOfRoutingEntryChanged = !address.equals((Object)oldRoutingEntry.getAddress()) || oldRoutingEntry.getIsGloballyVisible() != isGloballyVisible;
            if (addressOrVisibilityOfRoutingEntryChanged) {
                if (oldRoutingEntry.isSticky) {
                    logger.error("unable to update(participantId={}, address={}, isGloballyVisible={}, expiryDateMs={}, sticky={}) into routing table, since the participant ID is already associated with STICKY routing entry address={}, isGloballyVisible={}", new Object[]{participantId, address, isGloballyVisible, expiryDateMs, sticky, oldRoutingEntry.address, oldRoutingEntry.isGloballyVisible});
                } else if (this.addressValidator.allowUpdate(oldRoutingEntry, newRoutingEntry)) {
                    this.updateRoutingEntry(participantId, oldRoutingEntry, newRoutingEntry);
                } else {
                    logger.warn("unable to update(participantId={}, address={}, isGloballyVisible={}, expiryDateMs={}, sticky={}) into routing table, since the participant ID is already associated with routing entry address={}, isGloballyVisible={}", new Object[]{participantId, address, isGloballyVisible, expiryDateMs, sticky, oldRoutingEntry.address, oldRoutingEntry.isGloballyVisible});
                }
            } else {
                logger.trace("put(participantId={}, address={}, isGloballyVisible={}, expiryDateMs={}, sticky={}): Entry exists. Updating expiryDate and sticky-flag", new Object[]{participantId, address, isGloballyVisible, expiryDateMs, sticky});
                this.mergeRoutingEntryAttributes(oldRoutingEntry, expiryDateMs, sticky);
            }
        }
    }

    private void mergeRoutingEntryAttributes(RoutingEntry entry, long expiryDateMs, boolean isSticky) {
        if (entry.getExpiryDateMs() < expiryDateMs) {
            entry.setExpiryDateMs(expiryDateMs);
        }
        if (isSticky && !entry.getIsSticky()) {
            entry.setIsSticky(true);
        }
    }

    @Override
    public boolean containsKey(String participantId) {
        boolean containsKey = this.hashMap.containsKey(participantId);
        logger.trace("checking for participant: {} success: {}", (Object)participantId, (Object)containsKey);
        if (!containsKey) {
            this.dumpRoutingTableEntry();
        }
        return containsKey;
    }

    @Override
    public boolean getIsGloballyVisible(String participantId) {
        RoutingEntry routingEntry = (RoutingEntry)this.hashMap.get(participantId);
        if (routingEntry == null) {
            throw new JoynrRuntimeException("participantId doesn't exist in the routing table");
        }
        return routingEntry.getIsGloballyVisible();
    }

    @Override
    public long getExpiryDateMs(String participantId) {
        RoutingEntry routingEntry = (RoutingEntry)this.hashMap.get(participantId);
        if (routingEntry == null) {
            throw new JoynrRuntimeException("participantId doesn't exist in the routing table");
        }
        return routingEntry.getExpiryDateMs();
    }

    @Override
    public boolean getIsSticky(String participantId) {
        RoutingEntry routingEntry = (RoutingEntry)this.hashMap.get(participantId);
        if (routingEntry == null) {
            throw new JoynrRuntimeException("participantId doesn't exist in the routing table");
        }
        return routingEntry.getIsSticky();
    }

    @Override
    public void remove(String participantId) {
        RoutingEntry routingEntry = (RoutingEntry)this.hashMap.get(participantId);
        if (routingEntry != null) {
            if (routingEntry.isSticky) {
                logger.warn("Cannot remove sticky routing entry (participantId={}, address={}, isGloballyVisible={}, expiryDateMs={}, sticky={}) from routing table", new Object[]{participantId, routingEntry.getAddress(), routingEntry.getIsGloballyVisible(), routingEntry.getExpiryDateMs(), routingEntry.getIsSticky()});
            } else {
                logger.debug("removing(participantId={}, address={}, isGloballyVisible={}, expiryDateMs={}, sticky={}) from routing table", new Object[]{participantId, routingEntry.getAddress(), routingEntry.getIsGloballyVisible(), routingEntry.getExpiryDateMs(), routingEntry.getIsSticky()});
                this.hashMap.remove(participantId);
            }
        }
    }

    @Override
    public void apply(AddressOperation addressOperation) {
        if (addressOperation == null) {
            throw new IllegalArgumentException();
        }
        for (RoutingEntry routingEntry : this.hashMap.values()) {
            addressOperation.perform(routingEntry.getAddress());
        }
    }

    @Override
    public void purge() {
        logger.trace("purge: begin");
        Iterator it = this.hashMap.entrySet().iterator();
        long currentTimeMillis = System.currentTimeMillis();
        while (it.hasNext()) {
            Map.Entry e = it.next();
            if (logger.isTraceEnabled()) {
                logger.trace("check: participantId = {}, sticky = {}, expiryDateMs = {}, now = {}", new Object[]{e.getKey(), ((RoutingEntry)e.getValue()).getIsSticky(), ((RoutingEntry)e.getValue()).getExpiryDateMs(), currentTimeMillis});
            }
            if (((RoutingEntry)e.getValue()).getIsSticky() || ((RoutingEntry)e.getValue()).expiryDateMs >= currentTimeMillis) continue;
            if (logger.isTraceEnabled()) {
                logger.trace("purging(participantId={}, address={}, isGloballyVisible={}, expiryDateMs={}, sticky={}) from routing table", new Object[]{e.getKey(), ((RoutingEntry)e.getValue()).getAddress(), ((RoutingEntry)e.getValue()).getIsGloballyVisible(), ((RoutingEntry)e.getValue()).getExpiryDateMs(), ((RoutingEntry)e.getValue()).getIsSticky()});
            }
            it.remove();
        }
        logger.trace("purge: end");
    }
}

