/*
 * Decompiled with CFR 0.152.
 */
package io.joynr.messaging.routing;

import com.google.inject.Inject;
import com.google.inject.name.Named;
import io.joynr.messaging.persistence.MessagePersister;
import io.joynr.messaging.routing.DelayableImmutableMessage;
import io.joynr.messaging.routing.RoutingTable;
import java.util.Set;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.TimeUnit;
import joynr.ImmutableMessage;
import joynr.Message;
import joynr.system.RoutingTypes.Address;
import joynr.system.RoutingTypes.RoutingTypesUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MessageQueue {
    private static final Logger logger = LoggerFactory.getLogger(MessageQueue.class);
    public static final String MESSAGE_QUEUE_ID = "io.joynr.messaging.queue.id";
    public static final String PROPERTY_MESSAGE_QUEUE_SHUTDOWN_MAX_TIMEOUT = "io.joynr.messaging.queue.shutdown.timeout";
    private static final long STARTUP_GRACE_PERIOD_MS = 1000L;
    private DelayQueue<DelayableImmutableMessage> delayableImmutableMessages;
    private final long shutdownTimeoutMs;
    private final String messageQueueId;
    private final MessagePersister messagePersister;
    private final RoutingTable routingTable;

    @Inject
    public MessageQueue(DelayQueue<DelayableImmutableMessage> delayableImmutableMessages, MaxTimeoutHolder maxTimeoutHolder, @Named(value="io.joynr.messaging.queue.id") String messageQueueId, MessagePersister messagePersister, RoutingTable routingTable) {
        this.delayableImmutableMessages = delayableImmutableMessages;
        this.shutdownTimeoutMs = maxTimeoutHolder.getTimeout();
        this.messageQueueId = messageQueueId;
        this.messagePersister = messagePersister;
        this.routingTable = routingTable;
        this.fetchAndQueuePersistedMessages(delayableImmutableMessages, messageQueueId);
    }

    private void registerReplyToAddress(DelayableImmutableMessage delayableImmutableMessage) {
        ImmutableMessage message = delayableImmutableMessage.getMessage();
        Message.MessageType messageType = message.getType();
        if (!(messageType.equals((Object)Message.MessageType.VALUE_MESSAGE_TYPE_REQUEST) || messageType.equals((Object)Message.MessageType.VALUE_MESSAGE_TYPE_SUBSCRIPTION_REQUEST) || messageType.equals((Object)Message.MessageType.VALUE_MESSAGE_TYPE_BROADCAST_SUBSCRIPTION_REQUEST) || messageType.equals((Object)Message.MessageType.VALUE_MESSAGE_TYPE_MULTICAST_SUBSCRIPTION_REQUEST))) {
            return;
        }
        String replyTo = message.getReplyTo();
        if (replyTo != null && !replyTo.isEmpty()) {
            Address address = RoutingTypesUtil.fromAddressString((String)replyTo);
            boolean isGloballyVisible = true;
            long expiryDateMs = message.getTtlMs();
            this.routingTable.put(message.getSender(), address, true, expiryDateMs);
        }
    }

    private void restoreMessage(DelayableImmutableMessage delayableImmutableMessage) {
        delayableImmutableMessage.setDelay(1000L);
        this.delayableImmutableMessages.put(delayableImmutableMessage);
    }

    private void fetchAndQueuePersistedMessages(DelayQueue<DelayableImmutableMessage> delayableImmutableMessages, String messageQueueId) {
        Set<DelayableImmutableMessage> persistedFromLastRun = this.messagePersister.fetchAll(messageQueueId);
        if (persistedFromLastRun != null) {
            persistedFromLastRun.forEach(this::registerReplyToAddress);
            persistedFromLastRun.forEach(this::restoreMessage);
        }
    }

    void waitForQueueToDrain() {
        int remainingMessages = this.delayableImmutableMessages.size();
        logger.info("Joynr message queue stopping. Contains {} remaining messages.", (Object)remainingMessages);
        if (remainingMessages > 0) {
            long shutdownStart = System.currentTimeMillis();
            while (System.currentTimeMillis() - shutdownStart < this.shutdownTimeoutMs && this.delayableImmutableMessages.size() != 0) {
                try {
                    Thread.sleep(5L);
                }
                catch (InterruptedException e) {
                    logger.error("Interrupted while waiting for joynr message queue to drain.");
                    e.printStackTrace();
                }
            }
        }
        if ((remainingMessages = this.delayableImmutableMessages.size()) == 0) {
            logger.info("Joynr message queue successfully emptied.");
        } else {
            logger.info("Joynr message queue still contained {} messages at shutdown.", (Object)remainingMessages);
        }
    }

    public void put(DelayableImmutableMessage delayableImmutableMessage) {
        if (this.messagePersister.persist(this.messageQueueId, delayableImmutableMessage)) {
            logger.trace("Message {} was persisted for messageQueueId {}", (Object)delayableImmutableMessage.getMessage(), (Object)this.messageQueueId);
        } else {
            logger.trace("Message {} was not persisted for messageQueueId {}", (Object)delayableImmutableMessage.getMessage(), (Object)this.messageQueueId);
        }
        this.delayableImmutableMessages.put(delayableImmutableMessage);
    }

    public DelayableImmutableMessage poll(long timeout, TimeUnit unit) throws InterruptedException {
        DelayableImmutableMessage message = (DelayableImmutableMessage)this.delayableImmutableMessages.poll(timeout, unit);
        if (message != null) {
            this.messagePersister.remove(this.messageQueueId, message);
        }
        return message;
    }

    public static class MaxTimeoutHolder {
        @Inject(optional=true)
        @Named(value="io.joynr.messaging.queue.shutdown.timeout")
        private Long timeout = 5000L;

        public long getTimeout() {
            return this.timeout;
        }
    }
}

