/*
 * Decompiled with CFR 0.152.
 */
package io.kestra.core.storages.kv;

import io.kestra.core.exceptions.ResourceExpiredException;
import io.kestra.core.serializers.JacksonMapper;
import io.kestra.core.storages.FileAttributes;
import io.kestra.core.storages.StorageInterface;
import io.kestra.core.storages.StorageObject;
import io.kestra.core.storages.kv.KVEntry;
import io.kestra.core.storages.kv.KVStore;
import io.kestra.core.storages.kv.KVStoreException;
import io.kestra.core.storages.kv.KVValue;
import io.kestra.core.storages.kv.KVValueAndMetadata;
import io.kestra.core.utils.Rethrow;
import jakarta.annotation.Nullable;
import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.time.Duration;
import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Pattern;

public class InternalKVStore
implements KVStore {
    private static final Pattern DURATION_PATTERN = Pattern.compile("^P(?=[^T]|T.)(?:\\d*D)?(?:T(?=.)(?:\\d*H)?(?:\\d*M)?(?:\\d*S)?)?$");
    private final String namespace;
    private final String tenant;
    private final StorageInterface storage;

    public InternalKVStore(@Nullable String tenant, String namespace, StorageInterface storage) {
        this.namespace = Objects.requireNonNull(namespace, "namespace cannot be null");
        this.storage = Objects.requireNonNull(storage, "storage cannot be null");
        this.tenant = tenant;
    }

    @Override
    public String namespace() {
        return this.namespace;
    }

    @Override
    public void put(String key, KVValueAndMetadata value, boolean overwrite) throws IOException {
        KVStore.validateKey(key);
        if (!overwrite && this.exists(key)) {
            throw new KVStoreException(String.format("Cannot set value for key '%s'. Key already exists and `overwrite` is set to `false`.", key));
        }
        byte[] serialized = JacksonMapper.ofIon().writeValueAsBytes(value.value());
        this.storage.put(this.tenant, this.namespace, this.storageUri(key), new StorageObject(value.metadataAsMap(), new ByteArrayInputStream(serialized)));
    }

    @Override
    public Optional<KVValue> getValue(String key) throws IOException, ResourceExpiredException {
        return this.getRawValue(key).map(Rethrow.throwFunction(raw -> {
            String valueStr;
            Object value = JacksonMapper.ofIon().readValue(raw, Object.class);
            if (value instanceof String && DURATION_PATTERN.matcher(valueStr = (String)value).matches()) {
                return new KVValue(Duration.parse(valueStr));
            }
            return new KVValue(value);
        }));
    }

    public Optional<String> getRawValue(String key) throws IOException, ResourceExpiredException {
        StorageObject withMetadata;
        KVStore.validateKey(key);
        try {
            withMetadata = this.storage.getWithMetadata(this.tenant, this.namespace, this.storageUri(key));
        }
        catch (FileNotFoundException e) {
            return Optional.empty();
        }
        KVValueAndMetadata kvStoreValueWrapper = KVValueAndMetadata.from(withMetadata);
        Instant expirationDate = kvStoreValueWrapper.metadata().getExpirationDate();
        if (expirationDate != null && Instant.now().isAfter(expirationDate)) {
            this.delete(key);
            throw new ResourceExpiredException("The requested value has expired");
        }
        return Optional.of((String)kvStoreValueWrapper.value());
    }

    @Override
    public boolean delete(String key) throws IOException {
        KVStore.validateKey(key);
        URI uri = this.storageUri(key);
        boolean deleted = this.storage.delete(this.tenant, this.namespace, uri);
        URI metadataURI = URI.create(uri.getPath() + ".metadata");
        if (this.storage.exists(this.tenant, this.namespace, metadataURI)) {
            this.storage.delete(this.tenant, this.namespace, metadataURI);
        }
        return deleted;
    }

    @Override
    public List<KVEntry> list() throws IOException {
        List<FileAttributes> list = this.listAllFromStorage();
        return list.stream().map(Rethrow.throwFunction(KVEntry::from)).filter(kvEntry -> Optional.ofNullable(kvEntry.expirationDate()).map(expirationDate -> Instant.now().isBefore((Instant)expirationDate)).orElse(true)).toList();
    }

    @Override
    public List<KVEntry> listAll() throws IOException {
        List<FileAttributes> list = this.listAllFromStorage();
        return list.stream().map(Rethrow.throwFunction(KVEntry::from)).toList();
    }

    private List<FileAttributes> listAllFromStorage() throws IOException {
        try {
            return this.storage.list(this.tenant, this.namespace, this.storageUri(null));
        }
        catch (FileNotFoundException e) {
            return Collections.emptyList();
        }
    }

    @Override
    public Optional<KVEntry> get(String key) throws IOException {
        KVStore.validateKey(key);
        try {
            KVEntry entry = KVEntry.from(this.storage.getAttributes(this.tenant, this.namespace, this.storageUri(key)));
            if (entry.expirationDate() != null && Instant.now().isAfter(entry.expirationDate())) {
                this.delete(key);
                return Optional.empty();
            }
            return Optional.of(entry);
        }
        catch (FileNotFoundException e) {
            return Optional.empty();
        }
    }
}

