/*
 * Decompiled with CFR 0.152.
 */
package io.kestra.core.storages;

import io.kestra.core.exceptions.ResourceExpiredException;
import io.kestra.core.junit.annotations.KestraTest;
import io.kestra.core.repositories.ArrayListTotal;
import io.kestra.core.repositories.KvMetadataRepositoryInterface;
import io.kestra.core.serializers.JacksonMapper;
import io.kestra.core.storages.StorageContext;
import io.kestra.core.storages.StorageInterface;
import io.kestra.core.storages.StorageObject;
import io.kestra.core.storages.kv.InternalKVStore;
import io.kestra.core.storages.kv.KVEntry;
import io.kestra.core.storages.kv.KVMetadata;
import io.kestra.core.storages.kv.KVStore;
import io.kestra.core.storages.kv.KVValue;
import io.kestra.core.storages.kv.KVValueAndMetadata;
import io.kestra.core.utils.IdUtils;
import jakarta.inject.Inject;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.time.temporal.Temporal;
import java.time.temporal.TemporalUnit;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.assertj.core.api.Assertions;
import org.assertj.core.data.TemporalOffset;
import org.junit.jupiter.api.Test;

@KestraTest
class InternalKVStoreTest {
    private static final Instant date = Instant.now().truncatedTo(ChronoUnit.MILLIS);
    private static final Map<String, Object> complexValue = Map.of("some", "complex", "object", Map.of("with", "nested", "values", date));
    static final String TEST_KV_KEY = "my-key";
    @Inject
    private StorageInterface storageInterface;
    @Inject
    private KvMetadataRepositoryInterface kvMetadataRepository;

    InternalKVStoreTest() {
    }

    @Test
    void list() throws IOException, InterruptedException {
        Instant now = Instant.now();
        InternalKVStore kv = this.kv();
        Assertions.assertThat((int)kv.list().size()).isZero();
        String description = "myDescription";
        kv.put(TEST_KV_KEY, new KVValueAndMetadata(new KVMetadata(description, Duration.ofMinutes(5L)), complexValue));
        kv.put("my-second-key", new KVValueAndMetadata(new KVMetadata(null, Duration.ofMinutes(10L)), complexValue));
        kv.put("expired-key", new KVValueAndMetadata(new KVMetadata(null, Duration.ofMillis(1L)), complexValue));
        Thread.sleep(2L);
        ArrayListTotal list = kv.list();
        Assertions.assertThat((int)list.size()).isEqualTo(2);
        list.forEach(kvEntry -> {
            Assertions.assertThat((Instant)kvEntry.creationDate()).isCloseTo((Temporal)now, (TemporalOffset)Assertions.within((long)1L, (TemporalUnit)ChronoUnit.SECONDS));
            Assertions.assertThat((Instant)kvEntry.updateDate()).isCloseTo((Temporal)now, (TemporalOffset)Assertions.within((long)1L, (TemporalUnit)ChronoUnit.SECONDS));
        });
        Map map = list.stream().collect(Collectors.toMap(KVEntry::key, Function.identity()));
        Assertions.assertThat((int)map.size()).isEqualTo(2);
        KVEntry myKeyValue = (KVEntry)map.get(TEST_KV_KEY);
        Assertions.assertThat((myKeyValue.creationDate().plus(Duration.ofMinutes(4L)).isBefore(myKeyValue.expirationDate()) && myKeyValue.creationDate().plus(Duration.ofMinutes(6L)).isAfter(myKeyValue.expirationDate()) ? 1 : 0) != 0).isTrue();
        Assertions.assertThat((String)myKeyValue.description()).isEqualTo(description);
        KVEntry mySecondKeyValue = (KVEntry)map.get("my-second-key");
        Assertions.assertThat((mySecondKeyValue.creationDate().plus(Duration.ofMinutes(9L)).isBefore(mySecondKeyValue.expirationDate()) && mySecondKeyValue.creationDate().plus(Duration.ofMinutes(11L)).isAfter(mySecondKeyValue.expirationDate()) ? 1 : 0) != 0).isTrue();
        Assertions.assertThat((String)mySecondKeyValue.description()).isNull();
    }

    @Test
    void listAll() throws IOException {
        Instant now = Instant.now();
        InternalKVStore kv = this.kv();
        Assertions.assertThat((int)kv.list().size()).isZero();
        String description = "myDescription";
        kv.put(TEST_KV_KEY, new KVValueAndMetadata(new KVMetadata(description, Duration.ofMinutes(5L)), complexValue));
        kv.put("key-without-expiration", new KVValueAndMetadata(new KVMetadata(null, (Duration)null), complexValue));
        kv.put("expired-key", new KVValueAndMetadata(new KVMetadata(null, Duration.ofMillis(1L)), complexValue));
        List list = kv.listAll();
        Assertions.assertThat((int)list.size()).isEqualTo(3);
        list.forEach(kvEntry -> {
            Assertions.assertThat((Instant)kvEntry.creationDate()).isCloseTo((Temporal)now, (TemporalOffset)Assertions.within((long)1L, (TemporalUnit)ChronoUnit.SECONDS));
            Assertions.assertThat((Instant)kvEntry.updateDate()).isCloseTo((Temporal)now, (TemporalOffset)Assertions.within((long)1L, (TemporalUnit)ChronoUnit.SECONDS));
        });
        List<String> keys = list.stream().map(KVEntry::key).toList();
        Assertions.assertThat(keys).containsExactlyInAnyOrder((Object[])new String[]{TEST_KV_KEY, "key-without-expiration", "expired-key"});
    }

    @Test
    void put() throws IOException {
        InternalKVStore kv = this.kv();
        Instant before = Instant.now();
        String description = "myDescription";
        kv.put(TEST_KV_KEY, new KVValueAndMetadata(new KVMetadata(description, Duration.ofMinutes(5L)), complexValue));
        StorageObject withMetadata = this.storageInterface.getWithMetadata("main", kv.namespace(), URI.create("/" + kv.namespace().replace(".", "/") + "/_kv/my-key.ion"));
        String valueFile = new String(withMetadata.inputStream().readAllBytes());
        Instant expirationDate = Instant.parse((CharSequence)withMetadata.metadata().get("expirationDate"));
        Assertions.assertThat((expirationDate.isAfter(before.plus(Duration.ofMinutes(4L))) && expirationDate.isBefore(before.plus(Duration.ofMinutes(6L))) ? 1 : 0) != 0).isTrue();
        Assertions.assertThat((String)valueFile).isEqualTo(JacksonMapper.ofIon().writeValueAsString(complexValue));
        Assertions.assertThat((String)((String)withMetadata.metadata().get("description"))).isEqualTo(description);
        kv.put(TEST_KV_KEY, new KVValueAndMetadata(new KVMetadata(null, Duration.ofMinutes(10L)), (Object)"some-value"));
        withMetadata = this.storageInterface.getWithMetadata("main", kv.namespace(), URI.create("/" + kv.namespace().replace(".", "/") + "/_kv/my-key.ion.v2"));
        valueFile = new String(withMetadata.inputStream().readAllBytes());
        expirationDate = Instant.parse((CharSequence)withMetadata.metadata().get("expirationDate"));
        Assertions.assertThat((expirationDate.isAfter(before.plus(Duration.ofMinutes(9L))) && expirationDate.isBefore(before.plus(Duration.ofMinutes(11L))) ? 1 : 0) != 0).isTrue();
        Assertions.assertThat((String)valueFile).isEqualTo("\"some-value\"");
    }

    @Test
    void should_delete_with_metadata() throws IOException {
        InternalKVStore kv = this.kv();
        kv.put(TEST_KV_KEY, new KVValueAndMetadata(new KVMetadata("description", Duration.ofMinutes(5L)), complexValue));
        URI uri = kv.storageUri(TEST_KV_KEY);
        URI metadataURI = URI.create(uri.getPath() + ".metadata");
        Assertions.assertThat((boolean)this.storageInterface.exists("main", kv.namespace(), uri)).isTrue();
        Assertions.assertThat((boolean)this.storageInterface.exists("main", kv.namespace(), metadataURI)).isTrue();
        boolean deleted = kv.delete(TEST_KV_KEY);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)deleted);
        Assertions.assertThat((boolean)this.storageInterface.exists("main", kv.namespace(), uri)).isTrue();
        Assertions.assertThat((boolean)this.storageInterface.exists("main", kv.namespace(), metadataURI)).isTrue();
    }

    @Test
    void should_delete_without_metadata() throws IOException {
        InternalKVStore kv = this.kv();
        kv.put(TEST_KV_KEY, new KVValueAndMetadata(null, complexValue));
        URI uri = kv.storageUri(TEST_KV_KEY);
        URI metadataURI = URI.create(uri.getPath() + ".metadata");
        Assertions.assertThat((boolean)this.storageInterface.exists("main", kv.namespace(), uri)).isTrue();
        Assertions.assertThat((boolean)this.storageInterface.exists("main", kv.namespace(), metadataURI)).isFalse();
        boolean deleted = kv.delete(TEST_KV_KEY);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)deleted);
        Assertions.assertThat((boolean)this.storageInterface.exists("main", kv.namespace(), uri)).isTrue();
    }

    @Test
    void shouldGetGivenEntryWithNullValue() throws IOException, ResourceExpiredException {
        InternalKVStore kv = this.kv();
        kv.put(TEST_KV_KEY, new KVValueAndMetadata(new KVMetadata(null, Duration.ofMinutes(5L)), null));
        Optional value = kv.getValue(TEST_KV_KEY);
        Assertions.assertThat((Optional)value).isEqualTo(Optional.of(new KVValue(null)));
    }

    @Test
    void shouldGetGivenEntryWithComplexValue() throws IOException, ResourceExpiredException {
        InternalKVStore kv = this.kv();
        kv.put(TEST_KV_KEY, new KVValueAndMetadata(new KVMetadata(null, Duration.ofMinutes(5L)), complexValue));
        Optional value = kv.getValue(TEST_KV_KEY);
        Assertions.assertThat((Object)((KVValue)value.get())).isEqualTo((Object)new KVValue(complexValue));
    }

    @Test
    void shouldGetEmptyGivenNonExistingKey() throws IOException, ResourceExpiredException {
        InternalKVStore kv = this.kv();
        Optional value = kv.getValue(TEST_KV_KEY);
        Assertions.assertThat((boolean)value.isEmpty()).isTrue();
    }

    @Test
    void shouldThrowGivenExpiredEntry() throws IOException {
        InternalKVStore kv = this.kv();
        kv.put(TEST_KV_KEY, new KVValueAndMetadata(new KVMetadata(null, Duration.ofNanos(1L)), complexValue));
        org.junit.jupiter.api.Assertions.assertThrows(ResourceExpiredException.class, () -> kv.getValue(TEST_KV_KEY));
    }

    @Test
    void shouldGetKVValueAndMetadata() throws IOException {
        InternalKVStore kv = this.kv();
        KVValueAndMetadata val = new KVValueAndMetadata(new KVMetadata(null, Duration.ofMinutes(5L)), complexValue);
        kv.put(TEST_KV_KEY, val);
        Optional result = kv.findMetadataAndValue(TEST_KV_KEY);
        org.junit.jupiter.api.Assertions.assertEquals((Object)val.value(), (Object)((KVValueAndMetadata)result.get()).value());
        org.junit.jupiter.api.Assertions.assertEquals((Object)val.metadata().getDescription(), (Object)((KVValueAndMetadata)result.get()).metadata().getDescription());
        org.junit.jupiter.api.Assertions.assertEquals((Object)val.metadata().getExpirationDate().truncatedTo(ChronoUnit.MILLIS), (Object)((KVValueAndMetadata)result.get()).metadata().getExpirationDate().truncatedTo(ChronoUnit.MILLIS));
    }

    @Test
    void getShouldStillWorkWithoutMetadata() throws IOException, ResourceExpiredException {
        InternalKVStore kv = this.kv();
        String key = IdUtils.create();
        URI kvStorageUri = URI.create("kestra://" + StorageContext.kvPrefix((String)kv.namespace()) + "/" + key + ".ion");
        String value = "someValue";
        KVValueAndMetadata kvValueAndMetadata = new KVValueAndMetadata(new KVMetadata("some description", Instant.now().plus(Duration.ofMinutes(5L))), (Object)value);
        this.storageInterface.put("main", kv.namespace(), kvStorageUri, new StorageObject(kvValueAndMetadata.metadataAsMap(), (InputStream)new ByteArrayInputStream(JacksonMapper.ofIon().writeValueAsBytes(kvValueAndMetadata.value()))));
        Optional result = kv.getValue(key);
        Assertions.assertThat((boolean)result.isPresent()).isTrue();
        Assertions.assertThat((Object)((KVValue)result.get()).value()).isEqualTo((Object)value);
    }

    @Test
    void illegalKey() {
        InternalKVStore kv = this.kv();
        String expectedErrorMessage = "Key must start with an alphanumeric character (uppercase or lowercase) and can contain alphanumeric characters (uppercase or lowercase), dots (.), underscores (_), and hyphens (-) only.";
        IllegalArgumentException illegalArgumentException = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> KVStore.validateKey((String)"a/b"));
        Assertions.assertThat((String)illegalArgumentException.getMessage()).isEqualTo(expectedErrorMessage);
        illegalArgumentException = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> kv.getValue("a/b"));
        Assertions.assertThat((String)illegalArgumentException.getMessage()).isEqualTo(expectedErrorMessage);
        illegalArgumentException = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> kv.put("a/b", new KVValueAndMetadata(new KVMetadata(null, Duration.ofMinutes(5L)), (Object)"content")));
        Assertions.assertThat((String)illegalArgumentException.getMessage()).isEqualTo(expectedErrorMessage);
        illegalArgumentException = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> kv.delete("a/b"));
        Assertions.assertThat((String)illegalArgumentException.getMessage()).isEqualTo(expectedErrorMessage);
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(() -> KVStore.validateKey((String)"AN_UPPER.CASE-key"));
    }

    private InternalKVStore kv() {
        String namespaceId = "io.kestra." + IdUtils.create();
        return new InternalKVStore("main", namespaceId, this.storageInterface, this.kvMetadataRepository);
    }
}

