/*
 * Decompiled with CFR 0.152.
 */
package io.kestra.core.storage;

import com.google.common.io.CharStreams;
import io.kestra.core.junit.annotations.KestraTest;
import io.kestra.core.storages.FileAttributes;
import io.kestra.core.storages.StorageInterface;
import io.kestra.core.storages.StorageObject;
import io.kestra.core.utils.IdUtils;
import io.kestra.core.utils.Rethrow;
import jakarta.inject.Inject;
import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.Temporal;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.AssertionsForClassTypes;
import org.assertj.core.data.TemporalOffset;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Test;

@KestraTest
public abstract class StorageTestSuite {
    private static final String CONTENT_STRING = "Content";
    @Inject
    protected StorageInterface storageInterface;

    @Test
    void getPath() {
        String path = this.storageInterface.getPath(null);
        AssertionsForClassTypes.assertThat((String)path).isEqualTo("");
        path = this.storageInterface.getPath(URI.create("/folder1/folder2"));
        AssertionsForClassTypes.assertThat((String)path).isEqualTo("folder1/folder2");
        path = this.storageInterface.getPath("main", null);
        AssertionsForClassTypes.assertThat((String)path).isEqualTo("main/");
        path = this.storageInterface.getPath("main", URI.create("/folder1/folder2"));
        AssertionsForClassTypes.assertThat((String)path).isEqualTo("main/folder1/folder2");
    }

    @Test
    void get() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.get(tenantId, prefix);
    }

    @Test
    void getNoCrossTenant() throws Exception {
        String prefix = IdUtils.create();
        String fistTenant = IdUtils.create();
        String secondTenant = IdUtils.create();
        String fistTenantPath = "/" + prefix + "/storage/firstTenant.yml";
        this.putFile(fistTenant, fistTenantPath);
        String secondTenantPath = "/" + prefix + "/storage/secondTenant.yml";
        this.putFile(secondTenant, secondTenantPath);
        URI fistTenantUri = new URI(fistTenantPath);
        InputStream get = this.storageInterface.get(fistTenant, prefix, fistTenantUri);
        Assertions.assertThat((String)CharStreams.toString((Readable)new InputStreamReader(get))).isEqualTo(CONTENT_STRING);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.storageInterface.exists(fistTenant, prefix, fistTenantUri));
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.get(secondTenant, null, fistTenantUri));
        URI secondTenantUri = new URI(secondTenantPath);
        get = this.storageInterface.get(secondTenant, prefix, secondTenantUri);
        Assertions.assertThat((String)CharStreams.toString((Readable)new InputStreamReader(get))).isEqualTo(CONTENT_STRING);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.storageInterface.exists(secondTenant, prefix, secondTenantUri));
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.get(fistTenant, null, secondTenantUri));
    }

    @Test
    void getWithScheme() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.putFile(tenantId, "/" + prefix + "/storage/get.yml");
        InputStream getScheme = this.storageInterface.get(tenantId, prefix, new URI("kestra:///" + prefix + "/storage/get.yml"));
        Assertions.assertThat((String)CharStreams.toString((Readable)new InputStreamReader(getScheme))).isEqualTo(CONTENT_STRING);
    }

    private void get(String tenantId, String prefix) throws Exception {
        this.putFile(tenantId, "/" + prefix + "/storage/get.yml");
        this.putFile(tenantId, "/" + prefix + "/storage/level2/2.yml");
        URI item = new URI("/" + prefix + "/storage/get.yml");
        InputStream get = this.storageInterface.get(tenantId, prefix, item);
        Assertions.assertThat((String)CharStreams.toString((Readable)new InputStreamReader(get))).isEqualTo(CONTENT_STRING);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.storageInterface.exists(tenantId, prefix, item));
    }

    @Test
    void getNoTraversal() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.putFile(tenantId, "/" + prefix + "/storage/get.yml");
        this.putFile(tenantId, "/" + prefix + "/storage/level2/2.yml");
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.storageInterface.get(tenantId, prefix, new URI("kestra:///" + prefix + "/storage/level2/../get.yml")));
    }

    @Test
    void getFileNotFound() {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.get(tenantId, prefix, new URI("/" + prefix + "/storage/missing.yml")));
    }

    @Test
    void getInstanceResource() throws Exception {
        String prefix = IdUtils.create();
        this.putInstanceFile("/" + prefix + "/storage/get.yml");
        this.putInstanceFile("/" + prefix + "/storage/level2/2.yml");
        URI item = new URI("/" + prefix + "/storage/get.yml");
        InputStream get = this.storageInterface.getInstanceResource(prefix, item);
        Assertions.assertThat((String)CharStreams.toString((Readable)new InputStreamReader(get))).isEqualTo(CONTENT_STRING);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.storageInterface.existsInstanceResource(prefix, item));
    }

    @Test
    void filesByPrefix() throws IOException {
        String namespaceName = "filesByPrefix_test_namespace";
        this.storageInterface.put("main", namespaceName, URI.create("/filesByPrefix_test_namespace/file.txt"), (InputStream)new ByteArrayInputStream(new byte[0]));
        this.storageInterface.put("tenant", namespaceName, URI.create("/filesByPrefix_test_namespace/tenant_file.txt"), (InputStream)new ByteArrayInputStream(new byte[0]));
        this.storageInterface.put("main", namespaceName, URI.create("/filesByPrefix_test_namespace/another_file.json"), (InputStream)new ByteArrayInputStream(new byte[0]));
        this.storageInterface.put("main", namespaceName, URI.create("/filesByPrefix_test_namespace/folder/file.txt"), (InputStream)new ByteArrayInputStream(new byte[0]));
        this.storageInterface.put("main", namespaceName, URI.create("/filesByPrefix_test_namespace/folder/some.yaml"), (InputStream)new ByteArrayInputStream(new byte[0]));
        this.storageInterface.put("main", namespaceName, URI.create("/filesByPrefix_test_namespace/folder/sub/script.py"), (InputStream)new ByteArrayInputStream(new byte[0]));
        List res = this.storageInterface.allByPrefix("main", namespaceName, URI.create("kestra:///filesByPrefix_test_namespace/"), false);
        Assertions.assertThat((List)res).containsExactlyInAnyOrder((Object[])new URI[]{URI.create("kestra:///filesByPrefix_test_namespace/file.txt"), URI.create("kestra:///filesByPrefix_test_namespace/another_file.json"), URI.create("kestra:///filesByPrefix_test_namespace/folder/file.txt"), URI.create("kestra:///filesByPrefix_test_namespace/folder/some.yaml"), URI.create("kestra:///filesByPrefix_test_namespace/folder/sub/script.py")});
        res = this.storageInterface.allByPrefix("tenant", namespaceName, URI.create("/filesByPrefix_test_namespace"), false);
        Assertions.assertThat((List)res).containsExactlyInAnyOrder((Object[])new URI[]{URI.create("kestra:///filesByPrefix_test_namespace/tenant_file.txt")});
        res = this.storageInterface.allByPrefix("main", namespaceName, URI.create("/filesByPrefix_test_namespace/folder"), false);
        Assertions.assertThat((List)res).containsExactlyInAnyOrder((Object[])new URI[]{URI.create("kestra:///filesByPrefix_test_namespace/folder/file.txt"), URI.create("kestra:///filesByPrefix_test_namespace/folder/some.yaml"), URI.create("kestra:///filesByPrefix_test_namespace/folder/sub/script.py")});
        res = this.storageInterface.allByPrefix("main", namespaceName, URI.create("/filesByPrefix_test_namespace/folder/sub"), false);
        Assertions.assertThat((List)res).containsExactlyInAnyOrder((Object[])new URI[]{URI.create("kestra:///filesByPrefix_test_namespace/folder/sub/script.py")});
        res = this.storageInterface.allByPrefix("main", namespaceName, URI.create("/filesByPrefix_test_namespace/non-existing"), false);
        Assertions.assertThat((List)res).isEmpty();
    }

    @Test
    void objectsByPrefix() throws IOException {
        this.storageInterface.put("main", "some_namespace", URI.create("/some_namespace/file.txt"), (InputStream)new ByteArrayInputStream(new byte[0]));
        this.storageInterface.put("tenant", "some_namespace", URI.create("/some_namespace/tenant_file.txt"), (InputStream)new ByteArrayInputStream(new byte[0]));
        this.storageInterface.createDirectory("main", "some_namespace", URI.create("/some_namespace/folder/sub"));
        List res = this.storageInterface.allByPrefix("main", "some_namespace", URI.create("kestra:///some_namespace/"), true);
        Assertions.assertThat((List)res).containsExactlyInAnyOrder((Object[])new URI[]{URI.create("kestra:///some_namespace/file.txt"), URI.create("kestra:///some_namespace/folder/"), URI.create("kestra:///some_namespace/folder/sub/")});
        res = this.storageInterface.allByPrefix("tenant", "some_namespace", URI.create("/some_namespace"), true);
        Assertions.assertThat((List)res).containsExactlyInAnyOrder((Object[])new URI[]{URI.create("kestra:///some_namespace/tenant_file.txt")});
        res = this.storageInterface.allByPrefix("main", "some_namespace", URI.create("/some_namespace/folder"), true);
        Assertions.assertThat((List)res).containsExactlyInAnyOrder((Object[])new URI[]{URI.create("kestra:///some_namespace/folder/sub/")});
    }

    @Test
    void list() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.list(prefix, tenantId);
    }

    @Test
    void listNoTraversal() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2/1.yml", "/" + prefix + "/storage/another/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId, (String)s)));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.storageInterface.list(tenantId, prefix, new URI("/" + prefix + "/storage/level2/..")));
    }

    @Test
    void listNotFound() {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.list(tenantId, prefix, new URI("/" + prefix + "/storage/")));
    }

    @Test
    void listNoCrossTenant() throws Exception {
        String prefix = IdUtils.create();
        String tenantId1 = IdUtils.create();
        String tenantId2 = IdUtils.create();
        List<String> firstTenant = Arrays.asList("/" + prefix + "/with/1.yml", "/" + prefix + "/with/2.yml", "/" + prefix + "/with/3.yml");
        firstTenant.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId1, (String)s)));
        List<String> secondTenant = Arrays.asList("/" + prefix + "/notenant/1.yml", "/" + prefix + "/notenant/2.yml", "/" + prefix + "/notenant/3.yml");
        secondTenant.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId2, (String)s)));
        List with = this.storageInterface.list(tenantId1, prefix, new URI("/" + prefix + "/with"));
        Assertions.assertThat(with.stream().map(FileAttributes::getFileName).toList()).containsExactlyInAnyOrder((Object[])new String[]{"1.yml", "2.yml", "3.yml"});
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.list(tenantId1, prefix, new URI("/" + prefix + "/notenant/")));
        List notenant = this.storageInterface.list(tenantId2, prefix, new URI("/" + prefix + "/notenant"));
        Assertions.assertThat(notenant.stream().map(FileAttributes::getFileName).toList()).containsExactlyInAnyOrder((Object[])new String[]{"1.yml", "2.yml", "3.yml"});
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.list(tenantId2, prefix, new URI("/" + prefix + "/with/")));
    }

    @Test
    void listWithScheme() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2/1.yml", "/" + prefix + "/storage/another/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId, (String)s)));
        List list = this.storageInterface.list(tenantId, prefix, new URI("kestra:///" + prefix + "/storage"));
        Assertions.assertThat(list.stream().map(FileAttributes::getFileName).toList()).containsExactlyInAnyOrder((Object[])new String[]{"root.yml", "level1", "another"});
    }

    private void list(String prefix, String tenantId) throws Exception {
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2/1.yml", "/" + prefix + "/storage/another/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId, (String)s, Map.of("someMetadata", "someValue"))));
        List list = this.storageInterface.list(tenantId, prefix, null);
        Assertions.assertThat(list.stream().map(FileAttributes::getFileName).toList()).contains((Object[])new String[]{prefix});
        list = this.storageInterface.list(tenantId, prefix, new URI("/" + prefix + "/storage"));
        Assertions.assertThat(list.stream().map(FileAttributes::getFileName).toList()).containsExactlyInAnyOrder((Object[])new String[]{"root.yml", "level1", "another"});
        Assertions.assertThat((Map)list.stream().filter(f -> f.getFileName().equals("root.yml")).findFirst().get().getMetadata()).containsEntry((Object)"someMetadata", (Object)"someValue");
    }

    @Test
    void listInstanceResouces() throws Exception {
        String prefix = IdUtils.create();
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2/1.yml", "/" + prefix + "/storage/another/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putInstanceFile((String)s, Map.of("someMetadata", "someValue"))));
        List list = this.storageInterface.listInstanceResource(prefix, null);
        Assertions.assertThat(list.stream().map(FileAttributes::getFileName).toList()).contains((Object[])new String[]{prefix});
        list = this.storageInterface.listInstanceResource(prefix, new URI("/" + prefix + "/storage"));
        Assertions.assertThat(list.stream().map(FileAttributes::getFileName).toList()).containsExactlyInAnyOrder((Object[])new String[]{"root.yml", "level1", "another"});
        Assertions.assertThat((Map)list.stream().filter(f -> f.getFileName().equals("root.yml")).findFirst().get().getMetadata()).containsEntry((Object)"someMetadata", (Object)"someValue");
    }

    @Test
    void exists() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.exists(prefix, tenantId);
    }

    private void exists(String prefix, String tenantId) throws Exception {
        this.putFile(tenantId, "/" + prefix + "/storage/put.yml");
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/put.yml"))).isTrue();
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/notfound.yml"))).isFalse();
    }

    @Test
    void existsNoTraversal() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2/1.yml", "/" + prefix + "/storage/another/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId, (String)s)));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/level2/..")));
    }

    @Test
    void existsNoCrossTenant() throws Exception {
        String prefix = IdUtils.create();
        String tenantId1 = IdUtils.create();
        String tenantId2 = IdUtils.create();
        String firstTenant = "/" + prefix + "/storage/firstTenant.yml";
        this.putFile(tenantId1, firstTenant);
        String secondTenant = "/" + prefix + "/storage/secondTenant.yml";
        this.putFile(tenantId2, secondTenant);
        URI with = new URI(firstTenant);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.storageInterface.exists(tenantId1, prefix, with));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.storageInterface.exists(tenantId2, prefix, with));
        URI without = new URI(secondTenant);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.storageInterface.exists(tenantId1, prefix, without));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.storageInterface.exists(tenantId2, prefix, without));
    }

    @Test
    void existsWithScheme() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.putFile(tenantId, "/" + prefix + "/storage/get.yml");
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.storageInterface.exists(tenantId, prefix, new URI("kestra:///" + prefix + "/storage/get.yml")));
    }

    @Test
    void existsInstanceResource() throws Exception {
        String prefix = IdUtils.create();
        this.putInstanceFile("/" + prefix + "/storage/put.yml");
        Assertions.assertThat((boolean)this.storageInterface.existsInstanceResource(prefix, new URI("/" + prefix + "/storage/put.yml"))).isTrue();
        Assertions.assertThat((boolean)this.storageInterface.existsInstanceResource(prefix, new URI("/" + prefix + "/storage/notfound.yml"))).isFalse();
    }

    @Test
    void size() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.size(prefix, tenantId);
    }

    private void size(String prefix, String tenantId) throws Exception {
        URI put = this.putFile(tenantId, "/" + prefix + "/storage/put.yml");
        Assertions.assertThat((long)this.storageInterface.getAttributes(tenantId, prefix, new URI("/" + prefix + "/storage/put.yml")).getSize()).isEqualTo((long)CONTENT_STRING.length());
    }

    @Test
    void sizeNoTraversal() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId, (String)s)));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.storageInterface.getAttributes(tenantId, prefix, new URI("/" + prefix + "/storage/level2/../1.yml")).getSize());
    }

    @Test
    void sizeNotFound() {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.getAttributes(tenantId, prefix, new URI("/" + prefix + "/storage/")).getSize());
    }

    @Test
    void sizeNoCrossTenant() throws Exception {
        String prefix = IdUtils.create();
        String tenantId1 = IdUtils.create();
        String tenantId2 = IdUtils.create();
        String firstTenant = "/" + prefix + "/storage/firstTenant.yml";
        this.putFile(tenantId1, firstTenant);
        String secondTenant = "/" + prefix + "/storage/secondTenant.yml";
        this.putFile(tenantId2, secondTenant);
        URI with = new URI(firstTenant);
        Assertions.assertThat((long)this.storageInterface.getAttributes(tenantId1, prefix, with).getSize()).isEqualTo((long)CONTENT_STRING.length());
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.getAttributes(tenantId2, prefix, with).getSize());
        URI without = new URI(secondTenant);
        Assertions.assertThat((long)this.storageInterface.getAttributes(tenantId2, prefix, without).getSize()).isEqualTo((long)CONTENT_STRING.length());
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.getAttributes(tenantId1, prefix, without).getSize());
    }

    @Test
    void sizeWithScheme() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.putFile(tenantId, "/" + prefix + "/storage/get.yml");
        Assertions.assertThat((long)this.storageInterface.getAttributes(tenantId, prefix, new URI("kestra:///" + prefix + "/storage/get.yml")).getSize()).isEqualTo((long)CONTENT_STRING.length());
    }

    @Test
    void lastModifiedTime() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.lastModifiedTime(prefix, tenantId);
    }

    private void lastModifiedTime(String prefix, String tenantId) throws Exception {
        this.putFile(tenantId, "/" + prefix + "/storage/put.yml");
        Assertions.assertThat((long)this.storageInterface.getAttributes(tenantId, prefix, new URI("/" + prefix + "/storage/put.yml")).getLastModifiedTime()).isNotNull();
    }

    @Test
    void lastModifiedTimeNoTraversal() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId, (String)s)));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.storageInterface.getAttributes(tenantId, prefix, new URI("/" + prefix + "/storage/level2/../1.yml")).getLastModifiedTime());
    }

    @Test
    void lastModifiedTimeNotFound() {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.getAttributes(tenantId, prefix, new URI("/" + prefix + "/storage/")).getLastModifiedTime());
    }

    @Test
    void lastModifiedTimeNoCrossTenant() throws Exception {
        String prefix = IdUtils.create();
        String tenantId1 = IdUtils.create();
        String tenantId2 = IdUtils.create();
        String firstTenant = "/" + prefix + "/storage/firstTenant.yml";
        this.putFile(tenantId1, firstTenant);
        String secondTenant = "/" + prefix + "/storage/secondTenant.yml";
        this.putFile(tenantId2, secondTenant);
        URI with = new URI(firstTenant);
        Assertions.assertThat((long)this.storageInterface.getAttributes(tenantId1, prefix, with).getLastModifiedTime()).isNotNull();
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.getAttributes(tenantId2, prefix, with).getLastModifiedTime());
        URI without = new URI(secondTenant);
        Assertions.assertThat((long)this.storageInterface.getAttributes(tenantId2, prefix, without).getLastModifiedTime()).isNotNull();
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.getAttributes(tenantId1, prefix, without).getLastModifiedTime());
    }

    @Test
    void lastModifiedTimeWithScheme() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.putFile(tenantId, "/" + prefix + "/storage/get.yml");
        Assertions.assertThat((long)this.storageInterface.getAttributes(tenantId, prefix, new URI("kestra:///" + prefix + "/storage/get.yml")).getLastModifiedTime()).isNotNull();
    }

    @Test
    void getAttributes() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.getAttributes(prefix, tenantId);
    }

    private void getAttributes(String prefix, String tenantId) throws Exception {
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId, (String)s)));
        FileAttributes attr = this.storageInterface.getAttributes(tenantId, prefix, new URI("/" + prefix + "/storage/root.yml"));
        StorageTestSuite.compareFileAttribute(attr);
        attr = this.storageInterface.getAttributes(tenantId, prefix, new URI("/" + prefix + "/storage/level1"));
        StorageTestSuite.compareDirectory(attr);
    }

    @Test
    void getAttributesNoTraversal() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId, (String)s)));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.storageInterface.getAttributes(tenantId, prefix, new URI("/" + prefix + "/storage/level2/../1.yml")));
    }

    @Test
    void getAttributesNotFound() {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.getAttributes(tenantId, prefix, new URI("/" + prefix + "/storage/")));
    }

    @Test
    void getAttributesNoCrossTenant() throws Exception {
        String prefix = IdUtils.create();
        String tenantId1 = IdUtils.create();
        String tenantId2 = IdUtils.create();
        String firstTenant = "/" + prefix + "/storage/firstTenant.yml";
        this.putFile(tenantId1, firstTenant);
        String secondTenant = "/" + prefix + "/storage/secondTenant.yml";
        this.putFile(tenantId2, secondTenant);
        URI with = new URI(firstTenant);
        FileAttributes attr = this.storageInterface.getAttributes(tenantId1, prefix, with);
        Assertions.assertThat((String)attr.getFileName()).isEqualTo("firstTenant.yml");
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.getAttributes(tenantId2, prefix, with));
        URI without = new URI(secondTenant);
        attr = this.storageInterface.getAttributes(tenantId2, prefix, without);
        Assertions.assertThat((String)attr.getFileName()).isEqualTo("secondTenant.yml");
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.getAttributes(tenantId1, prefix, without));
    }

    @Test
    void getAttributesWithScheme() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.putFile(tenantId, "/" + prefix + "/storage/get.yml");
        FileAttributes attr = this.storageInterface.getAttributes(tenantId, prefix, new URI("kestra:///" + prefix + "/storage/get.yml"));
        Assertions.assertThat((String)attr.getFileName()).isEqualTo("get.yml");
    }

    @Test
    void getInstanceAttributes() throws Exception {
        String prefix = IdUtils.create();
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml");
        path.forEach(Rethrow.throwConsumer(this::putInstanceFile));
        FileAttributes attr = this.storageInterface.getInstanceAttributes(prefix, new URI("/" + prefix + "/storage/root.yml"));
        StorageTestSuite.compareFileAttribute(attr);
        attr = this.storageInterface.getInstanceAttributes(prefix, new URI("/" + prefix + "/storage/level1"));
        StorageTestSuite.compareDirectory(attr);
    }

    private static void compareDirectory(FileAttributes attr) {
        Assertions.assertThat((String)attr.getFileName()).isEqualTo("level1");
        Assertions.assertThat((Comparable)attr.getType()).isEqualTo((Object)FileAttributes.FileType.Directory);
        Instant lastModifiedInstant = Instant.ofEpochMilli(attr.getLastModifiedTime());
        Assertions.assertThat((Instant)lastModifiedInstant).isCloseTo((Temporal)Instant.now(), (TemporalOffset)Assertions.within((Duration)Duration.ofSeconds(10L)));
        Instant creationInstant = Instant.ofEpochMilli(attr.getCreationTime());
        Assertions.assertThat((Instant)creationInstant).isCloseTo((Temporal)Instant.now(), (TemporalOffset)Assertions.within((Duration)Duration.ofSeconds(10L)));
    }

    private static void compareFileAttribute(FileAttributes attr) {
        Assertions.assertThat((String)attr.getFileName()).isEqualTo("root.yml");
        Assertions.assertThat((Comparable)attr.getType()).isEqualTo((Object)FileAttributes.FileType.File);
        Assertions.assertThat((long)attr.getSize()).isEqualTo((long)CONTENT_STRING.length());
        Instant lastModifiedInstant = Instant.ofEpochMilli(attr.getLastModifiedTime());
        Assertions.assertThat((Instant)lastModifiedInstant).isCloseTo((Temporal)Instant.now(), (TemporalOffset)Assertions.within((Duration)Duration.ofSeconds(10L)));
        Instant creationInstant = Instant.ofEpochMilli(attr.getCreationTime());
        Assertions.assertThat((Instant)creationInstant).isCloseTo((Temporal)Instant.now(), (TemporalOffset)Assertions.within((Duration)Duration.ofSeconds(10L)));
    }

    @Test
    void put() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.put(tenantId, prefix);
    }

    @Test
    void put_PathWithTenantStringInIt() throws Exception {
        String tenantId = IdUtils.create();
        String prefix = tenantId + "/" + IdUtils.create();
        this.put(tenantId, prefix);
    }

    @Test
    void putFromAnotherFile() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.put(tenantId, prefix);
        URI putFromAnother = this.storageInterface.put(tenantId, prefix, new URI("/" + prefix + "/storage/put_from_another.yml"), this.storageInterface.get(tenantId, prefix, new URI("/" + prefix + "/storage/put.yml")));
        Assertions.assertThat((String)putFromAnother.toString()).isEqualTo(new URI("kestra:///" + prefix + "/storage/put_from_another.yml").toString());
        InputStream get = this.storageInterface.get(tenantId, prefix, new URI("/" + prefix + "/storage/put_from_another.yml"));
        Assertions.assertThat((String)CharStreams.toString((Readable)new InputStreamReader(get))).isEqualTo(CONTENT_STRING);
    }

    @Test
    void putWithScheme() throws URISyntaxException, IOException {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        URI uri = new URI("kestra:///" + prefix + "/storage/get.yml");
        this.storageInterface.put(tenantId, prefix, uri, (InputStream)new ByteArrayInputStream(CONTENT_STRING.getBytes()));
        InputStream getScheme = this.storageInterface.get(tenantId, prefix, new URI("/" + prefix + "/storage/get.yml"));
        Assertions.assertThat((String)CharStreams.toString((Readable)new InputStreamReader(getScheme))).isEqualTo(CONTENT_STRING);
    }

    @Test
    void putNoTraversal() throws URISyntaxException, IOException {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.storageInterface.createDirectory(tenantId, prefix, new URI("/" + prefix + "/storage/level1"));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.storageInterface.put(tenantId, prefix, new URI("kestra:///" + prefix + "/storage/level1/../get2.yml"), (InputStream)new ByteArrayInputStream(CONTENT_STRING.getBytes())));
    }

    private void put(String tenantId, String prefix) throws Exception {
        URI put = this.putFile(tenantId, "/" + prefix + "/storage/put.yml");
        InputStream get = this.storageInterface.get(tenantId, prefix, new URI("/" + prefix + "/storage/put.yml"));
        Assertions.assertThat((String)put.toString()).isEqualTo(new URI("kestra:///" + prefix + "/storage/put.yml").toString());
        Assertions.assertThat((String)CharStreams.toString((Readable)new InputStreamReader(get))).isEqualTo(CONTENT_STRING);
    }

    @Test
    void putInstanceResource() throws Exception {
        String prefix = IdUtils.create();
        URI put = this.putInstanceFile("/" + prefix + "/storage/put.yml");
        InputStream get = this.storageInterface.getInstanceResource(prefix, new URI("/" + prefix + "/storage/put.yml"));
        Assertions.assertThat((String)put.toString()).isEqualTo(new URI("kestra:///" + prefix + "/storage/put.yml").toString());
        Assertions.assertThat((String)CharStreams.toString((Readable)new InputStreamReader(get))).isEqualTo(CONTENT_STRING);
    }

    @Test
    void delete() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.delete(prefix, tenantId);
    }

    @Test
    void deleteNoTraversal() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2/1.yml", "/" + prefix + "/storage/another/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId, (String)s)));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.storageInterface.delete(tenantId, prefix, new URI("/" + prefix + "/storage/level2/../1.yml")));
    }

    @Test
    void deleteNotFound() throws URISyntaxException, IOException {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        Assertions.assertThat((boolean)this.storageInterface.delete(tenantId, prefix, new URI("/" + prefix + "/storage/"))).isFalse();
    }

    private void delete(String prefix, String tenantId) throws Exception {
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level12.yml", "/" + prefix + "/storage/file", "/" + prefix + "/storage/file.txt", "/" + prefix + "/storage/level1/level2/1.yml", "/" + prefix + "/storage/another/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId, (String)s)));
        boolean deleted = this.storageInterface.delete(tenantId, prefix, new URI("/" + prefix + "/storage/level1"));
        Assertions.assertThat((boolean)deleted).isTrue();
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/root.yml"))).isTrue();
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/another/1.yml"))).isTrue();
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/level1"))).isFalse();
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/level12.yml"))).isTrue();
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/level1/1.yml"))).isFalse();
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/level1/level2/1.yml"))).isFalse();
        deleted = this.storageInterface.delete(tenantId, prefix, new URI("/" + prefix + "/storage/root.yml"));
        Assertions.assertThat((boolean)deleted).isTrue();
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/root.yml"))).isFalse();
        deleted = this.storageInterface.delete(tenantId, prefix, new URI("/" + prefix + "/storage/file"));
        Assertions.assertThat((boolean)deleted).isTrue();
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/file"))).isFalse();
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/file.txt"))).isTrue();
    }

    @Test
    void deleteWithScheme() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.putFile(tenantId, "/" + prefix + "/storage/get.yml");
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.storageInterface.delete(tenantId, prefix, new URI("kestra:///" + prefix + "/storage/get.yml")));
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/get.yml"))).isFalse();
    }

    @Test
    void deleteInstanceResource() throws Exception {
        String prefix = IdUtils.create();
        List<String> paths = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2/1.yml");
        paths.forEach(Rethrow.throwConsumer(this::putInstanceFile));
        Assertions.assertThat((boolean)this.storageInterface.existsInstanceResource(prefix, new URI("/" + prefix + "/storage/root.yml"))).isTrue();
        Assertions.assertThat((boolean)this.storageInterface.existsInstanceResource(prefix, new URI("/" + prefix + "/storage/level1/1.yml"))).isTrue();
        Assertions.assertThat((boolean)this.storageInterface.existsInstanceResource(prefix, new URI("/" + prefix + "/storage/level1/level2/1.yml"))).isTrue();
        boolean deleted = this.storageInterface.deleteInstanceResource(prefix, new URI("/" + prefix + "/storage/level1"));
        Assertions.assertThat((boolean)deleted).isTrue();
        Assertions.assertThat((boolean)this.storageInterface.existsInstanceResource(prefix, new URI("/" + prefix + "/storage/root.yml"))).isTrue();
        Assertions.assertThat((boolean)this.storageInterface.existsInstanceResource(prefix, new URI("/" + prefix + "/storage/level1/1.yml"))).isFalse();
        Assertions.assertThat((boolean)this.storageInterface.existsInstanceResource(prefix, new URI("/" + prefix + "/storage/level1/level2/1.yml"))).isFalse();
    }

    @Test
    void createDirectory() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.createDirectory(prefix, tenantId);
    }

    @Test
    void createDirectoryNoTraversal() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2/1.yml", "/" + prefix + "/storage/another/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId, (String)s)));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.storageInterface.createDirectory(tenantId, prefix, new URI("/" + prefix + "/storage/level2/../newdir")));
    }

    private void createDirectory(String prefix, String tenantId) throws Exception {
        this.storageInterface.createDirectory(tenantId, prefix, new URI("/" + prefix + "/storage/level1"));
        FileAttributes attr = this.storageInterface.getAttributes(tenantId, prefix, new URI("/" + prefix + "/storage/level1"));
        Assertions.assertThat((String)attr.getFileName()).isEqualTo("level1");
        Assertions.assertThat((Comparable)attr.getType()).isEqualTo((Object)FileAttributes.FileType.Directory);
        Assertions.assertThat((long)attr.getLastModifiedTime()).isNotNull();
    }

    @Test
    void createDirectoryWithScheme() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.storageInterface.createDirectory(tenantId, prefix, new URI("kestra:///" + prefix + "/storage/level1"));
        FileAttributes attr = this.storageInterface.getAttributes(tenantId, prefix, new URI("/" + prefix + "/storage/level1"));
        Assertions.assertThat((String)attr.getFileName()).isEqualTo("level1");
        Assertions.assertThat((Comparable)attr.getType()).isEqualTo((Object)FileAttributes.FileType.Directory);
        Assertions.assertThat((long)attr.getLastModifiedTime()).isNotNull();
    }

    @Test
    void createDirectoryShouldBeRecursive() throws IOException {
        String prefix = IdUtils.create();
        this.storageInterface.createDirectory("main", prefix, URI.create("/" + prefix + "/first/second/third"));
        List list = this.storageInterface.list("main", prefix, URI.create("/" + prefix));
        MatcherAssert.assertThat((Object)list, (Matcher)Matchers.contains((Matcher)Matchers.hasProperty((String)"fileName", (Matcher)Matchers.is((Object)"first"))));
    }

    @Test
    void createInstanceDirectory() throws Exception {
        String prefix = IdUtils.create();
        this.storageInterface.createInstanceDirectory(prefix, new URI("/" + prefix + "/storage/level1"));
        FileAttributes attr = this.storageInterface.getInstanceAttributes(prefix, new URI("/" + prefix + "/storage/level1"));
        Assertions.assertThat((String)attr.getFileName()).isEqualTo("level1");
        Assertions.assertThat((Comparable)attr.getType()).isEqualTo((Object)FileAttributes.FileType.Directory);
        Assertions.assertThat((long)attr.getLastModifiedTime()).isNotNull();
    }

    @Test
    void move() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.move(prefix, tenantId);
    }

    @Test
    void moveNotFound() {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.move(tenantId, prefix, new URI("/" + prefix + "/storage/"), new URI("/" + prefix + "/test/")));
    }

    @Test
    void moveNoTraversal() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2/1.yml", "/" + prefix + "/storage/another/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId, (String)s)));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.storageInterface.move(tenantId, prefix, new URI("/" + prefix + "/storage/level2/../1.yml"), new URI("/" + prefix + "/storage/level2/1.yml")));
    }

    private void move(String prefix, String tenantId) throws Exception {
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2/2.yml", "/" + prefix + "/storage/another/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId, (String)s)));
        this.storageInterface.move(tenantId, prefix, new URI("/" + prefix + "/storage/level1"), new URI("/" + prefix + "/storage/moved"));
        List list = this.storageInterface.list(tenantId, prefix, new URI("/" + prefix + "/storage/moved"));
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/level1"))).isFalse();
        Assertions.assertThat(list.stream().map(FileAttributes::getFileName).toList()).containsExactlyInAnyOrder((Object[])new String[]{"level2", "1.yml"});
        list = this.storageInterface.list(tenantId, prefix, new URI("/" + prefix + "/storage/moved/level2"));
        Assertions.assertThat(list.stream().map(FileAttributes::getFileName).toList()).containsExactlyInAnyOrder((Object[])new String[]{"2.yml"});
        this.storageInterface.move(tenantId, prefix, new URI("/" + prefix + "/storage/root.yml"), new URI("/" + prefix + "/storage/root-moved.yml"));
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/root.yml"))).isFalse();
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/root-moved.yml"))).isTrue();
    }

    @Test
    void moveWithScheme() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.putFile(tenantId, "/" + prefix + "/storage/root.yml");
        this.storageInterface.move(tenantId, prefix, new URI("kestra:///" + prefix + "/storage/root.yml"), new URI("kestra:///" + prefix + "/storage/root-moved.yml"));
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/root.yml"))).isFalse();
        Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI("/" + prefix + "/storage/root-moved.yml"))).isTrue();
    }

    @Test
    void deleteByPrefix() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        this.deleteByPrefix(prefix, tenantId);
    }

    @Test
    void deleteByPrefix_PathWithTenantStringInIt() throws Exception {
        String tenantId = IdUtils.create();
        String prefix = tenantId + "/" + IdUtils.create();
        this.deleteByPrefix(prefix, tenantId);
    }

    @Test
    void deleteByPrefixNotFound() throws URISyntaxException, IOException {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        Assertions.assertThat((List)this.storageInterface.deleteByPrefix(tenantId, prefix, new URI("/" + prefix + "/storage/"))).containsExactlyInAnyOrder((Object[])new URI[0]);
    }

    @Test
    void deleteByPrefixNoTraversal() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2/1.yml", "/" + prefix + "/storage/another/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId, (String)s)));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.storageInterface.move(tenantId, prefix, new URI("/" + prefix + "/storage/level2/../1.yml"), new URI("/" + prefix + "/storage/level2/1.yml")));
    }

    private void deleteByPrefix(String prefix, String tenantId) throws Exception {
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId, (String)s)));
        List deleted = this.storageInterface.deleteByPrefix(tenantId, prefix, new URI("/" + prefix + "/storage/"));
        List<String> res = Arrays.asList("/" + prefix + "/storage", "/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2", "/" + prefix + "/storage/level1/level2/1.yml");
        Assertions.assertThat((List)deleted).containsExactlyInAnyOrder((Object[])((URI[])res.stream().map(s -> URI.create("kestra://" + s)).toArray(URI[]::new)));
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.get(tenantId, prefix, new URI("/" + prefix + "/storage/")));
        path.forEach(Rethrow.throwConsumer(s -> Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI((String)s))).isFalse()));
    }

    @Test
    void deleteByPrefixWithScheme() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        List<String> path = Arrays.asList("/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2/1.yml");
        path.forEach(Rethrow.throwConsumer(s -> this.putFile(tenantId, (String)s)));
        List deleted = this.storageInterface.deleteByPrefix(tenantId, prefix, new URI("/" + prefix + "/storage/"));
        List<String> res = Arrays.asList("/" + prefix + "/storage", "/" + prefix + "/storage/root.yml", "/" + prefix + "/storage/level1", "/" + prefix + "/storage/level1/1.yml", "/" + prefix + "/storage/level1/level2", "/" + prefix + "/storage/level1/level2/1.yml");
        Assertions.assertThat((List)deleted).containsExactlyInAnyOrder((Object[])((URI[])res.stream().map(s -> URI.create("kestra://" + s)).toArray(URI[]::new)));
        org.junit.jupiter.api.Assertions.assertThrows(FileNotFoundException.class, () -> this.storageInterface.get(tenantId, prefix, new URI("kestra:///" + prefix + "/storage/")));
        path.forEach(Rethrow.throwConsumer(s -> Assertions.assertThat((boolean)this.storageInterface.exists(tenantId, prefix, new URI((String)s))).isFalse()));
    }

    @Test
    void metadata() throws Exception {
        String prefix = IdUtils.create();
        String tenantId = IdUtils.create();
        Map<String, String> expectedMetadata = Map.of("someComplexKey1", "value1", "anotherComplexKey2", "value2");
        this.putFile(tenantId, "/" + prefix + "/storage/get.yml", expectedMetadata);
        StorageObject withMetadata = this.storageInterface.getWithMetadata(tenantId, null, new URI("kestra:///" + prefix + "/storage/get.yml"));
        Assertions.assertThat((String)CharStreams.toString((Readable)new InputStreamReader(withMetadata.inputStream()))).isEqualTo(CONTENT_STRING);
        Assertions.assertThat((Map)withMetadata.metadata()).isEqualTo(expectedMetadata);
    }

    private URI putFile(String tenantId, String path) throws Exception {
        return this.storageInterface.put(tenantId, null, new URI(path), (InputStream)new ByteArrayInputStream(CONTENT_STRING.getBytes()));
    }

    private URI putFile(String tenantId, String path, Map<String, String> metadata) throws Exception {
        return this.storageInterface.put(tenantId, null, new URI(path), new StorageObject(metadata, (InputStream)new ByteArrayInputStream(CONTENT_STRING.getBytes())));
    }

    private URI putInstanceFile(String path) throws Exception {
        return this.storageInterface.putInstanceResource(null, new URI(path), (InputStream)new ByteArrayInputStream(CONTENT_STRING.getBytes()));
    }

    private URI putInstanceFile(String path, Map<String, String> metadata) throws Exception {
        return this.storageInterface.putInstanceResource(null, new URI(path), new StorageObject(metadata, (InputStream)new ByteArrayInputStream(CONTENT_STRING.getBytes())));
    }
}

