package io.kiw.speedy;

import io.kiw.speedy.helper.ImmutableIntMap;
import io.kiw.speedy.management.HostRegistrationMessage;
import io.kiw.speedy.marshaller.MessageUnMarshaller;
import io.kiw.speedy.marshaller.PacketHandler;
import io.kiw.speedy.publisher.SchedulerThread;
import io.kiw.speedy.publisher.SpeedyMessagingPublisher;
import io.kiw.speedy.subscriber.*;
import io.kiw.speedy.wiring.SpeedyWiring;
import io.kiw.speedy.wiring.thread.ThreadHandler;

import java.io.IOException;
import java.util.Map;

public class SpeedyMessagingImpl implements SpeedyMessaging {
    public static final int DATAGRAM_LENGTH = 1480 * 4;
    private SpeedyMessagingSubscriber speedyMessagingSubscriber;
    private final SpeedyMessagingPublisher speedyMessagingPublisher;
    private final SpeedyHost localhost;
    private final ImmutableIntMap<PublisherBucket> publisherBuckets;
    private final ImmutableIntMap<SpeedyConnection> remoteConnections;
    private final Map<Integer, SubscriberChannelState> channelStates;
    private final SchedulerThread schedulerThread;
    private final SpeedyWiring wiring;
    private SpeedyMessagingSubscriberFactory speedyMessagingSubscriberFactory;

    public SpeedyMessagingImpl(final SpeedyHost localhost,
                               OnMessageErrorHandler subcribeErrorHandler,
                               SpeedyWiring wiring,
                               ImmutableIntMap<PublisherBucket> publisherBuckets,
                               ImmutableIntMap<SpeedyConnection> remoteConnections,
                               Map<Integer, SubscriberChannelState> channelStates,
                               PacketHandler packetHandler,
                               SchedulerThread schedulerThread) {
        this.wiring = wiring;
        this.localhost = localhost;
        this.publisherBuckets = publisherBuckets;
        this.remoteConnections = remoteConnections;
        this.channelStates = channelStates;
        this.schedulerThread = schedulerThread;

        speedyMessagingPublisher = new SpeedyMessagingPublisher(packetHandler, wiring, schedulerThread);
        speedyMessagingSubscriberFactory = new SpeedyMessagingSubscriberFactory(localhost, new MessageUnMarshaller(), remoteConnections, speedyMessagingPublisher,
                subcribeErrorHandler, wiring, publisherBuckets, schedulerThread, channelStates);
    }

    public void start() {
        subscribeToAllResponsesHandlers();
        speedyMessagingSubscriber = speedyMessagingSubscriberFactory.build();
        assertAllKeysHaveBeenSubscribedTo();
        wiring.start(speedyMessagingPublisher, speedyMessagingSubscriber, schedulerThread, remoteConnections);
        connectToRemoteHosts();
    }



    @Override
    public void subscribe(String key, SpeedyMessageHandler messageHandler) {
        speedyMessagingSubscriberFactory.addSubscriptionHandler(key, new GenericHandler(messageHandler));
    }

    @Override
    public void subscribe(String key, SpeedyMessageReplyHandler messageHandler) {
        speedyMessagingSubscriberFactory.addSubscriptionHandler(key, new GenericHandler(messageHandler));
    }

    @Override
    public void publish(String key, byte[] bytes) {
        speedyMessagingPublisher.publish(key, bytes);
    }

    @Override
    public void publish(String key, byte[] bytes, int dataLength) {
        speedyMessagingPublisher.publish(key, bytes, dataLength);
    }

    @Override
    public void request(String key, byte[] bytes, SpeedyMessageHandler responseHandler) {
        speedyMessagingPublisher.request(key, bytes, responseHandler);
    }

    @Override
    public void close() throws IOException {
        speedyMessagingPublisher.flush();
        if(speedyMessagingSubscriber != null)
        {
            speedyMessagingSubscriber.close();
        }
        speedyMessagingPublisher.close();
    }

    private void connectToRemoteHosts() {
        ThreadHandler threadHandler = wiring.getThreadHandler();
        for (SpeedyConnection remoteHost : remoteConnections.values().toArray(new SpeedyConnection[remoteConnections.size()])) {
            speedyMessagingPublisher.hostRegistration(new HostRegistrationMessage(localhost, remoteHost.getHost()), remoteHost);
        }
        threadHandler.join();
        wiring.connectIfSingleConnection(remoteConnections);
    }

    private void assertAllKeysHaveBeenSubscribedTo() {
        speedyMessagingSubscriberFactory.assertAllKeysHaveBeenSubscribedTo();
    }

    private void subscribeToAllResponsesHandlers() {
        for (SubscriberChannelState subscriberSequenceState : channelStates.values()) {
            if(subscriberSequenceState.getResponseKey() != null)
            {
                subscribe(subscriberSequenceState.getResponseKey(), subscriberSequenceState::handleResponse);
            }
        }
    }

}
