package io.kiw.speedy.builder;

import io.kiw.speedy.PublisherBucket;
import io.kiw.speedy.SpeedyConnection;
import io.kiw.speedy.SpeedyHost;
import io.kiw.speedy.SpeedyMessagingImpl;
import io.kiw.speedy.channel.TemporaryResponseHandler;
import io.kiw.speedy.channel.TemporaryResponseHandlerImpl;
import io.kiw.speedy.helper.ImmutableIntMap;
import io.kiw.speedy.helper.ImmutableMapFactory;
import io.kiw.speedy.marshaller.EventMarshaller;
import io.kiw.speedy.marshaller.PacketHandlerImpl;
import io.kiw.speedy.publisher.PacketFlusher;
import io.kiw.speedy.publisher.SchedulerThread;
import io.kiw.speedy.subscriber.*;
import io.kiw.speedy.wiring.MultiThreadSubscriberHandler;
import io.kiw.speedy.wiring.SpeedyWiring;
import io.kiw.speedy.wiring.UdpSpeedyWiring;
import io.kiw.tetryon.EventHandlerGroup;
import io.kiw.tetryon.Tetryon;

import java.util.*;

import static io.kiw.speedy.builder.KeysBuilder.RESPONSE_SUFFIX;

public class SpeedyMessagingBuilder {

    private final SpeedyHost localhost;
    private OnMessageErrorHandler onMessageErrorHandler = t -> {};
    private SpeedyWiring wiring;
    private final Map<Integer, SubscribingChannel> subscribingChannels;
    private final Map<Integer, PublishingChannel> publishingChannels;
    private final Set<SpeedyHost> remoteHosts;

    SpeedyMessagingBuilder(SpeedyHost localhost,
                           Map<Integer, PublishingChannel> publishingChannels,
                           Map<Integer, SubscribingChannel> subscribingChannels,
                           Set<SpeedyHost> remoteHosts) {
        this.localhost = localhost;
        this.publishingChannels = publishingChannels;
        this.subscribingChannels = subscribingChannels;
        this.remoteHosts = remoteHosts;
    }

    public SpeedyMessagingBuilder withOnMessageErrorHandler(OnMessageErrorHandler onMessageErrorHandler) {
        this.onMessageErrorHandler = onMessageErrorHandler;
        return this;
    }

    public SpeedyMessagingBuilder withWiring(SpeedyWiring wiring) {
        this.wiring = wiring;
        return this;
    }

    public SpeedyMessagingImpl build() {
        if(localhost == null)
        {
            throw new RuntimeException("Local speedy host must be defined.");
        }

        if(wiring == null)
        {
            wiring = new UdpSpeedyWiring(localhost.getPort());
        }

        SchedulerThread schedulerThread = new SchedulerThread(wiring::getNanoTime);
        wiring.addPulseHandler(schedulerThread::pulse);

        Map<Integer, SpeedyConnection> remoteConnections = new HashMap<>();
        Map<Integer, SubscriberChannelState> globalChannelState = new HashMap<>();
        for (SpeedyHost remoteHost : remoteHosts) {
            Set<Integer> registeredKeyIdentifiers = new HashSet<>();
            Map<Integer, SubscriberChannelState> channelStateForRemoteHost = new HashMap<>();
            for (Map.Entry<Integer, SubscribingChannel> subscribingChannelEntry : subscribingChannels.entrySet()) {

                SubscribingChannel subscribingChannel = subscribingChannelEntry.getValue();
                PublishingChannel publishingChannel = publishingChannels.get(subscribingChannelEntry.getKey());
                if(subscribingChannel.getPublishingHosts().contains(remoteHost) || publishingChannel.getSubscribingHosts().contains(remoteHost))
                {
                    for (String subscribingKey : subscribingChannel.getKeys()) {
                        registeredKeyIdentifiers.add(subscribingKey.hashCode());
                    }
                    TemporaryResponseHandler temporaryResponseHandler = publishingChannel.getChannelMode() == ChannelMode.SUBSCRIBER_ONLY ? new NoOpTemporaryResponseHandler():
                            new TemporaryResponseHandlerImpl(subscribingChannelEntry.getValue().getName().hashCode() + "-" + localhost.hashCode() + RESPONSE_SUFFIX);


                    SubscriberThreadHandler subscriberThreadHandler;
                    if (subscribingChannel.getSubscriberThreads() <= 1)
                    {
                        subscriberThreadHandler = new SubscriberSameThreadHandler(subscribingChannelEntry.getValue().getName());
                    }
                    else
                    {
                        List<Tetryon.EventHandler<HandleMessageEvent>> group = EventHandlerGroup.createGroup(() ->
                                new SubscriberConsumer(subscribingChannelEntry.getValue().getName()), subscribingChannel.getSubscriberThreads());

                        MultiThreadSubscriberHandler multiThreadSubscriberHandler = wiring.buildMultiThreadSubscriberHandler(group);
                        subscriberThreadHandler = new SubscriberMultiThreadHandler(multiThreadSubscriberHandler::handleMessage);
                    }

                    SubscriberChannelState subscriberChannelState = new SubscriberChannelState(0,
                            subscribingChannel.getWindowSizeOfRoute(),
                            subscriberThreadHandler,
                            temporaryResponseHandler
                            );
                    channelStateForRemoteHost.put(subscribingChannelEntry.getKey(), subscriberChannelState);
                    globalChannelState.put(subscribingChannelEntry.getKey(), subscriberChannelState);
                }
            }

            remoteConnections.put(remoteHost.hashCode(), new SpeedyConnection(remoteHost, registeredKeyIdentifiers, ImmutableMapFactory.initialiseIntMap(channelStateForRemoteHost)));
        }

        final Map<Integer, PublisherBucket> publisherBucketsMap = new HashMap<>();

        for (Map.Entry<Integer, PublishingChannel> publishingChannelEntry : publishingChannels.entrySet()) {

            createSequenceBucketAndMappings(publishingChannelEntry, publisherBucketsMap, remoteConnections, globalChannelState);
        }

        ImmutableIntMap<PublisherBucket> publisherBuckets = ImmutableMapFactory.initialiseIntMap(publisherBucketsMap);
        PacketFlusher packetflusher = new PacketFlusher(wiring::sendPacket);

        ImmutableIntMap<SpeedyConnection> tImmutableIntMap = ImmutableMapFactory.initialiseIntMap(remoteConnections);
        return new SpeedyMessagingImpl(
                localhost,
                onMessageErrorHandler,
                wiring,
                publisherBuckets,
                tImmutableIntMap,
                globalChannelState,
                wiring.wrapPacketHandler(
                        new PacketHandlerImpl(
                                packetflusher,
                                new EventMarshaller(packetflusher),
                                publisherBuckets.values().toArray(new PublisherBucket[publisherBuckets.size()]),
                            wiring)
                ),
                schedulerThread);
    }


    private void createSequenceBucketAndMappings(Map.Entry<Integer, PublishingChannel> publishingChannelEntry, Map<Integer, PublisherBucket> publisherBuckets,
                                                 Map<Integer, SpeedyConnection> allRemoteConnections, Map<Integer, SubscriberChannelState> globalSubscriberChannelState) {

        PublishingChannel publishingChannel = publishingChannelEntry.getValue();
        HashSet<String> keysToPublishto = new HashSet<>();
        keysToPublishto.addAll(publishingChannel.getKeys());

        publisherBuckets.put(publishingChannelEntry.getKey(),
                new PublisherBucket(
                        keysToPublishto,
                        publishingChannel.getWindowSizeOfRoute(),
                        localhost.hashCode(),
                        publishingChannelEntry.getKey(),
                        globalSubscriberChannelState.get(publishingChannelEntry.getKey()),
                        getFilteredSpeedyConnections(publishingChannel.getSubscribingHosts(), allRemoteConnections)
                ));
    }

    private SpeedyConnection[] getFilteredSpeedyConnections(Collection<SpeedyHost> publishingHosts, Map<Integer, SpeedyConnection> allRemoteConnections) {
        Set<SpeedyConnection> speedyConnections = new HashSet<>();
        for (SpeedyHost publishingHost : publishingHosts) {
            speedyConnections.add(allRemoteConnections.get(publishingHost.hashCode()));
        }

        return speedyConnections.toArray(new SpeedyConnection[speedyConnections.size()]);
    }
}
