package io.kiw.speedy.publisher;

import io.kiw.speedy.PublisherBucket;
import io.kiw.speedy.SpeedyConnection;
import io.kiw.speedy.exception.SpeedyMessagingNotInitiatedException;
import io.kiw.speedy.management.HostRegistrationMessage;
import io.kiw.speedy.management.ManagementKey;
import io.kiw.speedy.marshaller.PacketHandler;
import io.kiw.speedy.subscriber.SpeedyMessageHandler;
import io.kiw.speedy.wiring.SpeedyWiring;

import java.io.IOException;
import java.util.concurrent.TimeUnit;


public class SpeedyMessagingPublisher implements PublishPromise
{
    private final PacketHandler packetHandler;
    private final SpeedyWiring wiring;
    private final SchedulerThread schedulerThread;
    private volatile boolean initialised;

    public SpeedyMessagingPublisher(PacketHandler packetHandler,
                                    SpeedyWiring wiring, SchedulerThread schedulerThread) {
        this.packetHandler = packetHandler;
        this.wiring = wiring;
        this.schedulerThread = schedulerThread;
    }

    public void hostRegistration(HostRegistrationMessage message, SpeedyConnection speedyConnection) {
        final String key = ManagementKey.HOST_REGISTRATION.getKey();
        byte[] initiateHostRegistration = message.getMarshalled();

        wiring.connectToRemoteHost(speedyConnection, () -> publish(key, initiateHostRegistration));
    }

    public void publish(String key, byte[] data) {
        publish(key, data, data.length);
    }

    public void publish(String key, byte[] data, int dataLength) {
        if(!initialised)
        {
            throw new SpeedyMessagingNotInitiatedException("Publisher Non initiated");
        }
        int eventId = wiring.startEvent();
        boolean isManagementKey = ManagementKey.isManagementKey(key);
        packetHandler.handleEvent(key, data, dataLength, isManagementKey);

        wiring.completeEvent(eventId);
    }


    public void request(String key, byte[] data, SpeedyMessageHandler responseHandler) {
        int eventId = wiring.startEvent();
        boolean isManagementKey = ManagementKey.isManagementKey(key);
        packetHandler.handleEventAndResponseHandler(key, data, data.length, responseHandler, isManagementKey);

        wiring.completeEvent(eventId);
    }


    public void start()
    {
        enable();
    }

    public void enable() {
        initialised = true;
        schedulerThread.addJob(new SchedulerThread.ScheduledJob(TimeUnit.MILLISECONDS.toNanos(1), wiring.getNanoTime(), this::flushBucketsIfRequired));

    }

    private void flushBucketsIfRequired(long nanoTime)
    {
        packetHandler.flushBucketsIfRequired();
    }

    public void resend(PublisherBucket sequenceBucket, long firstPacketKeyMissing, long lastPacketKeyMissing) {
        packetHandler.resendPackets(sequenceBucket, firstPacketKeyMissing, lastPacketKeyMissing);
    }

    public synchronized void close() throws IOException {
        schedulerThread.stop();

        wiring.closePublisher();
    }

    public void flush() {
        packetHandler.flushBucketsIfRequired();
    }
}
