package io.kiw.speedy.subscriber;

import io.kiw.speedy.SpeedyConnection;
import io.kiw.speedy.SpeedyMessagingImpl;
import io.kiw.speedy.channel.NackSchedulerJob;
import io.kiw.speedy.helper.ImmutableIntMap;
import io.kiw.speedy.marshaller.MessageUnMarshaller;
import io.kiw.speedy.publisher.PublishPromise;
import io.kiw.speedy.publisher.SchedulerThread;
import io.kiw.speedy.wiring.SpeedyWiring;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.concurrent.TimeUnit;

public class SpeedyMessagingSubscriber implements Runnable{
    private final MessageUnMarshaller messageUnmarshaller;
    private final ImmutableIntMap<SpeedyConnection> remoteConnections;
    private final PublishPromise publishPromise;
    private final ByteBuffer buffer;
    private final OnMessageErrorHandler onMessageErrorHandler;
    private final SpeedyWiring wiring;
    private final SchedulerThread schedulerThread;
    private final ImmutableIntMap<GenericHandler> subscriptions;
    private final NackSchedulerJob nackSchedulerJob;

    public SpeedyMessagingSubscriber(MessageUnMarshaller messageUnMarshaller,
                                     ImmutableIntMap<SpeedyConnection> remoteConnections,
                                     PublishPromise publishPromise,
                                     OnMessageErrorHandler onMessageErrorHandler,
                                     SpeedyWiring wiring,
                                     NackSchedulerJob nackSchedulerJob,
                                     SchedulerThread schedulerThread,
                                     ImmutableIntMap<GenericHandler> subscriptions) {
        this.messageUnmarshaller = messageUnMarshaller;
        this.remoteConnections = remoteConnections;
        this.publishPromise = publishPromise;
        this.onMessageErrorHandler = onMessageErrorHandler;
        this.wiring = wiring;
        this.schedulerThread = schedulerThread;
        this.subscriptions = subscriptions;
        this.buffer = ByteBuffer.allocateDirect(SpeedyMessagingImpl.DATAGRAM_LENGTH);
        this.wiring.registerFragmentHandler(this::handleFragment);
        this.nackSchedulerJob = nackSchedulerJob;
    }

    @Override
    public void run()
    {
        while (!Thread.interrupted()) {
            handleFragment();
        }
    }

    private void handleFragment() {
        buffer.clear();
        wiring.receive(buffer);
        buffer.flip();
        SubscriberChannelState channelState = handlePacket(buffer);
        if(channelState != null)
        {
            handleRecoveryMessages(channelState);
        }
    }

    private void handleRecoveryMessages(SubscriberChannelState channelChannelState) {
        ByteBuffer bufferFromRecover;
        while((bufferFromRecover = channelChannelState.getNextRecoverMessage()).hasRemaining())
        {
            long packetSequenceNumber = bufferFromRecover.getLong();// packet channel number

            int publisherIdentifier = bufferFromRecover.getInt();
            bufferFromRecover.getInt(); // channel identifier we already know

            // execute messages from recovery packet
            while (bufferFromRecover.remaining() > 0)
            {
                messageUnmarshaller.unmarshallAndPotentiallyHandle(bufferFromRecover, subscriptions, onMessageErrorHandler, channelChannelState, publisherIdentifier, publishPromise);
            }

            // increment
            channelChannelState.increment(packetSequenceNumber);
        }
    }


    private SubscriberChannelState handlePacket(ByteBuffer buffer) {
        if(buffer.remaining() == 0)
        {
            return null;
        }
        long packetSequenceNumber = buffer.getLong();// packet sequence number

        int publisherIdentifier = buffer.getInt();
        SpeedyConnection speedyConnection = remoteConnections.get(publisherIdentifier);
        int channelIdentifier = buffer.getInt();
        SubscriberChannelState channelState = speedyConnection.getChannelSequenceState(channelIdentifier);
        long expectedPacketSequenceNumber = channelState.getSequenceNumber();
        if(packetSequenceNumber == expectedPacketSequenceNumber)
        {

            // execute messages from packet
            while (buffer.remaining() > 0)
            {
                messageUnmarshaller.unmarshallAndPotentiallyHandle(buffer, subscriptions, onMessageErrorHandler, channelState, publisherIdentifier, publishPromise);
            }

            // increment
            channelState.increment(packetSequenceNumber);
        }
        else if(packetSequenceNumber > expectedPacketSequenceNumber)
        {
            buffer.rewind();

            if(expectedPacketSequenceNumber == 0)
            {
                channelState.setSequenceNumber(packetSequenceNumber + 1);
                return handlePacket(buffer);
            }
            // NAK all that stuff
            channelState.copyToRecovery(buffer, packetSequenceNumber);
            nackSchedulerJob.onNack(channelIdentifier, publisherIdentifier, expectedPacketSequenceNumber, packetSequenceNumber  - 1);
            return null;
        }
        else if(packetSequenceNumber < expectedPacketSequenceNumber)
        {
            return null;
        }

        buffer.clear();
        return channelState;
    }

    public void close() throws IOException {
        wiring.closeSubscriber();
    }

    public void addNackScheduledJob() {
        this.schedulerThread.addJob(new SchedulerThread.ScheduledJob(TimeUnit.MICROSECONDS.toNanos(100), 0, nackSchedulerJob::onPulse));
    }
}
