package io.kiw.speedy.subscriber;

import io.kiw.speedy.channel.RecoveryBuffer;
import io.kiw.speedy.channel.TemporaryResponseHandler;
import io.kiw.speedy.publisher.PublishPromise;

import java.nio.ByteBuffer;

import static io.kiw.speedy.marshaller.PacketHandlerImpl.MAX_EVENT_SIZE;

public class SubscriberChannelState {
    private final SubscriberThreadHandler subscriberThreadHandler;
    private long sequenceNumber;
    private final RecoveryBuffer recoveryBuffer;
    private final ByteBuffer eventBuffer = ByteBuffer.allocateDirect(MAX_EVENT_SIZE);
    private final TemporaryResponseHandler temporaryResponseHandler;
    private long eventSequenceNumber;

    public SubscriberChannelState(long startingSequenceNumber, final int windowSize, final SubscriberThreadHandler subscriberThreadHandler, TemporaryResponseHandler temporaryResponseHandler) {
        this.sequenceNumber = startingSequenceNumber;
        this.recoveryBuffer = new RecoveryBuffer(windowSize);
        this.temporaryResponseHandler = temporaryResponseHandler;
        this.subscriberThreadHandler = subscriberThreadHandler;
    }

    public long getSequenceNumber() {
        return sequenceNumber;
    }

    public ByteBuffer getNextRecoverMessage() {
        return recoveryBuffer.getPacket(sequenceNumber);
    }

    public void increment(long packetSequenceNumber) {
        recoveryBuffer.markSeen(packetSequenceNumber);
        sequenceNumber++;
    }

    public void copyToRecovery(ByteBuffer buffer, long packetSequenceNumber) {
        recoveryBuffer.addMessageToPacketRecoveryBuffer(packetSequenceNumber, buffer);
    }

    public ByteBuffer getEventBuffer() {
        return eventBuffer;
    }

    public void setSequenceNumber(long sequenceNumber) {
        this.sequenceNumber = sequenceNumber;
    }

    public void handleResponse(ByteBuffer message) {
        temporaryResponseHandler.handle(message);
    }

    public String getResponseKey() {
        return temporaryResponseHandler.getResponseKey();
    }

    public void addResponseHandler(SpeedyMessageHandler responseHandler, int responseCountToExpect, long eventSequenceNumber) {
        temporaryResponseHandler.add(eventSequenceNumber, responseHandler, responseCountToExpect);
    }


    public void setEventSequenceNumber(long eventSequenceNumber) {
        this.eventSequenceNumber = eventSequenceNumber;
    }

    public void handleMessage(GenericHandler speedyMessageHandler, ByteBuffer eventBuffer, int publisherIdentifier, PublishPromise publishPromise) {
        subscriberThreadHandler.handleMessage(speedyMessageHandler, eventBuffer, publisherIdentifier, eventSequenceNumber, publishPromise);
    }
}
