package io.konig.shacl.io;

/*
 * #%L
 * Konig Core
 * %%
 * Copyright (C) 2015 - 2016 Gregory McFall
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */


import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.SKOS;
import org.openrdf.rio.RDFHandlerException;

import io.konig.activity.Activity;
import io.konig.core.Graph;
import io.konig.core.NamespaceManager;
import io.konig.core.RdbmsShapeValidator;
import io.konig.core.Vertex;
import io.konig.core.extract.ExtractException;
import io.konig.core.extract.OntologyExtractor;
import io.konig.core.impl.CompositeLocalNameService;
import io.konig.core.impl.MemoryGraph;
import io.konig.core.impl.MemoryNamespaceManager;
import io.konig.core.impl.RdfUtil;
import io.konig.core.impl.SimpleLocalNameService;
import io.konig.core.io.FileGetter;
import io.konig.core.pojo.EmitContext;
import io.konig.core.pojo.SimplePojoEmitter;
import io.konig.core.util.IOUtil;
import io.konig.core.vocab.Konig;
import io.konig.core.vocab.SH;
import io.konig.core.vocab.Schema;
import io.konig.shacl.Shape;

public class ShapeWriter {
	
	public static void write(OutputStream out, Shape...shapeList) throws RDFHandlerException, IOException {
		ShapeWriter writer = new ShapeWriter();
		
		for (Shape s : shapeList) {
			writer.writeTurtle(s, out);
		}
		
	}

	public void emitShape(Shape shape, Graph graph) {
		SimplePojoEmitter emitter = SimplePojoEmitter.getInstance();
		CompositeLocalNameService nameService = new CompositeLocalNameService(
			SimpleLocalNameService.getDefaultInstance(), graph);
		
		EmitContext context = new EmitContext(graph);
		context.addIriReference(SH.shape, SH.path, SH.targetClass, SH.valueClass, Konig.aggregationOf, Konig.rollUpBy,
				Konig.defaultShapeFor, Konig.inputShapeOf, Konig.tabularOriginShape, Konig.preferredTabularShape,
				Konig.explicitDerivedFrom, SKOS.BROADER, Konig.nodeShapeCube, Schema.isPartOf,
				SH.xone, SH.and, SH.or, SH.not);
		context.setLocalNameService(nameService);
		context.addIgnoredProperty(Konig.equivalentPath);
		emitter.emit(context, shape, graph);
	}
	

	public void writeTurtle(NamespaceManager nsManager, Shape shape, OutputStream out) throws RDFHandlerException, IOException {
		try (OutputStreamWriter writer = new OutputStreamWriter(out)) {
			writeTurtle(nsManager, shape, out);
		}
	}
	
	public void writeTurtle(Shape shape, OutputStream out) throws RDFHandlerException, IOException {
		try (OutputStreamWriter writer = new OutputStreamWriter(out)) {
			writeTurtle(MemoryNamespaceManager.getDefaultInstance(), shape, writer);
		}
	}
	
	
	public void writeTurtle(NamespaceManager nsManager, Shape shape, Writer writer) throws RDFHandlerException, IOException {
		MemoryGraph graph = new MemoryGraph();
		graph.setNamespaceManager(nsManager);
		
		emitShape(shape, graph);
		
		RdfUtil.prettyPrintTurtle(nsManager, graph, writer);
		
	}
	
	public void writeTurtle(NamespaceManager nsManager, Shape shape, File file) throws RDFHandlerException, IOException {

		MemoryGraph graph = new MemoryGraph();
		graph.setNamespaceManager(nsManager);
		RdbmsShapeValidator validator = new RdbmsShapeValidator();
		if (shape != null && validator.validate(shape)){
			shape.setTabularOriginShape(null);
			graph.edge(shape.getId(), RDF.TYPE, Konig.TabularNodeShape);
		}
		emitShape(shape, graph);
		
		RdfUtil.prettyPrintTurtle(nsManager, graph, file);
	}
	
	/**
	 * Write shapes that were generated by specific processes. 
	 * @param shapeList  The collection of shapes from which generated shapes will be selected
	 * @param fileGetter The FileGetter that provides the File to which any given Shape should be written.
	 * @param activityWhiteList The specific processes for which shapes will be written.
	 * @throws IOException 
	 * @throws RDFHandlerException 
	 */
	public void writeGeneratedShapes(
		NamespaceManager nsManager, 
		Collection<Shape> shapeList, 
		FileGetter fileGetter,
		Set<URI> activityWhitelist) throws RDFHandlerException, IOException {
		for (Shape shape : shapeList) {
			Resource resource = shape.getId();
			if (resource instanceof URI) {

				Activity prov = shape.getWasGeneratedBy();
				if (prov!=null && activityWhitelist.contains(prov.getType())) {
					URI uri = (URI) resource;
					File file = fileGetter.getFile(uri);
					if (file != null) {
						writeTurtle(nsManager, shape, file);
					}
				}
				
				
			}
		}
	}
	
	public void writeShapes(Graph graph, FileGetter fileGetter) throws IOException, ShapeWriteException {
		OntologyExtractor extractor = new OntologyExtractor();
		List<Vertex> list = graph.v(SH.Shape).in(RDF.TYPE).toVertexList();
		for (Vertex shape : list) {
			writeShape(shape, fileGetter, extractor);
		}
	}

	private void writeShape(Vertex shape, FileGetter fileGetter, OntologyExtractor extractor) throws IOException, ShapeWriteException {
		Resource shapeId = shape.getId();
		if (shapeId instanceof URI) {
			URI shapeURI = (URI) shapeId;
			File file = fileGetter.getFile(shapeURI);
			FileWriter out = new FileWriter(file);
			Graph target = new MemoryGraph();
			try {
				extractor.extract(shape, target);
				target.setNamespaceManager(shape.getGraph().getNamespaceManager());
			
				RdfUtil.prettyPrintTurtle(target, out);
			} catch (ExtractException | RDFHandlerException e) {
				throw new ShapeWriteException(e);
			} finally {
				IOUtil.close(out, file.getName());
			}
		}
		
	}

}
