package io.lacuna.bifurcan.durable.blocks;

import io.lacuna.bifurcan.*;
import io.lacuna.bifurcan.durable.BlockPrefix;
import io.lacuna.bifurcan.durable.BlockPrefix.BlockType;
import io.lacuna.bifurcan.durable.Util;
import io.lacuna.bifurcan.durable.io.DurableBuffer;

import java.util.Iterator;

import static io.lacuna.bifurcan.durable.Encodings.encodeBlock;

/**
 * An indexed list, encoded as:
 * - the number of elements [VLQ]
 * - number of SkipTable tiers [uint8]
 * - a SkipTable over the blocks of elements (unless number of tiers is 0)
 * - zero or more ENCODED blocks generated by {@link IDurableEncoding.List#elementEncoding()}
 */
public class List {

  public static <V> void encode(Iterator<V> it, IDurableEncoding.List listEncoding, DurableOutput out) {
    SkipTable.Writer skipTable = new SkipTable.Writer();
    DurableBuffer elements = new DurableBuffer();

    IDurableEncoding elementEncoding = listEncoding.elementEncoding();
    Iterator<IList<V>> blocks = Util.partitionBy(
        it,
        DurableEncodings.blockSize(elementEncoding),
        elementEncoding::isSingleton);

    long index = 0;
    while (blocks.hasNext()) {
      IList<V> b = blocks.next();
      skipTable.append(index, elements.written());
      encodeBlock((IList<Object>) b, elementEncoding, elements);
      index += b.size();
    }

    long size = index;
    DurableBuffer.flushTo(out, BlockType.LIST, acc -> {
      acc.writeUVLQ(size);
      acc.writeUnsignedByte(skipTable.tiers());

      if (skipTable.tiers() > 0) {
        skipTable.flushTo(acc);
      } else {
        skipTable.free();
      }

      elements.flushTo(acc);
    });
  }

  public static DurableList decode(IDurableEncoding.List encoding, IDurableCollection.Root root, DurableInput.Pool pool) {
    DurableInput in = pool.instance();

    BlockPrefix prefix = in.readPrefix();
    assert (prefix.type == BlockType.LIST);
    long pos = in.position();

    long size = in.readUVLQ();
    int skipTableTiers = in.readUnsignedByte();

    SkipTable skipTable = null;
    if (skipTableTiers > 0) {
      DurableInput skipIn = in.sliceBlock(BlockType.TABLE);
      skipTable = new SkipTable(root == null ? skipIn.pool() : () -> root.cached(skipIn), skipTableTiers);
    }

    DurableInput.Pool elements = in.sliceBytes((pos + prefix.length) - in.position()).pool();

    return new DurableList(pool, root, size, skipTable, elements, encoding);
  }
}
