package io.lemoncloud.core.architecture.data

import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.flatMapConcat
import kotlinx.coroutines.flow.flatMapLatest
import kotlinx.coroutines.flow.flatMapMerge
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.flowOf
import kotlinx.coroutines.flow.flowOn
import kotlinx.coroutines.flow.map
import kotlin.coroutines.CoroutineContext

class ResultFlow<T>(
    private val flow: Flow<Result<T>>,
) {
    val value: Flow<Result<T>>
        get() = flow

    suspend fun collect(
        onSuccess: suspend (T) -> Unit,
        onFailure: suspend (Throwable) -> Unit,
    ) {
        value.collect {
            it.fold(
                onSuccess = { value -> onSuccess(value) },
                onFailure = { exception -> onFailure(exception) },
            )
        }
    }

    suspend fun first(
        onSuccess: suspend (T) -> Unit,
        onFailure: suspend (Throwable) -> Unit,
    ) {
        value.first().fold(
            onSuccess = { value -> onSuccess(value) },
            onFailure = { exception -> onFailure(exception) },
        )
    }

    fun <R> transform(transform: suspend (T) -> R): ResultFlow<R> =
        ResultFlow(
            value.map { result ->
                result.fold(
                    onSuccess = { value -> runCatching { transform(value) } },
                    onFailure = { exception -> Result.failure(exception) },
                )
            },
        )

    fun <R> transform(
        onSuccess: suspend (T) -> R,
        onFailure: (Throwable) -> Throwable,
    ): ResultFlow<R> =
        ResultFlow(
            value.map { result ->
                result.fold(
                    onSuccess = { value -> runCatching { onSuccess(value) } },
                    onFailure = { exception -> Result.failure(onFailure(exception)) },
                )
            },
        )

    fun <R> transformWithFailure(
        onSuccess: suspend (T) -> R,
        onFailure: (Throwable) -> R,
    ): ResultFlow<R> =
        ResultFlow(
            value.map { result ->
                result.fold(
                    onSuccess = { value -> runCatching { onSuccess(value) } },
                    onFailure = { exception -> runCatching { onFailure(exception) } },
                )
            },
        )

    fun action(onAction: suspend (T) -> Unit): ResultFlow<T> =
        ResultFlow(
            value.map { result ->
                result.fold(
                    onSuccess = { value ->
                        runCatching { onAction(value) }.fold(
                            onSuccess = { Result.success(value) },
                            onFailure = { Result.failure(it) },
                        )
                    },
                    onFailure = { exception -> Result.failure(exception) },
                )
            },
        )

    fun <T2, R> combine(
        flow: ResultFlow<T2>,
        transform: (T, T2) -> R,
    ) = ResultFlow(
        value.combine(flow.value) { result1, result2 ->
            result1.fold(
                onSuccess = { value1 ->
                    result2.fold(
                        onSuccess = { value2 -> runCatching { transform(value1, value2) } },
                        onFailure = { exception -> Result.failure(exception) },
                    )
                },
                onFailure = { exception -> Result.failure(exception) },
            )
        },
    )

    @OptIn(ExperimentalCoroutinesApi::class)
    fun <T2> flatMapLatest(flow: suspend (T) -> ResultFlow<T2>): ResultFlow<T2> =
        ResultFlow(
            value.flatMapLatest { result ->
                result.fold(
                    onSuccess = { value -> flow(value).value },
                    onFailure = { error -> flowOf(Result.failure(error)) },
                )
            },
        )

    @OptIn(ExperimentalCoroutinesApi::class)
    fun <T2> flatMapConcat(flow: suspend (T) -> ResultFlow<T2>): ResultFlow<T2> =
        ResultFlow(
            value.flatMapConcat { result ->
                result.fold(
                    onSuccess = { value -> flow(value).value },
                    onFailure = { error -> flowOf(Result.failure(error)) },
                )
            },
        )

    @OptIn(ExperimentalCoroutinesApi::class)
    fun <T2> flatMapMerge(flow: suspend (T) -> ResultFlow<T2>): ResultFlow<T2> =
        ResultFlow(
            value.flatMapMerge { result ->
                result.fold(
                    onSuccess = { value -> flow(value).value },
                    onFailure = { error -> flowOf(Result.failure(error)) },
                )
            },
        )

    fun flowOn(context: CoroutineContext): ResultFlow<T> = ResultFlow(value.flowOn(context))


    companion object {
        fun <T> Flow<Result<T>>.asResultFlow(): ResultFlow<T> = ResultFlow(this)

        suspend fun <T> Flow<T>.asResultFlow(): ResultFlow<T> =
            ResultFlow(
                flow {
                    this@asResultFlow
                        .catch { emit(Result.failure(it)) }
                        .collect { emit(Result.success(it)) }
                },
            )

        fun <T> valueOf(invoke: suspend () -> T) =
            ResultFlow(
                flow {
                    emit(runCatching { invoke() })
                },
            )
    }
}
