package io.leopard.topnb.service;

import io.leopard.topnb.model.Performance;
import io.leopard.topnb.model.PerformanceVO;
import io.leopard.topnb.util.DefaultPerformanceComparator;
import io.leopard.topnb.util.PerformanceUtil;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

public class PerformanceHandlerImpl implements PerformanceHandler {

	protected Log logger = LogFactory.getLog(this.getClass());

	@Autowired
	private PerformanceService performanceService;

	// @Autowired
	// private PerformanceBeanService performanceBeanService;

	@Override
	public List<PerformanceVO> list(String entryName, String postdate) {
		List<Performance> performanceList = performanceService.listAll(entryName);
		if (performanceList == null || performanceList.size() == 0) {
			return new ArrayList<PerformanceVO>();
		}

		List<PerformanceVO> performanceVOList = new ArrayList<PerformanceVO>();
		for (Performance performance : performanceList) {
			performanceVOList.add(this.toPerformanceVO(performance));
		}
		this.ratio(performanceVOList, entryName);

		Collections.sort(performanceVOList, new DefaultPerformanceComparator());// 默认排序,先按分类排序，再按总次数排序.
		// Collections.sort(performanceVOList, new AvgTimeComparator());//
		// 默认排序,先按分类排序，再按总次数排序.

		return performanceVOList;
	}

	/**
	 * 获取各模块的总耗时.
	 * 
	 * @param performanceVOList
	 * @return
	 */
	protected Map<String, Long> getModuleTotalTime(List<PerformanceVO> performanceVOList) {
		Map<String, Long> map = new HashMap<String, Long>();
		for (PerformanceVO performanceVO : performanceVOList) {
			String moduleName = performanceVO.getModuleName();
			Long totalTime = map.get(moduleName);
			if (totalTime == null) {
				totalTime = 0L;
			}
			totalTime += performanceVO.getTotalTime();
			map.put(moduleName, totalTime);
		}
		return map;
	}

	/**
	 * 获取各接口的总耗时.
	 * 
	 * @param performanceVOList
	 * @return
	 */
	protected Map<String, Long> getInterfaceTotalTime(List<PerformanceVO> performanceVOList) {
		Map<String, Long> map = new HashMap<String, Long>();
		for (PerformanceVO performanceVO : performanceVOList) {
			String simpleClassName = performanceVO.getSimpleClassName();

			if (simpleClassName.endsWith("CacheImpl")) {
				continue;
			}
			String interfaceName = performanceVO.getInterfaceName();
			Long totalTime = map.get(interfaceName);
			if (totalTime == null) {
				totalTime = 0L;
			}
			totalTime += performanceVO.getTotalTime();
			map.put(interfaceName, totalTime);
		}
		return map;
	}

	/**
	 * 获取各模块的总耗时.
	 * 
	 * @param performanceVOList
	 * @return
	 */
	protected long getAllTotalTime(List<PerformanceVO> performanceVOList, String threadName) {
		long totalTime = 0;
		if (StringUtils.isNotEmpty(threadName)) {
			totalTime = this.getAllTotalTimeByEntry(performanceVOList);
			if (totalTime > -1) {
				return totalTime;
			}
			else {
				totalTime = 0;
			}
		}

		for (PerformanceVO performanceVO : performanceVOList) {
			totalTime += performanceVO.getTotalTime();
		}
		return totalTime;
	}

	protected long getAllTotalTimeByEntry(List<PerformanceVO> performanceVOList) {
		for (PerformanceVO performanceVO : performanceVOList) {
			if (this.isController(performanceVO.getMethodName())) {
				return performanceVO.getTotalTime();
			}
		}
		return -1;

	}

	/**
	 * 计算耗时比率.
	 * 
	 * @param performanceVOList
	 */
	protected void ratio(List<PerformanceVO> performanceVOList, String entryName) {
		long allTotalTime = this.getAllTotalTime(performanceVOList, entryName);
		Map<String, Long> moduleTotalTimeMap = this.getModuleTotalTime(performanceVOList);
		Map<String, Long> interfaceTotalTimeMap = this.getInterfaceTotalTime(performanceVOList);
		// Json.printMap(interfaceTotalTimeMap, "interfaceTotalTimeMap");
		for (PerformanceVO performanceVO : performanceVOList) {
			String moduleName = performanceVO.getModuleName();
			String interfaceName = performanceVO.getInterfaceName();
			Long interfaceTotalTime = interfaceTotalTimeMap.get(interfaceName);
			// System.err.println("moduleName:" + moduleName + " current:" +
			// performanceVO.getTotalTime() + " total:" +
			// moduleTotalTimeMap.get(moduleName));
			// System.err.println("interfaceName:" + interfaceName);
			double interfaceRatio = this.percent(performanceVO.getTotalTime(), interfaceTotalTime);
			double moduleRatio = percent(performanceVO.getTotalTime(), moduleTotalTimeMap.get(moduleName));
			double timeRatio = this.percent(performanceVO.getTotalTime(), allTotalTime);

			performanceVO.setInterfaceRatio(interfaceRatio);
			performanceVO.setModuleRatio(moduleRatio);
			performanceVO.setTimeRatio(timeRatio);
		}
	}

	protected double percent(double current, Long total) {
		if (current == 0) {
			return 0;
		}
		if (total == null || total == 0) {
			return 0;
		}
		double avg = current / total;
		double percent = (avg * 100);
		String percent2 = new DecimalFormat(".0").format(percent);
		return Double.parseDouble(percent2);
	}

	/**
	 * 每秒平均值.
	 * 
	 * @param count
	 *            总数
	 * @param time
	 *            秒数
	 * @return 平均值
	 */
	public static long perSecondAvg(long count, long time) {
		if (time <= 0) {
			return 0;
		}
		return count * 1000 / time;
	}

	protected PerformanceVO toPerformanceVO(Performance performance) {
		PerformanceVO performanceVO = new PerformanceVO();

		int threadSize = 1;
		String longMethodName = performance.getMethodName();

		long totalCount = performance.getCount();
		Long totalMilliSeconds = performance.getTotalTime() / 1000L / threadSize;
		long totalSeconds = totalMilliSeconds / 1000L;
		long avgCount = perSecondAvg(totalCount, totalMilliSeconds);

		double avgTime = PerformanceUtil.avgTime(totalCount, totalMilliSeconds);

		String categoryName = PerformanceUtil.getModuleName(longMethodName);
		String moduleName = PerformanceUtil.getModuleName(longMethodName);
		String interfaceName = PerformanceUtil.getInterfaceName(longMethodName);
		String simpleMethodName = PerformanceUtil.parseSimpleMethodName(longMethodName);
		String simpleClassName = PerformanceUtil.parseSimpleClassName(longMethodName);

		performanceVO.setCount(totalCount);
		performanceVO.setTotalTime(performance.getTotalTime());
		performanceVO.setMethodName(longMethodName);
		performanceVO.setSimpleMethodName(simpleMethodName);
		performanceVO.setSimpleClassName(simpleClassName);

		performanceVO.setTotalSeconds(totalSeconds);
		performanceVO.setTotalMilliSeconds(totalMilliSeconds);
		performanceVO.setAvgCount(avgCount);
		performanceVO.setAvgTime(avgTime);
		performanceVO.setModuleName(moduleName);
		performanceVO.setCategoryName(categoryName);
		performanceVO.setInterfaceName(interfaceName);

		// String countStr = StringUtils.rightPad(Long.toString(totalCount),
		// 11);
		// String totalTimeStr =
		// StringUtils.rightPad(Long.toString(totalMilliSeconds) + "ms", 12);
		// String totalSecondStr = StringUtils.rightPad(Long.toString(second) +
		// "s", 7);
		// String avgCountStr = StringUtils.rightPad(Long.toString(avgCount),
		// 7);
		// String avgTimeStr =
		// StringUtils.rightPad(PerformanceUtil.avgTime(avgCount) + "ms", 7);

		// String message = ("totalCount:" + countStr + " totalTime:" +
		// totalTimeStr + " totalSecond:" + totalSecondStr + " avgCount:" +
		// avgCountStr + " avgTime:" + avgTimeStr + " methodName:" +
		// methodName);

		return performanceVO;
	}

	private boolean isController(String longMethodName) {
		String className = PerformanceUtil.parseClassName(longMethodName);
		if (className.endsWith("Controller")) {
			if (!className.endsWith("ErrorController")) {
				return true;
			}
		}
		return false;
	}

}
