/*
 * Decompiled with CFR 0.152.
 */
package io.logz.sender;

import io.logz.sender.DiskQueue;
import io.logz.sender.FormattedLogMessage;
import io.logz.sender.HttpsRequestConfiguration;
import io.logz.sender.HttpsSyncSender;
import io.logz.sender.InMemoryQueue;
import io.logz.sender.LogsQueue;
import io.logz.sender.SenderStatusReporter;
import io.logz.sender.com.google.common.hash.Hashing;
import io.logz.sender.com.google.gson.Gson;
import io.logz.sender.com.google.gson.JsonElement;
import io.logz.sender.com.google.gson.JsonObject;
import io.logz.sender.com.google.gson.JsonSyntaxException;
import io.logz.sender.exceptions.LogzioParameterErrorException;
import io.logz.sender.exceptions.LogzioServerErrorException;
import io.opentelemetry.api.common.Attributes;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.SpanContext;
import io.opentelemetry.sdk.resources.Resource;
import io.opentelemetry.semconv.resource.attributes.ResourceAttributes;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;

public class LogzioSender {
    private static final int MAX_SIZE_IN_BYTES = 0x300000;
    private static final int MAX_LOG_SIZE_IN_BYTES = 500000;
    private static final int MAX_LOG_LINE_SIZE_IN_BYTES = 32700;
    private static final String CUT_EXCEEDING_LOG = "cut";
    private static final String DROP_EXCEEDING_LOG = "drop";
    private static final String TRUNCATED_MESSAGE_SUFFIX = "...truncated";
    private static final Map<AbstractMap.SimpleImmutableEntry<String, String>, LogzioSender> logzioSenderInstances = new HashMap<AbstractMap.SimpleImmutableEntry<String, String>, LogzioSender>();
    private static final int FINAL_DRAIN_TIMEOUT_SEC = 20;
    private final LogsQueue logsQueue;
    private final int drainTimeout;
    private final String exceedMaxSizeAction;
    private final boolean debug;
    private final SenderStatusReporter reporter;
    private ScheduledExecutorService tasksExecutor;
    private final AtomicBoolean drainRunning = new AtomicBoolean(false);
    private final HttpsSyncSender httpsSyncSender;
    private final boolean withOpentelemetryContext;

    private LogzioSender(HttpsRequestConfiguration httpsRequestConfiguration, int drainTimeout, boolean debug, SenderStatusReporter reporter, ScheduledExecutorService tasksExecutor, LogsQueue logsQueue, String exceedMaxSizeAction, boolean withOpentelemetryContext) throws LogzioParameterErrorException {
        if (logsQueue == null || reporter == null || httpsRequestConfiguration == null) {
            throw new LogzioParameterErrorException("logsQueue=" + String.valueOf(logsQueue) + " reporter=" + String.valueOf(reporter) + " httpsRequestConfiguration=" + String.valueOf(httpsRequestConfiguration), "For some reason could not initialize URL. Cant recover..");
        }
        this.exceedMaxSizeAction = this.validateAndGetExceedMaxSizeAction(exceedMaxSizeAction);
        this.logsQueue = logsQueue;
        this.drainTimeout = drainTimeout;
        this.debug = debug;
        this.reporter = reporter;
        this.httpsSyncSender = new HttpsSyncSender(httpsRequestConfiguration, reporter);
        this.tasksExecutor = tasksExecutor;
        this.withOpentelemetryContext = withOpentelemetryContext;
        this.debug("Created new LogzioSender class");
    }

    private String validateAndGetExceedMaxSizeAction(String exceedMaxSizeAction) throws LogzioParameterErrorException {
        if (exceedMaxSizeAction != null && Arrays.asList(CUT_EXCEEDING_LOG, DROP_EXCEEDING_LOG).contains(exceedMaxSizeAction.toLowerCase())) {
            return exceedMaxSizeAction.toLowerCase();
        }
        throw new LogzioParameterErrorException("exceedMaxSizeAction=" + exceedMaxSizeAction, "invalid parameter value");
    }

    private static LogzioSender getLogzioSender(HttpsRequestConfiguration httpsRequestConfiguration, int drainTimeout, boolean debug, SenderStatusReporter reporter, ScheduledExecutorService tasksExecutor, LogsQueue logsQueue, String exceedMaxSizeAction, boolean withOpentelemetryContext) throws LogzioParameterErrorException {
        String tokenHash = Hashing.sha256().hashString(httpsRequestConfiguration.getLogzioToken(), StandardCharsets.UTF_8).toString().substring(0, 7);
        AbstractMap.SimpleImmutableEntry<String, String> tokenAndTypePair = new AbstractMap.SimpleImmutableEntry<String, String>(tokenHash, httpsRequestConfiguration.getLogzioType());
        LogzioSender logzioSenderInstance = logzioSenderInstances.get(tokenAndTypePair);
        if (logzioSenderInstance == null) {
            if (logsQueue == null) {
                throw new LogzioParameterErrorException("logsQueue", "null");
            }
            LogzioSender logzioSender = new LogzioSender(httpsRequestConfiguration, drainTimeout, debug, reporter, tasksExecutor, logsQueue, exceedMaxSizeAction, withOpentelemetryContext);
            logzioSenderInstances.put(tokenAndTypePair, logzioSender);
            return logzioSender;
        }
        reporter.info("Already found appender configured for type " + httpsRequestConfiguration.getLogzioType() + ", re-using the same one.");
        if (logzioSenderInstance.tasksExecutor.isTerminated()) {
            reporter.info("The old task executor is terminated! replacing it with a new one");
            logzioSenderInstance.tasksExecutor = tasksExecutor;
        }
        return logzioSenderInstance;
    }

    private void addOpenTelemetryContext(JsonObject jsonMessage) {
        SpanContext spanContext;
        Span currentSpan = Span.current();
        if (currentSpan != null && (spanContext = currentSpan.getSpanContext()).isValid()) {
            jsonMessage.addProperty("trace_id", spanContext.getTraceId());
            jsonMessage.addProperty("span_id", spanContext.getSpanId());
            Resource resource = Resource.getDefault();
            Attributes attributes = resource.getAttributes();
            String serviceName = (String)attributes.get(ResourceAttributes.SERVICE_NAME);
            jsonMessage.addProperty("service_name", serviceName);
        }
    }

    public void start() {
        this.tasksExecutor.scheduleWithFixedDelay(this::drainQueueAndSend, 0L, this.drainTimeout, TimeUnit.SECONDS);
    }

    public void stop() {
        ExecutorService executorService = Executors.newSingleThreadExecutor();
        this.debug("Got stop request, Submitting a final drain queue task to drain before shutdown. Will timeout in 20 seconds.");
        try {
            executorService.submit(this::drainQueue).get(20L, TimeUnit.SECONDS);
        }
        catch (InterruptedException | ExecutionException | TimeoutException e) {
            this.debug("Waited 20 seconds, but could not finish draining. quitting.", e);
        }
        finally {
            executorService.shutdownNow();
        }
    }

    public void drainQueueAndSend() {
        try {
            if (this.drainRunning.get()) {
                this.debug("Drain is running so we won't run another one in parallel");
                return;
            }
            this.drainRunning.set(true);
            this.drainQueue();
        }
        catch (Exception e) {
            this.reporter.error("Uncaught error from Logz.io sender", e);
        }
        finally {
            this.drainRunning.set(false);
        }
    }

    public void clearQueue() throws IOException {
        this.logsQueue.clear();
    }

    public void send(JsonObject jsonMessage) {
        if (this.withOpentelemetryContext) {
            this.addOpenTelemetryContext(jsonMessage);
        }
        int jsonByteLength = jsonMessage.toString().getBytes(StandardCharsets.UTF_8).length;
        String jsonMessageField = jsonMessage.get("message").getAsString();
        if (jsonByteLength > 500000 || jsonMessageField.length() >= 32700) {
            int truncatedMessageSize = Math.min(32700 - TRUNCATED_MESSAGE_SUFFIX.length(), jsonMessageField.getBytes(StandardCharsets.UTF_8).length - (jsonByteLength - 500000) - TRUNCATED_MESSAGE_SUFFIX.length());
            if (truncatedMessageSize <= 0 || this.exceedMaxSizeAction.equals(DROP_EXCEEDING_LOG)) {
                this.debug(truncatedMessageSize <= 0 ? "Message field is empty after truncating, dropping log" : "Dropping oversized log");
                return;
            }
            String truncatedMessage = jsonMessageField.substring(0, truncatedMessageSize) + TRUNCATED_MESSAGE_SUFFIX;
            jsonMessage.addProperty("message", truncatedMessage);
            this.debug("Truncated oversized log");
        }
        this.logsQueue.enqueue(jsonMessage.toString().getBytes(StandardCharsets.UTF_8));
    }

    public void send(byte[] jsonStringAsUTF8ByteArray) {
        Gson gson = new Gson();
        boolean dropLog = false;
        try {
            if (jsonStringAsUTF8ByteArray.length > 500000 && this.exceedMaxSizeAction.equals(CUT_EXCEEDING_LOG)) {
                String jsonString = new String(jsonStringAsUTF8ByteArray, StandardCharsets.UTF_8);
                JsonObject json = gson.fromJson(jsonString, JsonElement.class).getAsJsonObject();
                String messageString = json.get("message").getAsString();
                int truncatedMessageSize = Math.min(32700 - TRUNCATED_MESSAGE_SUFFIX.length(), messageString.getBytes(StandardCharsets.UTF_8).length - (jsonString.getBytes(StandardCharsets.UTF_8).length - 500000) - TRUNCATED_MESSAGE_SUFFIX.length());
                if (truncatedMessageSize <= 0) {
                    dropLog = true;
                } else {
                    String truncatedMessage = messageString.substring(0, truncatedMessageSize) + TRUNCATED_MESSAGE_SUFFIX;
                    json.addProperty("message", truncatedMessage);
                    jsonStringAsUTF8ByteArray = json.toString().getBytes(StandardCharsets.UTF_8);
                    this.debug("Truncated oversized log");
                }
            }
        }
        catch (JsonSyntaxException | IndexOutOfBoundsException e) {
            dropLog = true;
        }
        if (dropLog || this.exceedMaxSizeAction.equals(DROP_EXCEEDING_LOG)) {
            this.debug(dropLog ? "Message field is empty after truncating, dropping log" : "Dropping oversized log");
            return;
        }
        this.logsQueue.enqueue(jsonStringAsUTF8ByteArray);
    }

    private List<FormattedLogMessage> dequeueUpToMaxBatchSize() {
        ArrayList<FormattedLogMessage> logsList = new ArrayList<FormattedLogMessage>();
        int totalSize = 0;
        while (!this.logsQueue.isEmpty()) {
            byte[] message = this.logsQueue.dequeue();
            if (message == null || message.length <= 0) continue;
            logsList.add(new FormattedLogMessage(message));
            if ((totalSize += message.length) < 0x300000) continue;
            break;
        }
        return logsList;
    }

    private void drainQueue() {
        this.debug("Attempting to drain queue");
        if (!this.logsQueue.isEmpty()) {
            while (!this.logsQueue.isEmpty()) {
                List<FormattedLogMessage> logsList = this.dequeueUpToMaxBatchSize();
                try {
                    this.httpsSyncSender.sendToLogzio(logsList);
                }
                catch (LogzioServerErrorException e) {
                    this.debug("Could not send log to logz.io: ", e);
                    this.debug("Will retry in the next interval");
                    logsList.forEach(logMessage -> this.logsQueue.enqueue(logMessage.getMessage()));
                    break;
                }
                if (!Thread.interrupted()) continue;
                this.debug("Stopping drainQueue to thread being interrupted");
                break;
            }
        }
    }

    private void debug(String message) {
        if (this.debug) {
            this.reporter.info("DEBUG: " + message);
        }
    }

    private void debug(String message, Throwable e) {
        if (this.debug) {
            this.reporter.info("DEBUG: " + message, e);
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private boolean debug = false;
        private int drainTimeoutSec = 5;
        private SenderStatusReporter reporter;
        private ScheduledExecutorService tasksExecutor;
        private InMemoryQueue.Builder inMemoryQueueBuilder;
        private DiskQueue.Builder diskQueueBuilder;
        private HttpsRequestConfiguration httpsRequestConfiguration;
        private String exceedMaxSizeAction = "cut";
        private boolean withOpentelemetryContext = true;

        public Builder setWithOpentelemetryContext(boolean withOpentelemetryContext) {
            this.withOpentelemetryContext = withOpentelemetryContext;
            return this;
        }

        public Builder setExceedMaxSizeAction(String exceedMaxSizeAction) {
            this.exceedMaxSizeAction = exceedMaxSizeAction;
            return this;
        }

        public Builder setDrainTimeoutSec(int drainTimeoutSec) {
            this.drainTimeoutSec = drainTimeoutSec;
            return this;
        }

        public Builder setDebug(boolean debug) {
            this.debug = debug;
            return this;
        }

        public Builder setTasksExecutor(ScheduledExecutorService tasksExecutor) {
            this.tasksExecutor = tasksExecutor;
            return this;
        }

        public Builder setReporter(SenderStatusReporter reporter) {
            this.reporter = reporter;
            return this;
        }

        public Builder setHttpsRequestConfiguration(HttpsRequestConfiguration httpsRequestConfiguration) {
            this.httpsRequestConfiguration = httpsRequestConfiguration;
            return this;
        }

        public InMemoryQueue.Builder withInMemoryQueue() {
            if (this.inMemoryQueueBuilder == null) {
                this.inMemoryQueueBuilder = InMemoryQueue.builder(this);
            }
            return this.inMemoryQueueBuilder;
        }

        public DiskQueue.Builder withDiskQueue() {
            if (this.diskQueueBuilder == null) {
                this.diskQueueBuilder = DiskQueue.builder(this, this.tasksExecutor);
            }
            return this.diskQueueBuilder;
        }

        void setDiskQueueBuilder(DiskQueue.Builder diskQueueBuilder) {
            this.diskQueueBuilder = diskQueueBuilder;
        }

        void setInMemoryQueueBuilder(InMemoryQueue.Builder inMemoryQueueBuilder) {
            this.inMemoryQueueBuilder = inMemoryQueueBuilder;
        }

        public LogzioSender build() throws LogzioParameterErrorException, IOException {
            return LogzioSender.getLogzioSender(this.httpsRequestConfiguration, this.drainTimeoutSec, this.debug, this.reporter, this.tasksExecutor, this.getLogsQueue(), this.exceedMaxSizeAction, this.withOpentelemetryContext);
        }

        private LogsQueue getLogsQueue() throws LogzioParameterErrorException, IOException {
            if (this.diskQueueBuilder != null) {
                this.diskQueueBuilder.setDiskSpaceTasks(this.tasksExecutor);
                this.diskQueueBuilder.setReporter(this.reporter);
                return this.diskQueueBuilder.build();
            }
            this.inMemoryQueueBuilder.setReporter(this.reporter);
            return this.inMemoryQueueBuilder.build();
        }
    }
}

