package io.lsn.spring.mf.domain.oauth2;

import io.lsn.spring.mf.configuration.MFProperties;
import io.lsn.spring.mf.configuration.domain.WlBrokerProperties;
import io.lsn.spring.mf.domain.oauth2.entity.OAuth2Response;
import io.lsn.spring.mf.domain.oauth2.handler.RestClientErrorHandler;
import io.lsn.spring.mf.domain.oauth2.service.TokenService;
import io.lsn.spring.rest.helper.LoggingInterceptor;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.client.BufferingClientHttpRequestFactory;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.stereotype.Service;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import java.time.LocalDateTime;

@Service
@Order(Ordered.HIGHEST_PRECEDENCE)
public class OAuth2Service {

    private WlBrokerProperties properties;
    private TokenService tokenService;
    private RestTemplateBuilder restTemplateBuilder;

    public OAuth2Service(MFProperties properties,
                         TokenService tokenService,
                         RestTemplateBuilder restTemplateBuilder) {
        this.properties = properties.getWl().getBroker();
        this.tokenService = tokenService;
        this.restTemplateBuilder = restTemplateBuilder;
    }

    public String getAuthorizationHeader() {
        OAuth2Response tokenData = tokenService.getFromBucket();
        if (tokenData == null || tokenData.getAccessToken() == null || LocalDateTime.now().isAfter(tokenData.getExpiresDate())) {
            return creteNewSession();
        }
        return tokenData.getAccessToken();
    }

    private String creteNewSession() {
        String url = String.format("%s/oauth/token", this.properties.getUrl());
        String uri = UriComponentsBuilder.fromHttpUrl(url)
                .queryParam("grant_type", "password")
                .queryParam("username", properties.getUsername())
                .queryParam("password", properties.getPassword())
                .toUriString();

        OAuth2Response newTokenData = buildRestTemplate()
                .exchange(uri, HttpMethod.POST, buildHttpRequest(properties.getClientId(), properties.getClientSecret()), OAuth2Response.class)
                .getBody();
        return storeTokenData(newTokenData);
    }

    private String storeTokenData(OAuth2Response newTokenData) {
        tokenService.putIntoBucket(newTokenData);
        return newTokenData.getAccessToken();
    }

    private RestTemplate buildRestTemplate() {
        return restTemplateBuilder
                .additionalMessageConverters(new MappingJackson2HttpMessageConverter())
                .additionalInterceptors(new LoggingInterceptor())
                .additionalCustomizers((restTemplate) -> {
                    SimpleClientHttpRequestFactory factory = new SimpleClientHttpRequestFactory();
                    factory.setOutputStreaming(false);
                    restTemplate.setRequestFactory(new BufferingClientHttpRequestFactory(factory));
                })
                .errorHandler(new RestClientErrorHandler())
                .build();
    }

    private HttpEntity<MultiValueMap<String, String>> buildHttpRequest(String username, String password) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);
        headers.setBasicAuth(username, password);
        return new HttpEntity<>(headers);
    }

}

