package io.lsn.spring.mf.domain.wl;

import io.lsn.logger.factory.LoggerFactory;
import io.lsn.logger.factory.logger.Logger;
import io.lsn.spring.mf.configuration.MFProperties;
import io.lsn.spring.mf.configuration.domain.WlRegistryProperties;
import io.lsn.spring.mf.domain.oauth2.OAuth2Service;
import io.lsn.spring.mf.domain.oauth2.handler.RestClientErrorHandler;
import io.lsn.spring.mf.domain.vat.entity.rabbit.VatTransportResponse;
import io.lsn.spring.mf.domain.wl.domain.Response;
import io.lsn.spring.mf.domain.wl.domain.VatStatusFlag;
import io.lsn.spring.mf.domain.wl.domain.VerificationException;
import io.lsn.spring.mf.domain.wl.domain.external.EntityResponse;
import io.lsn.spring.mf.domain.wl.domain.external.ResultException;
import io.lsn.spring.rest.RestClient;
import io.lsn.spring.rest.helper.LoggingInterceptor;
import io.lsn.spring.utilities.configuration.condition.ConditionalOnConfiguration;
import io.lsn.spring.utilities.json.mapper.JsonMapper;
import org.apache.commons.lang.StringUtils;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.http.*;
import org.springframework.http.client.BufferingClientHttpRequestFactory;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.stereotype.Service;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import java.io.IOException;
import java.time.LocalDate;
import java.util.Collections;

@Service
@ConditionalOnConfiguration(name = "io.lsn.spring.wl.api", type = ConditionalOnConfiguration.Type.SERVICE)
public class WlRegistryService {

    private static final Logger logger = LoggerFactory.getLogger(WlRegistryService.class);
    private static final String HEADER_NAME_AUTHORIZATION = "Authorization";

    private RestClient restClient;
    private WlRegistryProperties wlRegistryProperties;
    private OAuth2Service oAuth2Service;
    private RestTemplateBuilder restTemplateBuilder;

    public WlRegistryService(RestClient restClient,
                             MFProperties mfProperties,
                             OAuth2Service oAuth2Service,
                             RestTemplateBuilder restTemplateBuilder) {
        this.restClient = restClient;
        this.wlRegistryProperties = mfProperties.getWl();
        this.oAuth2Service = oAuth2Service;
        this.restTemplateBuilder = restTemplateBuilder;
    }

    /**
     * search by tax id
     *
     * @param taxId tax id
     * @param iban  iban for verification
     */
    public Response searchNip(String taxId, String iban) throws VerificationException {
        try {
            ResponseEntity<EntityResponse> result = restClient.get(wlRegistryProperties.getApiUrl() + "/search/nip/" + taxId + "?date=" + LocalDate.now().toString(), EntityResponse.class);
            String ibanNoPrefix = iban.replaceAll("[^\\d]", "");

            if (result.getBody() == null) {
                throw new VerificationException(String.valueOf(result.getStatusCode()));
            }

            if (result.getBody().getException() != null) {
                throw new VerificationException(result.getBody().getException().getCode(), result.getBody().getException().getMessage());
            }

            Response response = new Response()
                    .setResponseId(result.getBody().getResult().getRequestId())
                    .setVatStatusFlag(VatStatusFlag.NOT_FOUND);

            if (result.getBody().getResult().getSubject() != null) {
                String address = result.getBody().getResult().getSubject().getWorkingAddress();
                if (StringUtils.isBlank(address)) {
                    address = result.getBody().getResult().getSubject().getResidenceAddress();
                }

                response.setName(result.getBody().getResult().getSubject().getName())
                        .setHasIban(result.getBody().getResult().getSubject().getAccountNumbers().contains(ibanNoPrefix))
                        .setAddress(address)
                        .setVatStatusFlag(VatStatusFlag.byValue(result.getBody().getResult().getSubject().getStatusVat()));
            }

            return response;

        } catch (RuntimeException exception) {
            if (exception instanceof HttpClientErrorException && ((HttpClientErrorException) exception).getRawStatusCode() == 400) {
                HttpClientErrorException e = ((HttpClientErrorException) exception);
                try {
                    ResultException r = JsonMapper.getInstance().getObjectMapper().readValue(e.getResponseBodyAsString(), ResultException.class);
                    throw new VerificationException(r.getCode(), r.getMessage());
                } catch (IOException ex) {
                    // do nothing
                }
            }
            throw new VerificationException("500", exception.getMessage());
        }
    }

    public Response searchNipViaBroker(String companyCode, String taxId, String iban) throws VerificationException {
        try {
            String url = String.format("%s/api/public/vat", wlRegistryProperties.getBroker().getUrl());
            String uri = UriComponentsBuilder.fromHttpUrl(url)
                    .queryParam("companyCode", companyCode)
                    .queryParam("taxId", taxId)
                    .queryParam("iban", iban)
                    .toUriString();

            ResponseEntity<VatTransportResponse> rest = buildRestTemplate()
                    .exchange(uri, HttpMethod.GET, buildHttpEntity(), VatTransportResponse.class);
            VatTransportResponse vatResponse = rest.getBody();
            if (vatResponse.getErrorCode() != null) {
                throw new VerificationException(vatResponse.getError().get("code"), vatResponse.getError().get("message"));
            }
            return vatResponse.getResponse();
        } catch (Exception exception) {
            throw new VerificationException("500", exception.getMessage());
        }
    }


    protected HttpEntity<MultiValueMap<String, String>> buildHttpEntity() {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        String authorization = String.format("Bearer %s", this.oAuth2Service.getAuthorizationHeader());
        headers.put(HEADER_NAME_AUTHORIZATION, Collections.singletonList(authorization));
        return new HttpEntity<>(headers);
    }

    protected RestTemplate buildRestTemplate() {
        RestTemplateBuilder builder = restTemplateBuilder
                .additionalMessageConverters(new MappingJackson2HttpMessageConverter())
                .interceptors(new LoggingInterceptor())
                .additionalCustomizers((restTemplate) -> {
                    SimpleClientHttpRequestFactory factory = new SimpleClientHttpRequestFactory();
                    factory.setOutputStreaming(false);
                    restTemplate.setRequestFactory(new BufferingClientHttpRequestFactory(factory));
                })
                .errorHandler(new RestClientErrorHandler());
        return builder.build();

    }
}
