/**
 * Copyright 2017 Pivotal Software, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.micrometer.core.instrument.stats.hist;

import java.util.ArrayList;

/**
 * Built off of {@link com.netflix.spectator.api.histogram.PercentileBuckets}, but based on observed double values.
 *
 * @author Jon Schneider
 */
public class PercentileBuckets {
    public static double bucketFunction(double d) {
        return BUCKET_VALUES[indexOf(d)];
    }

    /**
     * Returns the value the index of the bucket that should be used for {@code v}.
     */
    private static int indexOf(double v) {
        if (v <= 0) {
            return 0;
        } else if (v <= 4) {
            return (int) v;
        } else {
            int lz = Long.numberOfLeadingZeros((long) v);
            int shift = 64 - lz - 1;
            long prevPowerOf2 = ((long) v >> shift) << shift;
            long prevPowerOf4 = prevPowerOf2;
            if (shift % 2 != 0) {
                shift--;
                prevPowerOf4 = prevPowerOf2 >> 1;
            }

            long base = prevPowerOf4;
            long delta = base / 3;
            int offset = (int) ((v - base) / delta);
            int pos = offset + POWER_OF_4_INDEX[shift / 2];
            return (pos >= BUCKET_VALUES.length - 1) ? BUCKET_VALUES.length - 1 : pos + 1;
        }
    }

    // Number of positions of base-2 digits to shift when iterating over the long space.
    private static final int DIGITS = 2;

    // Bucket values to use, see static block for initialization.
    private static final double[] BUCKET_VALUES;

    // Keeps track of the positions for even powers of 4 within BUCKET_VALUES. This is used to
    // quickly compute the offset for a long without traversing the array.
    private static final int[] POWER_OF_4_INDEX;

    // The set of buckets is generated by using powers of 4 and incrementing by one-third of the
    // previous power of 4 in between as long as the value is less than the next power of 4 minus
    // the delta.
    //
    // <pre>
    // Base: 1, 2, 3
    //
    // 4 (4^1), delta = 1
    //     5, 6, 7, ..., 14,
    //
    // 16 (4^2), delta = 5
    //    21, 26, 31, ..., 56,
    //
    // 64 (4^3), delta = 21
    // ...
    // </pre>
    static {
        ArrayList<Integer> powerOf4Index = new ArrayList<>();
        powerOf4Index.add(0);

        ArrayList<Long> buckets = new ArrayList<>();
        buckets.add(1L);
        buckets.add(2L);
        buckets.add(3L);

        int exp = DIGITS;
        while (exp < 64) {
            long current = 1L << exp;
            long delta = current / 3;
            long next = (current << DIGITS) - delta;

            powerOf4Index.add(buckets.size());
            while (current < next) {
                buckets.add(current);
                current += delta;
            }
            exp += DIGITS;
        }
        buckets.add(Long.MAX_VALUE);

        BUCKET_VALUES = new double[buckets.size()];
        for (int i = 0; i < buckets.size(); ++i) {
            BUCKET_VALUES[i] = buckets.get(i);
        }

        POWER_OF_4_INDEX = new int[powerOf4Index.size()];
        for (int i = 0; i < powerOf4Index.size(); ++i) {
            POWER_OF_4_INDEX[i] = powerOf4Index.get(i);
        }
    }
}
